/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Array;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This class holds response in form of key-value pairs for keys in the {@link Parameter.Key}
 * and are received from a node/device.
 * This class can be used to pass as argument to functions like {@link Device#sendInfoResponse}.
 */
public class InfoResponse {

  private static final Class<?>[] PARAM_CLASS_LUT;
  private static final Class<?>[] SEARCH_CLASS_LUT;

  static {
    PARAM_CLASS_LUT = new Class<?>[Parameter.Key.values().length];

    PARAM_CLASS_LUT[Parameter.Key.USER_DEFINED.ordinal()] = UserDefined.class;
    PARAM_CLASS_LUT[Parameter.Key.USERNAME.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.PASSWORD.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.PIN.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.OTP.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.INSTALL_CODE.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.CSA.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.DSK.ordinal()] = Parameter.StringValue.class;
    PARAM_CLASS_LUT[Parameter.Key.RESOURCE_NAME.ordinal()] = Parameter.StringValue.class;

    PARAM_CLASS_LUT[Parameter.Key.AVAIL_LISTEN_PORT.ordinal()] = ListenPort.class;
    PARAM_CLASS_LUT[Parameter.Key.CONTENT_SEARCH.ordinal()] = ContentSearch.class;
    PARAM_CLASS_LUT[Parameter.Key.CONTENT_PLAYBACK.ordinal()] = ContentPlayback.class;
    PARAM_CLASS_LUT[Parameter.Key.STREAM_BUFFER_SIZES.ordinal()] = Parameter.BufferSizeValue.class;
    PARAM_CLASS_LUT[Parameter.Key.PAIRING_TYPE.ordinal()] = Parameter.PairingTypeValue.class;
    PARAM_CLASS_LUT[Parameter.Key.TIMEZONE.ordinal()] = Parameter.TimeZoneValue.class;

    PARAM_CLASS_LUT[Parameter.Key.REMOTE_PAIRING_CODES.ordinal()] = RemotePairingCode.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_PAIR_CAPABILITIES.ordinal()] =
        RemotePairingCapabilities.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_SEARCH.ordinal()] = Search.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_BUTTON_INFO.ordinal()] = RemoteButtonInfo.class;

    SEARCH_CLASS_LUT = new Class<?>[Parameter.SearchType.values().length];

    SEARCH_CLASS_LUT[Parameter.SearchType.BRAND.ordinal()] = BrandSearchResults.class;
  }

  protected transient Command.State state;

  protected transient String networkId;

  // the node from which infoRequest came by or the device from which response is received
  private transient long requestNodeId;
  private transient long infoRequestId;
  private transient long cmdSeqNum;

  private final List<Parameter<Value>> parameters;

  /**
   * A marker interface for values corresponding to key as in {@link Parameter.Key}
   * for InfoResponse.
   */
  public interface Value extends Parameter.Value {
  }

  /**
   * A marker interface to classify search results.
   */
  public interface SearchResults {
  }

  /**
   * This class encapsulates brand information such as id and name.
   */
  public static class BrandSearchResults implements SearchResults {
    @SerializedName(Constants.BRAND_ID)
    public final int brandId;
    @SerializedName(Constants.BRAND_NAME)
    public final String brandName;

    public BrandSearchResults(int brandId, String brandName) {
      this.brandId = brandId;
      this.brandName = brandName;
    }
  }

  /**
   * This class acts as a value of response corresponding to {@link Parameter.Key#REMOTE_SEARCH}.
   *
   * @see InfoRequest.Search
   */
  public static class Search implements Value {
    @SerializedName(Constants.TOTAL_SEARCH_COUNT)
    public final long totalSearchCount;
    @SerializedName(Constants.SEARCH_TYPE)
    public final Parameter.SearchType searchType;
    @SerializedName(Constants.SEARCH_RESULTS)
    public final SearchResults[] searchResults;

    /**
     * A constructor for this value.
     *
     * @param totalSearchCount    count of all results provided in response
     * @param searchType          an enum constant denoting type of search to be made, ex: brand
     *                             {@link Parameter.SearchType}
     * @param searchResults       array of search results comprising brand information
     */
    public Search(long totalSearchCount, Parameter.SearchType searchType,
                  SearchResults[] searchResults) {
      this.totalSearchCount = totalSearchCount;
      this.searchType = searchType;
      this.searchResults = searchResults;
    }
  }

  /**
   * This class acts as value of response corresponding to
   * {@link Parameter.Key#REMOTE_PAIRING_CODES}.
   *
   * @see InfoRequest.RemotePairingCode
   * @see CapabilityRemoteControl.SetSearchModeResponse
   */
  public static class RemotePairingCode implements Value {
    @SerializedName(Constants.PAIRING_CODES)
    public int[] pairingCodes;
  }

  /**
   * This class acts as value of response corresponding to {@link Parameter.Key#REMOTE_BUTTON_INFO}.
   *
   * @see InfoRequest.RemoteButtonInfo
   */
  public static final class RemoteButtonInfo implements Value {
    @SerializedName(Constants.BUTTON_INFO)
    public List<CapabilityRemoteControl.RemoteControlButtonInfo> buttonsInfo;

    public RemoteButtonInfo(List<CapabilityRemoteControl.RemoteControlButtonInfo> buttonsInfo) {
      this.buttonsInfo = buttonsInfo;
    }
  }

  /**
   * This class acts as a value of response corresponding to
   * {@link Parameter.Key#REMOTE_PAIR_CAPABILITIES}.
   *
   * @see InfoRequest.RemotePairingCapabilities
   */
  public static class RemotePairingCapabilities implements Value {
    @SerializedName(Constants.CAPABILITY_ID)
    private final Capability.CapabilityId capId;
    @SerializedName(Constants.COMMAND_ID_ARR)
    private final int[] commandSupportedArr;
    @SerializedName(Constants.ATTR_ID_ARR)
    private final int[] attrIdArr;

    /**
     * A constructor for this value.
     *
     * @param capId                  capability id corresponding to resource in request
     * @param commandSupportedArr    supported command ids array
     * @param attrIdArr              attribute id array
     */
    public RemotePairingCapabilities(Capability.CapabilityId capId, int[] commandSupportedArr,
                                     int[] attrIdArr) {
      this.capId = capId;
      this.commandSupportedArr = commandSupportedArr;
      this.attrIdArr = attrIdArr;
    }

    public Capability.CapabilityId getCapabilityId() {
      return capId;
    }

    /**
     * A function to fetch list of supported command ids as received in response.
     *
     * @return list of supported command ids
     */
    public List<CommandIdInterface> getCommandSupportedArr() {
      if (null == commandSupportedArr) {
        return Collections.emptyList();
      }

      ArrayList<CommandIdInterface> supportedCommands = new ArrayList<>();

      for (int command : commandSupportedArr) {
        supportedCommands.add(capId.getCommandId(command));
      }

      return supportedCommands;
    }

    /**
     * A function to fetch list of attribute ids received in response.
     *
     * @return list of attribute ids
     */
    public List<Capability.AttributeId> getAttributes() {
      if (null == attrIdArr) {
        return Collections.emptyList();
      }

      ArrayList<Capability.AttributeId> supportedAttr = new ArrayList<>();

      for (int attrId : attrIdArr) {
        supportedAttr.add(capId.getAttributeId(attrId));
      }

      return supportedAttr;
    }
  }

  /**
   * This class forms a value of response corresponding to {@link Parameter.Key#CONTENT_PLAYBACK}.
   *
   * @see InfoRequest.ContentPlayback
   */
  public static class ContentPlayback implements Value {
    @SerializedName(Constants.PLAYBACK_URL)
    public String playbackUrl;
    @SerializedName(Constants.REQ_PLAYBACK_ERROR)
    public CapabilityStorageControl.PlaybackError playbackError;
    @SerializedName(Constants.STREAM_PROTOCOL_TYPE)
    public CapabilityStorageControl.StreamProtocol streamProtocol;
  }

  /**
   * This class forms a value of response corresponding to {@link Parameter.Key#CONTENT_SEARCH}.
   *
   * @see InfoRequest.ContentSearch
   */
  public static class ContentSearch implements Value {
    @SerializedName(Constants.TOTAL_SEARCH_COUNT)
    public long totalSearchCount;
    @SerializedName(Constants.CONTENT_METADATA_ARR)
    public StorageContentMetadata[] contentMetadata;
  }

  /**
   * This class forms a value of response corresponding to {@link Parameter.Key#USER_DEFINED}.
   *
   * @see InfoRequest.UserDefined
   */
  public static class UserDefined implements Value {
    // TODO: parser
    public String response;
  }

  /**
   * This class forms a value of response corresponding to {@link Parameter.Key#AVAIL_LISTEN_PORT}.
   * Doesn't use a type adapter.
   *
   * @see InfoRequest.ListenPort
   */
  public static class ListenPort implements Value {
    @SerializedName(Constants.PORTS)
    public int[] ports;

    public ListenPort() {
    }
  }

  /**
   * Constructor for current class.
   *
   * @param infoRequest      InfoRequest corresponding to this response
   * @param parameters       collection of value responses
   */
  public InfoResponse(InfoRequest infoRequest, List<Parameter<Value>> parameters) {
    // infoRequest's requestNodeId should be the responseNodeId
    this(infoRequest.requestId, infoRequest.cmdSeqNum, infoRequest.networkId,
        infoRequest.deviceNodeId, parameters);
  }

  public InfoResponse(long infoRequestId, long cmdSeqNum, List<Parameter<Value>> parameters) {
    this(infoRequestId, cmdSeqNum, null, parameters);
  }

  public InfoResponse(long infoRequestId, long cmdSeqNum, Device device,
                      List<Parameter<Value>> parameters) {
    this(infoRequestId, cmdSeqNum, (null == device) ? null : device.getParent().getId(),
        (null == device) ? -1 : device.getId(), parameters);
  }

  /**
   * An overloaded constructor for this class.
   *
   * @param infoRequestId     id for the info request for which this response is being created
   * @param cmdSeqNum         command sequence number
   * @param networkId         The network to which the device belongs
   * @param requestNodeId     The node from which infoRequest came by or
   *                          the device from which response is received
   * @param parameters        The params which are responded
   */
  public InfoResponse(long infoRequestId, long cmdSeqNum, String networkId, long requestNodeId,
                      List<Parameter<Value>> parameters) {
    this.infoRequestId = infoRequestId;
    this.parameters = new ArrayList<>(parameters);
    this.networkId = networkId;
    this.requestNodeId = requestNodeId;
    this.cmdSeqNum = cmdSeqNum;
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(InfoResponse.class, new InfoResponseParser());
    Command.GSON_BUILDER.registerTypeAdapter(Search.class, new SearchResultsParser());
  }

  public Command.State getState() {
    return this.state;
  }

  public void setDevice(Device device) {
    this.requestNodeId = (null == device) ? -1 : device.getId();
  }

  public Device getDevice() {
    // TODO: check if requestNodeId to be used or responseNodeId to be used.
    return Utils.getDevice(CocoClient.getInstance().getNetworkMap(), networkId, requestNodeId);
  }

  protected void setCmdSeqNum(long cmdSeqNum) {
    this.cmdSeqNum = cmdSeqNum;
  }

  protected void setInfoRequestId(long infoRequestId) {
    this.infoRequestId = infoRequestId;
  }

  public List<Parameter<Value>> getParameters() {
    return parameters;
  }

  /**
   * A function to fetch keys from parameters .
   *
   * @return The array of keys in the parameters
   */
  public Parameter.Key[] getKeys() {
    Parameter.Key[] keys = new Parameter.Key[parameters.size()];

    for (int i = 0; i < parameters.size(); i++) {
      keys[i] = parameters.get(i).key;
    }

    return keys;
  }

  protected void setRequestNodeId(long requestNodeId) {
    this.requestNodeId = requestNodeId;
  }

  public long getRequestNodeId() {
    return requestNodeId;
  }

  public long getInfoRequestId() {
    return infoRequestId;
  }

  public long getCmdSeqNum() {
    return cmdSeqNum;
  }

  protected JsonElement toJson() {
    return Command.GSON_BUILDER.create().toJsonTree(this);
  }

  private static final class InfoResponseParser implements JsonSerializer<InfoResponse>,
      JsonDeserializer<InfoResponse> {

    @Override
    public InfoResponse deserialize(JsonElement jsonElement, Type typeOfT,
                                    JsonDeserializationContext context) throws
        JsonParseException {
      Parameter.Key[] paramKeys;
      Value[] paramValues;
      JsonArray paramsJson;
      Gson gson = Command.GSON_BUILDER.create();
      JsonObject json;

      if (!jsonElement.isJsonObject()) {
        return new InfoResponse(-1, -1, new ArrayList<>(0));
      }

      json = jsonElement.getAsJsonObject();

      paramsJson = json.getAsJsonArray(Constants.INFO_RESPONSES);

      if (null != paramsJson) {
        int index = 0;
        paramKeys = new Parameter.Key[paramsJson.size()];
        paramValues = new Value[paramsJson.size()];

        for (JsonElement paramJsonElement : paramsJson) {
          JsonObject paramJson = paramJsonElement.getAsJsonObject();
          Parameter.Key key =
              gson.fromJson(paramJson.get(Constants.PARAMETER_KEY), Parameter.Key.class);

          Type keyType = PARAM_CLASS_LUT[key.ordinal()];

          paramKeys[index] = key;

          if (null != keyType) {
            Value value = gson.fromJson(paramJson.get(Constants.PARAMETER_VALUE), keyType);
            paramValues[index] = value;
          }

          index++;
        }
      } else {
        paramKeys = new Parameter.Key[0];
        paramValues = new Value[0];
      }

      List<Parameter<Value>> parameters = new ArrayList<>(paramKeys.length);

      for (int i = 0; i < paramKeys.length; i++) {
        parameters.add(new Parameter<>(paramKeys[i], paramValues[i]));
      }

      return new InfoResponse(-1, -1, parameters);
    }

    @Override
    public JsonElement serialize(InfoResponse infoResponse, Type typeOfSrc,
                                 JsonSerializationContext context) {
      JsonObject json = new JsonObject();
      Gson gson = Command.GSON_BUILDER.create();

      json.add(Constants.INFO_RESPONSES, gson.toJsonTree(infoResponse.parameters));
      return json;
    }
  }

  private static final class SearchResultsParser
      implements JsonSerializer<Search>, JsonDeserializer<Search> {

    @Override
    public Search deserialize(JsonElement jsonElement, Type type, JsonDeserializationContext jdc)
        throws JsonParseException {
      JsonObject jsonObject = jsonElement.getAsJsonObject();

      long searchCount = jsonObject.get(Constants.TOTAL_SEARCH_COUNT).getAsLong();
      Parameter.SearchType searchType =
          jdc.deserialize(jsonObject.get(Constants.SEARCH_TYPE), Parameter.SearchType.class);
      SearchResults[] searchResults = jdc.deserialize(jsonObject.get(Constants.SEARCH_RESULTS),
          Array.newInstance(SEARCH_CLASS_LUT[searchType.ordinal()], 0).getClass());

      return new Search(searchCount, searchType, searchResults);
    }

    @Override
    public JsonElement serialize(Search search, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return jsonSerializationContext.serialize(search);
    }
  }
}
