/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Objects;

/**
 * Class used to identify the target system and load the corresponding library.
 */
class JniLoader {

  public static final String TAG = "JniLoader";

  private static final String LIBRARY_FORMATTER = "/native/%s/%s/libcocojni.%s";

  private static String getOsName() {

    String jvmName = System.getProperty("java.vm.name", "").toLowerCase();
    String osName = System.getProperty("os.name", "").toLowerCase();

    if (jvmName.startsWith("dalvik") && osName.startsWith("linux")) {
      osName = "android";
    } else if (jvmName.startsWith("robovm") && osName.startsWith("darwin")) {
      osName = "ios";
    } else if (osName.startsWith("mac") || osName.startsWith("darwin")) {
      osName = "mac";
    } else if (osName.startsWith("windows")) {
      osName = "windows";
    } else if (osName.startsWith("linux")) {
      osName = "linux";
    } else {
      throw new IllegalStateException("unknown os: " + osName);
    }

    Log.d(TAG, "osName: " + osName);

    return osName;
  }

  private static String getOsArch() {

    String osArch = System.getProperty("os.arch", "").toLowerCase();

    if (osArch.equals("i386") || osArch.equals("i486") || osArch.equals("i586")
        || osArch.equals("i686")) {
      osArch = "x86";
    } else if (osArch.equals("amd64") || osArch.equals("x86-64") || osArch.equals("x64")) {
      osArch = "x86_64";
    } else if (osArch.startsWith("aarch64") || osArch.startsWith("armv8")
        || osArch.startsWith("arm64")) {
      osArch = "armv8";
    } else if (osArch.startsWith("arm")) {
      osArch = "armv7";
    } else {
      throw new IllegalStateException("unknown arch: " + osArch);
    }

    return osArch;
  }

  private static String getExtension() {
    String extension;
    String osName = System.getProperty("os.name", "").toLowerCase();

    if (osName.startsWith("linux")) {
      extension = "so";
    } else if (osName.startsWith("mac") || osName.startsWith("darwin")) {
      extension = "dylib";
    } else if (osName.startsWith("windows")) {
      extension = "dll";
    } else {
      throw new IllegalStateException("unknown os: " + osName);
    }

    return extension;
  }

  /**
   * TODO: load stream if and only if the cached file and current vary.
   *
   * @param cwd The path to where the resource will be extracted
   */
  static void load(String cwd) {

    File extractedLib;
    String libResource;

    long startTime = System.currentTimeMillis();

    try {
      System.loadLibrary("cocojni");
      return;
    } catch (UnsatisfiedLinkError err) {
      Log.w(TAG, "load from env failed", err);
      Log.d(TAG, "load failed in: " + (System.currentTimeMillis() - startTime));
    }

    libResource =
        String.format(Locale.ENGLISH, LIBRARY_FORMATTER, getOsName(), getOsArch(), getExtension());
    extractedLib = new File(cwd, libResource);

    Log.d(TAG, "lib path: " + extractedLib.getAbsolutePath());

    try {
      extractedLib.mkdirs();
      extractedLib.delete();
      extractedLib.createNewFile();
    } catch (IOException e) {
      Log.w(TAG, "create/delete file failed", e);
    }

    Log.d(TAG, "loading from resource stream: " + libResource);

    try (InputStream is = JniLoader.class.getResourceAsStream(libResource);
         FileOutputStream fos = new FileOutputStream(extractedLib)) {

      Objects.requireNonNull(is);

      int readLength;
      byte[] buffer = new byte[1024];

      while (-1 != (readLength = is.read(buffer))) {
        fos.write(buffer, 0, readLength);
      }
    } catch (IOException | NullPointerException e) {
      Log.w(TAG, "load from resource stream failed", e);
    }

    Log.d(TAG, "load: copy resource complete in: " + (System.currentTimeMillis() - startTime));

    System.load(extractedLib.getAbsolutePath());

    Log.d(TAG, "load: complete in: " + (System.currentTimeMillis() - startTime));
  }
}
