/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.Arrays;

/**
 * Logger for cococlientsdk-java.
 */
public class Log {
  public static final int VERBOSE = 2;
  public static final int DEBUG = 3;
  public static final int INFO = 4;
  public static final int WARN = 5;
  public static final int ERROR = 6;
  public static final int ASSERT = 7;

  private static final boolean DEBUG_LOGS_ENABLED = true;

  protected static final String[] LOG_LEVEL_TO_STR = new String[ASSERT + 1];

  static {
    Arrays.fill(LOG_LEVEL_TO_STR, "?");

    LOG_LEVEL_TO_STR[VERBOSE] = "v";
    LOG_LEVEL_TO_STR[DEBUG] = "d";
    LOG_LEVEL_TO_STR[INFO] = "i";
    LOG_LEVEL_TO_STR[WARN] = "w";
    LOG_LEVEL_TO_STR[ERROR] = "e";
    LOG_LEVEL_TO_STR[ASSERT] = "a";
  }

  /**
   * Logger interface instance aids in printing out different log types to stack trace.
   */
  public interface Logger {
    default boolean isLogEnabled(int priority) {
      return (DEBUG_LOGS_ENABLED) || (DEBUG != priority);
    }

    /**
     * A function to print out the logs.
     *
     * @param priority     integer representing any logs i.e, VERBOSE,DEBUG, INFO, WARN, ERROR,
     *                     ASSERT, etc.
     * @param tag          Stringified class name
     * @param msg          message to be logged to stack trace
     * @param tr           Throwable to be logged
     * @return             integer denoting success or failure of log
     */
    default int println(int priority, String tag, String msg, Throwable tr) {
      if (!isLogEnabled(priority)) {
        return -1;
      }

      long now = System.currentTimeMillis();

      System.out.printf("%d %s/%s: %s", now, LOG_LEVEL_TO_STR[priority], tag, msg);

      if (null != tr) {
        System.out.printf("\n%s", getStackAsString(tr));
      }

      System.out.println();

      return 0;
    }

    /**
     * A function to append string name of throwable.
     *
     * @param tr Throwable to be logged
     * @return Stringified throwable appended to stacktrace.
     */
    default String getStackAsString(Throwable tr) {

      if (null == tr) {
        return "";
      }

      StringBuilder builder = new StringBuilder();

      StackTraceElement[] stack = tr.getStackTrace();

      builder.append(tr.getMessage());
      builder.append('\n');

      for (StackTraceElement ste : stack) {
        builder.append(ste);
        builder.append('\n');
      }

      return builder.toString();
    }
  }

  static Logger logger = new Logger() {
  };

  //CHECKSTYLE:OFF
  public static int v(String tag, String msg) {
    return logger.println(VERBOSE, tag, msg, null);
  }

  public static int v(String tag, String msg, Throwable tr) {
    return logger.println(VERBOSE, tag, msg, tr);
  }

  public static int d(String tag, String msg) {
    return logger.println(DEBUG, tag, msg, null);
  }

  public static int d(String tag, String msg, Throwable tr) {
    return logger.println(DEBUG, tag, msg, tr);
  }

  public static int i(String tag, String msg) {
    return logger.println(INFO, tag, msg, null);
  }

  public static int i(String tag, String msg, Throwable tr) {
    return logger.println(INFO, tag, msg, tr);
  }

  public static int w(String tag, String msg) {
    return logger.println(WARN, tag, msg, null);
  }

  public static int w(String tag, String msg, Throwable tr) {
    return logger.println(WARN, tag, msg, tr);
  }

  public static int e(String tag, String msg) {
    return logger.println(ERROR, tag, msg, null);
  }

  public static int e(String tag, String msg, Throwable tr) {
    return logger.println(ERROR, tag, msg, tr);
  }
  //CHECKSTYLE:ON
}

