/*
 * @author    Rohan Sahay, rohansahay@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.annotations.SerializedName;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Network class is blueprint for a network which is consumed by the developer.
 * It possesses all devices, zones, scenes, rules mapped with their ids.
 */
public class Network implements Iterable<Device> {
  /**
   * removes the network from the hashmap after connect status callback with possible
   * disconnected states.
   * When state belongs to any of these the network will be removed from memory
   * {@link #createClearOnDisconnectionFlag} {@link #containsClearFlag}.
   */
  static int clearNetworkForState = createClearOnDisconnectionFlag(
      State.CONNECTION_ERROR,
      State.NETWORK_RESET,
      State.DISCONNECTED);

  /**
   * An enum denoting possible values for command ids.
   */
  public enum CommandId implements CommandIdInterface {
    SET_ZONE_INFO,
    REM_ZONE,
    ADD_ZONE_RES,
    REM_ZONE_RES,
    MOVE_ZONE_RES,
    SET_SCENE_INFO,
    REM_SCENE,
    ADD_SCENE_RES_ACT,
    UPDATE_SCENE_RES_ACT,
    REM_SCENE_RES_ACT,
    SET_RULE_INFO,
    REM_RULE,
    ADD_RULE_RES_COND,
    UPDATE_RULE_RES_COND,
    REM_RULE_RES_COND,
    ADD_RULE_SCH_COND,
    UPDATE_RULE_SCH_COND,
    REM_RULE_SCH_COND,
    ADD_RULE_RES_ACT,
    UPDATE_RULE_RES_ACT,
    REM_RULE_RES_ACT,
    ADD_RULE_SCENE_ACT,
    REM_RULE_SCENE_ACT,
    SET_SCENE_METADATA,
    SAVE_SCENE_SETTINGS,
    SAVE_RULE_SETTINGS,
    ASSIGN_TO_TEMPLATE;

    static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  /**
   * A function to create flags to clear network from memory in case of disconnection.
   *
   * @param states  array of states
   * @return        an integer flag
   */
  public static int createClearOnDisconnectionFlag(State... states) {
    int flags = 0;

    for (State state : states) {
      flags |= (1 << state.ordinal());
    }

    return flags;
  }

  /**
   * A function to check if the state contains a clear flag for network i.e, if state is either of
   * {@link State#NETWORK_RESET}. {@link State#CONNECTION_ERROR}, {@link State#DISCONNECTED}.
   *
   * @param state  State under interest to check for clear
   * @return       True if state of network contains clear flag
   */
  public static boolean containsClearFlag(State state) {
    int flag = 1 << state.ordinal();

    return flag == (clearNetworkForState & flag);
  }

  /**
   * state enum denoting if a network is connected/connecting etc..,
   */
  public enum State {
    CONNECTED,
    CONNECTING,
    CONNECTION_ERROR,
    NETWORK_RESET,
    DISCONNECTED,
    BLOCKED,
    NOT_PERMITTED;

    static State getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }

    public boolean isConnected() {
      return (CONNECTED == this);
    }

    public boolean isConnecting() {
      return (CONNECTING == this);
    }

    public boolean isConnectedOrConnecting() {
      return isConnecting() || isConnected();
    }

    public boolean isEquivalentToDisconnected() {
      return !isConnectedOrConnecting();
    }

    public boolean isLeftOrDeletedOrBlocked() {
      return isLeftOrDeleted() || isBlocked();
    }

    public boolean isLeftOrDeleted() {
      return NETWORK_RESET == this;
    }

    public boolean isBlocked() {
      return BLOCKED == this;
    }

    public boolean connectionNotPermitted() {
      return NOT_PERMITTED == this;
    }
  }

  /**
   * UserRole enum denoting the role of the user in a given network.
   */
  public enum UserRole {
    OWNER,
    ADMIN,
    GUEST;

    public static UserRole getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * AccessType enum denoting what app is allowed to access.
   */
  public enum AccessType {
    LOCAL,
    REMOTE;

    public static AccessType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * NetworkType enum denoting classifies a network object.
   */
  public enum NetworkType {
    IOT,
    CALL_NET;

    public static NetworkType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  @SerializedName(Constants.NETWORK_ID)
  private final String id;
  @SerializedName(Constants.NETWORK_NAME)
  private String name;
  @SerializedName(Constants.NETWORK_USER_ROLE)
  private UserRole userRole;
  @SerializedName(Constants.NETWORK_ACCESS_TYPE)
  private AccessType accessType;
  @SerializedName(Constants.NETWORK_TYPE)
  private NetworkType networkType;

  private transient State state = State.DISCONNECTED;
  private transient ConnectivityStateManager.State connectivityManagerState =
      ConnectivityStateManager.State.OFFLINE;

  @SerializedName(Constants.NETWORK_DEVICES)
  private Map<Long, Device> deviceMap;
  @SerializedName(Constants.NETWORK_ZONES)
  private Map<Integer, Zone> zoneMap;
  @SerializedName(Constants.NETWORK_SCENES)
  private Map<Integer, Scene> sceneMap;
  @SerializedName(Constants.NETWORK_RULES)
  private Map<Integer, Rule> ruleMap;

  @SerializedName(Constants.RESOURCE_TEMPLATES)
  private List<ResourceTemplate> resourceTemplates;
  @SerializedName(Constants.NETWORK_METADATA)
  private String metadata;

  private transient Iterable<Device> deviceIterable;
  private transient Iterable<Zone> zoneIterable;
  private transient Iterable<Scene> sceneIterable;
  private transient Iterable<Rule> ruleIterable;

  /**
   * The constructor of the class. Constructor is protected to make sure that networks can be only
   * created by the cococlientsdk-java.
   *
   * @param id Id of the current network.
   */
  protected Network(String id) {
    Zone defaultZone;

    // Adding default zone
    getZoneMap().put(Zone.DEFAULT_ZONE_ID,
        defaultZone = Factory.createZone(Zone.DEFAULT_ZONE_ID, this));
    defaultZone.internalSetName(Zone.defaultZoneName);

    this.id = id;
    this.userRole = UserRole.GUEST;
    this.accessType = AccessType.REMOTE;
  }

  /**
   * A function to connect to the network with inviteUrl.
   *
   * @param inviteUrl   The URL which is provided by the Cloud
   * @param networkId   The networkID of the network to connect
   * @param networkName The name of the network to connect
   * @param userRole    The role of the user
   * @param accessType  The accessType needed
   * @param nodeId      The type of node that is being added
   */
  public static void connectWithInvite(String networkId, String networkName,
                                       NetworkType networkType, UserRole userRole,
                                       AccessType accessType,
                                       String inviteUrl, long nodeId) {

    Network network = CocoClient.getInstance().getNetwork(networkId);

    if (null == network) {
      network = Factory.createNetwork(networkId);
    }

    network.internalSetNetworkType(networkType);
    network.internalSetName(networkName);
    network.internalSetAccessType(accessType);
    network.internalSetUserRole(userRole);

    connectWithInvite(network, inviteUrl, nodeId);
  }

  /**
   * A function to connect to the network with inviteUrl.
   *
   * @param network   The network to which to connect
   * @param inviteUrl The URL which is provided by the Cloud
   * @param nodeId    The type of node that is being added
   */
  public static void connectWithInvite(Network network, String inviteUrl, long nodeId) {
    CocoClient.getInstance().getNativeHandler().connectWithInvite(network, inviteUrl, nodeId);
    CocoClient.getInstance().internalAddNetwork(network);
  }

  /**
   * To connect to an individual network. This function must be used to connect to single network.
   *
   * @throws RuntimeException Throws an exception on failure during communication.
   */
  public void connect() {
    connect(0);
  }

  /**
   * To connect to an individual network.This function must be used to connect to single network.
   *
   * @param port           This function takes a port over which the communication will
   *                       take place NOTE: The port should be a free port or this call
   *                       will fail If 0 is passed there will be a search and free port
   *                       will be chosen automatically.
   * @throws RuntimeException Throws an exception on failure during communication.
   */
  public void connect(int port) {
    Map<String, Network> networkMap = CocoClient.getInstance().getNetworkMap();

    CocoClient.getInstance().getNativeHandler().connect(this, port);

    // push to map after connect. If, connect throws exception. The networks don't remain in map
    networkMap.put(getId(), this);
  }

  /**
   * This function is used to disconnect from a network and delete from local database accordingly.
   *
   * @throws RuntimeException Throw an exception on failures during communication.
   */
  public void disconnect() {
    CocoClient.getInstance().getNativeHandler().disconnect(this);
  }

  /**
   * This function is used to leave the network.
   */
  public void leave(LeaveStatusListener listener) {
    CocoClient.getInstance().getNativeHandler().leaveNetwork(this, listener);
  }

  /**
   * A function to manage the network like, add zone, remove zone and other stuff.
   *
   * @param command The command which has to be sent
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void sendNetworkManagementCommand(Command<CommandId> command,
                                           NetworkManagementStatusListener listener) {
    CocoClient.getInstance().getNativeHandler()
        .sendNetworkManagementCommand(this, command, listener);
  }

  /**
   *  A function to create a zone in the network.
   *
   * @param zoneName The name of the zone.
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void createZone(String zoneName, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new SetZone(zoneName), listener);
  }

  /**
   * A function to delete a zone from the network.
   *
   * @param zone     The zone which has to deleted
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteZone(Zone zone, NetworkManagementStatusListener listener) {
    deleteZone(new Zone[] {zone}, listener);
  }

  /**
   * A function to delete zones from the network.
   *
   * @param zones   The zones which are to deleted
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteZone(Zone[] zones, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new RemoveZone(zones), listener);
  }

  /**
   * A function to move resources of a zone to another zone.
   *
   * @param fromZone The zone from which the resources has to be moved
   * @param toZone   The zone to which the resource has to be moved
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void moveResources(Zone fromZone, Zone toZone, Resource[] resources,
                            NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new MoveResourcesBetweenZones(fromZone, toZone, resources),
        listener);
  }

  /**
   * A function to create scene in a network.
   *
   * @param sceneName The name of the scene
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void createScene(String sceneName, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new SetScene(sceneName), listener);
  }

  /**
   * A function to delete a scene from the network.
   *
   * @param scene Scene which has to be deleted
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteScene(Scene scene, NetworkManagementStatusListener listener) {
    deleteScene(new Scene[] {scene}, listener);
  }

  /**
   * A function to delete scenes in a network.
   *
   * @param scenes The scenes to be deleted
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteScene(Scene[] scenes, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new RemoveScene(scenes), listener);
  }

  /**
   * A function to create rule in the network.
   *
   * @param ruleName The name of the rule
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void createRule(String ruleName, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new SetRule(ruleName), listener);
  }

  /**
   * A function to delete rule in a network.
   *
   * @param rule     Rule which has to be deleted
   * @param listener The listener which will be triggered with
   *       {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteRule(Rule rule, NetworkManagementStatusListener listener) {
    deleteRule(new Rule[] {rule}, listener);
  }

  /**
   * A function to delete rules in a network.
   *
   * @param rules    The rules to be deleted
   * @param listener The listener which will be triggered with
   *        {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void deleteRule(Rule[] rules, NetworkManagementStatusListener listener) {
    sendNetworkManagementCommand(new RemoveRule(rules), listener);
  }

  /**
   * This function is used to get the Id that uniquely identifies the current Network.
   *
   * @return String: Id of the Network
   */
  public final String getId() {
    return this.id;
  }

  /**
   * This function is used to get the Name of the network.
   *
   * @return String: Name of the Network
   */
  public String getName() {
    return this.name;
  }

  /**
   * This function is used to get the current state of the Network.
   *
   * @return State: The enum indicating the state of the network
   */
  public State getState() {
    return this.state;
  }

  /**
   * This function is used to get the connectivity state of the network. NOTE:
   * This will only work after {@link ConnectivityStateManager#getInstance}.
   *
   * @return ConnectivityStateManager.State: the enum indicating the state of the network
   */
  public ConnectivityStateManager.State getConnectivityManagerState() {
    return this.connectivityManagerState;
  }

  /**
   * This function is used to get role of the current user in this network.
   *
   * @return UserRole: The enum indicating the role of the user
   */
  public UserRole getUserRole() {
    return this.userRole;
  }

  /**
   * This function is used to return the access type of the user.
   *
   * @return AccessType: The enum indicating the AccessType of the user (sometimes
   *        a root user may want to drop his privileges and also sometimes a
   *        guest may have higher privileges given by the admin)
   */
  public AccessType getAccessType() {
    return this.accessType;
  }

  /**
   * This function is used to get the type of the network.
   *
   * @return NetworkType: The enum indicating the type of the network
   */
  public NetworkType getNetworkType() {
    return this.networkType;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map the DeviceIds
   * to the Device Objects. This function can also be used to data-bind the sdk to the application.
   *
   * @return The key being the deviceNodeId and the value
   *         being the Device Object. NOTE: This HashMap is for read-only any
   *         misuse may cause the program to misbehave and even crash.
   */
  public Map<Long, Device> getDeviceMap() {
    if (null == this.deviceMap) {
      this.deviceMap = new ConcurrentHashMap<>();
    }

    return this.deviceMap;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the zoneIds to the Zone Objects.
   *
   * @return The key being the zoneId and the value
   *         being the Zone Object. NOTE: This HashMap is for read-only any
   *         misuse may cause the program to misbehave and even crash.
   */
  public Map<Integer, Zone> getZoneMap() {
    if (null == this.zoneMap) {
      this.zoneMap = new ConcurrentHashMap<>();
    }

    return this.zoneMap;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the sceneIds to the Scene Objects.
   *
   * @return The key being the sceneId and the value
   *         being the Scene Object. NOTE: This hashmap is for read-only any
   *         misuse may cause the program to misbehave and even crash.
   */
  public Map<Integer, Scene> getSceneMap() {
    if (null == this.sceneMap) {
      this.sceneMap = new ConcurrentHashMap<>();
    }

    return this.sceneMap;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the ruleIds to the Rule Objects.
   *
   * @return The key being the ruleId and the value
   *        being the Rule Object. NOTE: This HashMap is for read-only any
   *        misuse may cause the program to misbehave and even crash.
   */
  public Map<Integer, Rule> getRuleMap() {
    if (null == this.ruleMap) {
      this.ruleMap = new ConcurrentHashMap<>();
    }

    return this.ruleMap;
  }

  /**
   * This function is used to get the list of resource templates.
   *
   * @return ResourceTemplate list
   */
  public List<ResourceTemplate> getResourceTemplates() {
    return resourceTemplates;
  }

  /**
   * This function is used to get the zone of this network.
   *
   * @return Zone with the given zoneId
   */
  public final <T extends Device> T getDevice(long deviceNodeId) {
    return Utils.castUp(getDeviceMap().get(deviceNodeId));
  }

  /**
   * This function is used to get the zone of this network.
   *
   * @return Zone with the given zoneId
   */
  public final <T extends Zone> T getZone(int zoneId) {
    return Utils.castUp(getZoneMap().get(zoneId));
  }

  /**
   * This function is used to get the scene of this network.
   *
   * @return Scene with the given zoneId
   */
  public final <T extends Scene> T getScene(int sceneId) {
    return Utils.castUp(getSceneMap().get(sceneId));
  }

  /**
   * This function is used to get the rules of this network.
   *
   * @return Rule with the given ruleId
   */
  public final <T extends Rule> T getRule(int ruleId) {
    return Utils.castUp(getRuleMap().get(ruleId));
  }

  /**
   * A function to serialize the whole memory.
   *
   * @see CocoClient#restoreNetworkState(String)
   * @return The serializedJson which represents the complete memory state.
   */
  public String saveState() {
    return DeepSerializer.serialize(this);
  }

  /**
   * A function to restore network information from memory.
   *
   * @see CocoClient#restoreNetworkState(String)
   * @param networkState JSON string which was serialized to memory during
   *                    {@link Network#saveState()}
   */
  public void restoreNetworkState(String networkState) {
    Network duplicateWithState = CocoClient.getInstance().restoreNetworkState(networkState);

    if (!this.getId().equals(duplicateWithState.getId())) {
      throw new IllegalArgumentException(
          "id mismatch: this:" + getId() + ", that: " + duplicateWithState.getId());
    }

    // copying deviceMap, zoneMap, sceneMap, ruleMap
    // name, accessType, userRole aren't necessary
    // Since, If used properly, this object will contain
    // updated info than the one from save state

    // NOTE: This will just perform a union of both the data from
    // save state and the current data

    for (Device d : duplicateWithState.getDeviceIterable()) {
      if (containsDevice(d.getId())) {
        continue;
      }

      internalAddDevice(d);
    }

    for (Zone z : duplicateWithState.getZoneIterable()) {
      if (containsZone(z.getId())) {
        continue;
      }

      internalAddZone(z);
    }

    for (Scene s : duplicateWithState.getSceneIterable()) {
      if (containsScene(s.getId())) {
        continue;
      }

      internalAddScene(s);
    }

    for (Rule r : duplicateWithState.getRuleIterable()) {
      if (containsRule(r.getId())) {
        continue;
      }

      internalAddRule(r);
    }
  }

  protected void internalRemoveNetwork() {
  }

  protected void internalSetResourceTemplates(List<ResourceTemplate> resourceTemplates) {
    if (null == resourceTemplates) {
      resourceTemplates = new ArrayList<>();
    }

    this.resourceTemplates = resourceTemplates;
  }

  protected void internalAddDevice(Device device) {
    getDeviceMap().put(device.getId(), device);
  }

  protected void internalRemoveDevice(long deviceNodeId) {
    getDeviceMap().remove(deviceNodeId);
  }

  public boolean containsDevice(long deviceNodeId) {
    return getDeviceMap().containsKey(deviceNodeId);
  }

  protected void internalAddZone(Zone zone) {
    getZoneMap().put(zone.getId(), zone);
  }

  protected void internalRemoveZone(int zoneId) {
    getZoneMap().remove(zoneId);
  }

  public boolean containsZone(int zoneId) {
    return getZoneMap().containsKey(zoneId);
  }

  protected void internalAddScene(Scene scene) {
    getSceneMap().put(scene.getId(), scene);
  }

  protected void internalRemoveScene(int sceneId) {
    getSceneMap().remove(sceneId);
  }

  public boolean containsScene(int sceneId) {
    return getSceneMap().containsKey(sceneId);
  }

  protected void internalAddRule(Rule rule) {
    getRuleMap().put(rule.getId(), rule);
  }

  protected void internalRemoveRule(int ruleId) {
    getRuleMap().remove(ruleId);
  }

  public boolean containsRule(int ruleId) {
    return getRuleMap().containsKey(ruleId);
  }

  protected void internalSetName(String name) {
    this.name = name;
  }

  protected void internalSetUserRole(UserRole userRole) {
    this.userRole = userRole;
  }

  protected void internalSetAccessType(AccessType accessType) {
    this.accessType = accessType;
  }

  protected void internalSetState(State state) {
    this.state = state;
  }

  protected void internalSetConnectivityState(ConnectivityStateManager.State state) {
    this.connectivityManagerState = state;
  }

  protected void internalSetNetworkType(NetworkType type) {
    this.networkType = type;
  }

  protected void internalSetMetadata(String metadata) {
    this.metadata = metadata;
  }

  public String getMetadata() {
    return metadata;
  }

  @Override
  public final int hashCode() {
    return getId().hashCode();
  }

  @Override
  public Iterator<Device> iterator() {
    return getDeviceIterable().iterator();
  }

  /**
   * This function fetches device iterable to traverse over list of devices.
   *
   * @return A device iterable
   */
  public Iterable<Device> getDeviceIterable() {
    if (null != deviceIterable) {
      return deviceIterable;
    }

    return (deviceIterable = () -> getDeviceMap().values().iterator());
  }

  /**
   * This function fetches device iterable to traverse over list of zones.
   *
   * @return A zone iterable
   */
  public Iterable<Zone> getZoneIterable() {
    if (null != zoneIterable) {
      return zoneIterable;
    }

    return (zoneIterable = () -> getZoneMap().values().iterator());
  }

  /**
   * This function fetches device iterable to traverse over list of scenes.
   *
   * @return A scene iterable
   */
  public Iterable<Scene> getSceneIterable() {
    if (null != sceneIterable) {
      return sceneIterable;
    }

    return (sceneIterable = () -> getSceneMap().values().iterator());
  }

  /**
   * This function fetches device iterable to traverse over list of rules.
   *
   * @return A rule iterable
   */
  public Iterable<Rule> getRuleIterable() {
    if (null != ruleIterable) {
      return ruleIterable;
    }

    return (ruleIterable = () -> getRuleMap().values().iterator());
  }

  /**
   * An api to send data to the specified nodes or all nodes of network.
   *
   * @param data    The data that has to be sent
   * @param nodeIds The nodes which will get the data.
   *                null to send to all nodes.
   */
  public void sendData(String data, long[] nodeIds) {
    CocoClient.getInstance().getNativeHandler().sendData(this, data, nodeIds);
  }

  /**
   * An api to send data to the specified nodes or all nodes of network.
   *
   * @param contentTime The time at which the data is sent
   * @param data        The data that has to be sent
   * @param nodeIds     The nodes which will get the data.
   *                    null to send to all nodes.
   */
  public void sendContentInfo(long contentTime, String data, long[] nodeIds) {
    CocoClient.getInstance().getNativeHandler().sendContentInfo(this, contentTime, data, nodeIds);
  }

  @Override
  public String toString() {
    return "Network{"
        + "id='" + id + '\''
        + ", name='" + name + '\''
        + ", userRole=" + userRole
        + ", accessType=" + accessType
        + ", state=" + state
        + ", connectivityManagerState=" + connectivityManagerState
        + ", networkType=" + networkType
        + '}';
  }

  /**
   * This class defines setting necessary args before connecting to a network.
   */
  public static final class ConnectArgs {
    private String networkId;
    private String inviteUrl;
    private long nodeId;

    // NOTE: These might get removed in future
    private String networkName;
    private UserRole userRole;
    private AccessType accessType;
    private NetworkType networkType;

    public ConnectArgs() {
    }

    public ConnectArgs setNetworkId(String networkId) {
      this.networkId = networkId;
      return this;
    }

    public String getNetworkId() {
      return networkId;
    }

    public ConnectArgs setNetworkType(NetworkType networkType) {
      this.networkType = networkType;
      return this;
    }

    public NetworkType getNetworkType() {
      return networkType;
    }

    public ConnectArgs setInviteUrl(String inviteUrl) {
      this.inviteUrl = inviteUrl;
      return this;
    }

    public String getInviteUrl() {
      return inviteUrl;
    }

    public ConnectArgs setNodeId(long nodeId) {
      this.nodeId = nodeId;
      return this;
    }

    public long getNodeId() {
      return nodeId;
    }

    public ConnectArgs setAccessType(AccessType accessType) {
      this.accessType = accessType;
      return this;
    }

    public ConnectArgs setUserRole(UserRole userRole) {
      this.userRole = userRole;
      return this;
    }

    public ConnectArgs setNetworkName(String networkName) {
      this.networkName = networkName;
      return this;
    }

    public void connect() {
      Network.connectWithInvite(getNetworkId(), this.networkName, this.networkType, this.userRole,
          this.accessType, getInviteUrl(), getNodeId());
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while sending creating/changing zone name commands.
   *
   * @see Zone#setName(String, NetworkManagementStatusListener)
   */
  public static class SetZone extends Command<CommandId> {
    @SerializedName(Constants.ZONE_ID)
    public int zoneId = Zone.DEFAULT_ZONE_ID;
    @SerializedName(Constants.ZONE_NAME)
    public String zoneName;

    public SetZone(String zoneName) {
      this(null, zoneName);
    }

    /**
     * An overloaded constructor for this command.
     *
     * @param zone      Zone in interest to be updated with new params
     * @param zoneName  Zone name to replace existing name
     */
    public SetZone(Zone zone, String zoneName) {
      super(CommandId.SET_ZONE_INFO);

      if (null != zone) {
        this.zoneId = zone.getId();
      }
      this.zoneName = zoneName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while sending set zone info commands.
   *
   * @see Network#deleteZone(Zone, NetworkManagementStatusListener)
   */
  public static class RemoveZone extends Command<CommandId> {
    @SerializedName(Constants.ZONE_IDS)
    public int[] zoneIds;

    /**
     * An overloaded constructor for this command.
     *
     * @param zones array of zones to be removed
     */
    public RemoveZone(Zone[] zones) {
      super(CommandId.REM_ZONE);

      this.zoneIds = new int[zones.length];

      for (int i = 0; i < zones.length; i++) {
        zoneIds[i] = zones[i].getId();
      }
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while sending add resources to zone commands.
   */
  @Deprecated
  public static class AddResourcesToZone extends Command<CommandId> {
    @SerializedName(Constants.ZONE_ID)
    public int zoneId;
    @SerializedName(Constants.RESOURCES)
    public Resource[] resources;

    /**
     * A constructor for this command.
     *
     * @param zone         Zone in interest to which the resources are to be added
     * @param resources    array of resources to be added to the zone param
     */
    public AddResourcesToZone(Zone zone, Resource[] resources) {
      super(CommandId.ADD_ZONE_RES);

      this.zoneId = zone.getId();
      this.resources = resources;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while moving resources between zones is the command.
   *
   * @see Network#moveResources(Zone, Zone, Resource[], NetworkManagementStatusListener)
   */
  public static class MoveResourcesBetweenZones extends Command<CommandId> {
    @SerializedName(Constants.OLD_ZONE_ID)
    public int fromZoneId;
    @SerializedName(Constants.NEW_ZONE_ID)
    public int toZoneId;
    @SerializedName(Constants.RESOURCES)
    public Resource[] resources;

    /**
     * A constructor for this command.
     *
     * @param fromZone   Zone param from which resources are to be moved
     * @param toZone     Zone param to which resources are to be moved
     * @param resources  array of resources which are to be moved
     */
    public MoveResourcesBetweenZones(Zone fromZone, Zone toZone, Resource[] resources) {
      super(CommandId.MOVE_ZONE_RES);

      this.fromZoneId = fromZone.getId();
      this.toZoneId = toZone.getId();
      this.resources = resources;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while removing resources from zone is the command.
   */
  @Deprecated
  public static class RemoveResourcesFromZone extends Command<CommandId> {
    @SerializedName(Constants.ZONE_ID)
    public int zoneId;
    @SerializedName(Constants.RESOURCES)
    public Resource[] resources;

    /**
     * A constructor for this command.
     *
     * @param zone        zone in interest from which resources are to be removed
     * @param resources   array of resources to be removed
     */
    public RemoveResourcesFromZone(Zone zone, Resource[] resources) {
      super(CommandId.REM_ZONE_RES);

      this.zoneId = zone.getId();
      this.resources = resources;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while creating/changing scene info is the command.
   *
   * @see Network#createScene(String, NetworkManagementStatusListener)
   */
  public static class SetScene extends Command<CommandId> {
    @SerializedName(Constants.SCENE_ID)
    public int sceneId = Scene.DEFAULT_SCENE_ID;
    @SerializedName(Constants.SCENE_NAME)
    public String sceneName;

    public SetScene(String sceneName) {
      this(null, sceneName);
    }

    /**
     * A constructor for current command.
     *
     * @param scene        The scene in interest whose name change is requested
     * @param sceneName    scene name to replace the existing name
     */
    public SetScene(Scene scene, String sceneName) {
      super(CommandId.SET_SCENE_INFO);

      if (null != scene) {
        this.sceneId = scene.getId();
      }

      this.sceneName = sceneName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while setting scene metadata is the command.
   *
   * @see Scene#setMetadata(String, NetworkManagementStatusListener)
   */
  public static class SetSceneMetadata extends Command<CommandId> {
    @SerializedName(Constants.SCENE_ID)
    public int sceneId = Scene.DEFAULT_SCENE_ID;
    @SerializedName(Constants.SCENE_METADATA)
    public String sceneMetadata;

    /**
     * A constructor for this command.
     *
     * @param scene            Scene under interest to change metadata
     * @param sceneMetadata    String metadata to be set to the scene param
     */
    public SetSceneMetadata(Scene scene, String sceneMetadata) {
      super(CommandId.SET_SCENE_METADATA);

      if (null != scene) {
        this.sceneId = scene.getId();
      }

      this.sceneMetadata = sceneMetadata;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while deleting scene is the command.
   *
   * @see Network#deleteScene(Scene, NetworkManagementStatusListener)
   */
  public static class RemoveScene extends Command<CommandId> {
    @SerializedName(Constants.SCENE_IDS)
    public int[] sceneIds;

    /**
     * A constructor for this command.
     *
     * @param scenes  array of scenes to be removed
     */
    public RemoveScene(Scene[] scenes) {
      super(CommandId.REM_SCENE);

      sceneIds = new int[scenes.length];

      for (int i = 0; i < scenes.length; i++) {
        sceneIds[i] = scenes[i].getId();
      }
    }
  }

  /**
   * This class acts as super class for the commands around adding/saving scene resource actions.
   */
  public abstract static class SceneResourceActions extends Command<CommandId> {
    @SerializedName(Constants.SCENE_ID)
    public int sceneId;
    @SerializedName(Constants.RESOURCE_ACTIONS)
    public ResourceAction[] resourceActions;

    /**
     * Constructor for scene resource actions.
     *
     * @param scene            scene for which resource actions are defined
     * @param resourceActions  commands resource must execute when scene triggers
     * @param commandId        CommandId as in {@link Network.CommandId}
     */
    public SceneResourceActions(Scene scene, ResourceAction[] resourceActions,
                                CommandId commandId) {
      super(commandId);

      this.sceneId = scene.getId();
      this.resourceActions = resourceActions;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while adding scene resource action is the command.
   *
   * @see Scene#addResourceActions(ResourceAction[], NetworkManagementStatusListener)
   */
  public static class AddSceneResourceAction extends SceneResourceActions {
    /**
     * Constructor for this command.
     *
     * @param scene             scene for which a resource actions are added
     * @param resourceActions   array of resource actions to be added
     */
    public AddSceneResourceAction(Scene scene, ResourceAction[] resourceActions) {
      super(scene, resourceActions, CommandId.ADD_SCENE_RES_ACT);

      for (ResourceAction action : resourceActions) {
        if (ResourceAction.DEFAULT_RESOURCE_ACTION_ID != action.getId()) {
          throw new IllegalArgumentException(
              "expecting id: " + ResourceAction.DEFAULT_RESOURCE_ACTION_ID + ", got: "
                  + action.getId());
        }
      }
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while updating scene resource action is the command.
   *
   * @see Scene#updateResourceActions(int[], ResourceAction[], NetworkManagementStatusListener)
   */
  public static class UpdateSceneResourceActions extends SceneResourceActions {
    /**
     * A function to create a copy of given resource action array. NOTE: this is not a deep copy.
     *
     * @param actionIds    integer ids for resource actions
     * @param actions      array of resource actions
     * @return copy of resource actions array param
     */
    public static ResourceAction[] copy(int[] actionIds, ResourceAction[] actions) {
      if (actionIds.length != actions.length) {
        throw new IllegalArgumentException(
            "array lengths mismatch: " + actionIds.length + ", " + actions.length);
      }

      ResourceAction[] copy = new ResourceAction[actions.length];

      for (int i = 0; i < actions.length; i++) {
        ResourceAction action =
            new ResourceAction(actionIds[i], actions[i].getCapability(), actions[i].getCommand());
        copy[i] = action;
      }

      return copy;
    }

    /**
     * Constructor for this command.
     *
     * @param scene              scene which requires to be updated resource actions
     * @param resourceActionIds  integer array of ids for resource actions
     * @param resourceActions    resource actions to be updated with existing ones
     */
    public UpdateSceneResourceActions(Scene scene, int[] resourceActionIds,
                                      ResourceAction[] resourceActions) {
      super(scene, copy(resourceActionIds, resourceActions), CommandId.UPDATE_SCENE_RES_ACT);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while saving scene resource action is the command.
   *
   * @see Scene#saveResourceActions(ResourceAction[], NetworkManagementStatusListener)
   */
  public static class SaveSceneResourceActions extends SceneResourceActions {
    public SaveSceneResourceActions(Scene scene, ResourceAction[] resourceActions) {
      super(scene, resourceActions, CommandId.SAVE_SCENE_SETTINGS);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while removing scene resource action is the command.
   *
   * @see Scene#removeResourceActions(ResourceAction[], NetworkManagementStatusListener)
   */
  public static class RemoveSceneResourceActions extends Command<CommandId> {
    @SerializedName(Constants.SCENE_ID)
    public int sceneId;
    @SerializedName(Constants.RESOURCE_ACTIONS)
    public int[] resourceActions;

    /**
     * Constructor for this command.
     *
     * @param scene            Scene in interest to have its resource actions removed
     * @param resourceActions  array of resource actions to be deleted from scene param
     */
    public RemoveSceneResourceActions(Scene scene, ResourceAction[] resourceActions) {
      super(CommandId.REM_SCENE_RES_ACT);

      this.resourceActions = new int[resourceActions.length];

      for (int i = 0; i < resourceActions.length; i++) {
        this.resourceActions[i] = resourceActions[i].getId();
      }

      this.sceneId = scene.getId();
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while creating/changing rule info is the command.
   *
   * @see Network#createRule(String, NetworkManagementStatusListener)
   */
  public static class SetRule extends Command<CommandId> {
    @SerializedName(Constants.RULE_ID)
    public int ruleId = Rule.DEFAULT_RULE_ID;
    @SerializedName(Constants.RULE_NAME)
    public String ruleName;

    public SetRule(String ruleName) {
      this(null, ruleName);
    }

    /**
     * A constructor for current command.
     *
     * @param rule        The rule in interest whose name change is requested
     * @param ruleName    rule name to replace the existing name
     */
    public SetRule(Rule rule, String ruleName) {
      super(CommandId.SET_RULE_INFO);

      if (null != rule) {
        this.ruleId = rule.getId();
      }

      this.ruleName = ruleName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendNetworkManagementCommand} while deleting rule is the command.
   *
   * @see Network#deleteRule(Rule[], NetworkManagementStatusListener)
   */
  public static class RemoveRule extends Command<CommandId> {
    @SerializedName(Constants.RULE_IDS)
    public int[] ruleIds;

    /**
     * A constructor for this command.
     *
     * @param rules  array of rules to be removed
     */
    public RemoveRule(Rule[] rules) {
      super(CommandId.REM_RULE);

      ruleIds = new int[rules.length];

      for (int i = 0; i < rules.length; i++) {
        ruleIds[i] = rules[i].getId();
      }
    }
  }

  /**
   * This class can be sent as a command to
   * {@link Network#sendNetworkManagementCommand(Command, NetworkManagementStatusListener)}
   * to save the rule.
   */
  public static class SaveRule extends Command<CommandId> {

    @SerializedName(Constants.RULE_ID)
    public int id;
    @SerializedName(Constants.RULE_NAME)
    public String name;

    @SerializedName(Constants.RESOURCE_CONDITIONS)
    public List<ResourceCondition> resourceConditions;
    @SerializedName(Constants.SCHEDULE_CONDITIONS)
    public List<ScheduleCondition> scheduleConditions;
    @SerializedName(Constants.RESOURCE_ACTIONS)
    public List<ResourceAction> resourceActions;
    @SerializedName(Constants.SCENE_ACTIONS)
    public List<Scene> sceneActions;

    /**
     * A constructor for this command.
     */
    public SaveRule() {
      super(CommandId.SAVE_RULE_SETTINGS);

      this.id = Rule.DEFAULT_RULE_ID;
    }

    /**
     * A function to verify the condition data types.
     *
     * @return True if data type is correct
     */
    public boolean areConditionDataTypesCorrect() {
      for (ResourceCondition cond : resourceConditions) {
        Attribute attribute = cond.getAttribute();
        Attribute.DataType dataType = (null == attribute) ? null : attribute.getDataType();

        if (cond.getAttributeDataType() != dataType) {
          return false;
        }
      }

      return true;
    }

    /**
     * A function to rectify the condition data types using
     * {@link ResourceCondition.EditableResourceCondition}.
     */
    public void correctConditionsDataTypes() {
      List<ResourceCondition> conditions = new ArrayList<>();

      for (ResourceCondition cond : resourceConditions) {
        if (cond instanceof ResourceCondition.EditableResourceCondition) {
          ResourceCondition.EditableResourceCondition editableCond =
              ((ResourceCondition.EditableResourceCondition) cond);
          editableCond.setDataType(cond.getAttribute().getDataType());
          continue;
        }

        ResourceCondition.EditableResourceCondition editableCond =
            new ResourceCondition.EditableResourceCondition(cond.getAttribute(),
                cond.getOperatorType(),
                cond.getConditionDuration(),
                cond.getThresholdAttributeValue());
        conditions.add(editableCond);
      }

      resourceConditions.clear();
      resourceConditions.addAll(conditions);
    }
  }

  /**
   * Template is a predefined blueprint of zone, device and resource combo
   * which marks a smoother experience for owner and technician in managing resources.
   * This class can be sent as a command to assign a resource to this template using
   * {@link Network#sendNetworkManagementCommand(Command, NetworkManagementStatusListener)}.
   */
  public static class AssignToTemplate extends Command<CommandId> {
    @SerializedName(Constants.ZONE_ID)
    public final int zoneId;

    @SerializedName(Constants.RESOURCE_TEMPLATE_DEVICE_ID)
    private final long deviceId;

    @SerializedName(Constants.RESOURCE_TEMPLATE_ID)
    private final String resourceTemplateId;

    @SerializedName(Constants.RESOURCE_EUI)
    private final String resourceEui;

    /**
     * Constructor for current class.
     *
     * @param zoneId               id of zone
     * @param deviceId             device node id of the template
     * @param resourceTemplateId   id of the resource template
     * @param resourceEui          id of the resource to be assigned to template
     */
    public AssignToTemplate(int zoneId, long deviceId, String resourceTemplateId,
                            String resourceEui) {
      super(CommandId.ASSIGN_TO_TEMPLATE);

      this.zoneId = zoneId;
      this.deviceId = deviceId;
      this.resourceTemplateId = resourceTemplateId;
      this.resourceEui = resourceEui;
    }
  }

  /**
   * This class encapsulates basic information about resource such as zoneId, icon, name etc.
   * which serves as a blueprint for a resource.
   */
  public static class ResourceTemplate {
    @SerializedName(Constants.RESOURCE_TEMPLATE_ID)
    private final String resourceTemplateId;

    @SerializedName(Constants.ZONE_ID)
    private final int zoneId;

    @SerializedName(Constants.RESOURCE_TEMPLATE_NAME)
    private final String name;

    @SerializedName(Constants.RESOURCE_TEMPLATE_ICON)
    private final String icon;

    /**
     * A constructor for current class.
     *
     * @param resourceTemplateId  id of the resource template
     * @param zoneId              id of zone corresponding to the resource
     * @param name                name of the resource
     * @param icon                icon of the resource
     */
    public ResourceTemplate(String resourceTemplateId, int zoneId, String name, String icon) {
      this.resourceTemplateId = resourceTemplateId;
      this.zoneId = zoneId;
      this.name = name;
      this.icon = icon;
    }

    public String getResourceTemplateId() {
      return resourceTemplateId;
    }

    public int getZoneId() {
      return zoneId;
    }

    public String getName() {
      return name;
    }

    public String getIcon() {
      return icon;
    }

    @Override
    public String toString() {
      return "ResourceTemplate{"
          + "resourceTemplateId='" + resourceTemplateId + '\''
          + ", zoneId=" + zoneId
          + ", name='" + name + '\''
          + ", icon='" + icon + '\''
          + '}';
    }
  }

  static final class SdkContext {
    private final String networkId;

    SdkContext(String networkId) {
      this.networkId = networkId;
    }

    public String getNetworkId() {
      return networkId;
    }
  }

  /**
   * Listener used to listen updates of
   * {@link Network#leave(LeaveStatusListener)}.
   */
  public interface LeaveStatusListener extends Listener {
    void onResponse(Command.State state, Throwable tr);
  }

  /**
   * Listener used to listen updates of
   * {@link Network#sendNetworkManagementCommand(Command, NetworkManagementStatusListener)}.
   */
  public interface NetworkManagementStatusListener extends Listener {
    void onStatusChanged(CommandResponse<Network.CommandId> commandResponse, Throwable tr);
  }
}
