/*
 * @author   Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.time.ZoneId;

/**
 * Parameter class aids in defining key parameters for {@link InfoRequest} and {@link InfoResponse}.
 *
 * @param <T> Type can be any immediate child class of {@link Parameter.Value}
 */
public class Parameter<T extends Parameter.Value> {
  private static final String[] PARAM_KEY_LUT = new String[Parameter.Key.values().length];

  static {
    PARAM_KEY_LUT[Parameter.Key.USERNAME.ordinal()]      = Constants.PARAM_KEY_USERNAME;
    PARAM_KEY_LUT[Parameter.Key.PASSWORD.ordinal()]      = Constants.PARAM_KEY_PASSWORD;
    PARAM_KEY_LUT[Parameter.Key.PIN.ordinal()]           = Constants.PARAM_KEY_PIN;
    PARAM_KEY_LUT[Parameter.Key.OTP.ordinal()]           = Constants.PARAM_KEY_OTP;
    PARAM_KEY_LUT[Parameter.Key.INSTALL_CODE.ordinal()]  = Constants.PARAM_KEY_INSTALL_CODE;
    PARAM_KEY_LUT[Parameter.Key.CSA.ordinal()]           = Constants.PARAM_KEY_CSA;
    PARAM_KEY_LUT[Parameter.Key.DSK.ordinal()]           = Constants.PARAM_KEY_DSK;
    PARAM_KEY_LUT[Parameter.Key.RESOURCE_NAME.ordinal()] = Constants.PARAM_KEY_RESOURCE_NAME;
  }

  private static final class KeyParser implements JsonSerializer<Key>, JsonDeserializer<Key> {

    @Override
    public JsonElement serialize(Key key, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(key.getInt());
    }

    @Override
    public Key deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws
        JsonParseException {
      int key = json.getAsInt();
      return Key.getEnum(key);
    }
  }

  private static final class PairingTypeParser
      implements JsonSerializer<PairingType>, JsonDeserializer<PairingType> {

    @Override
    public PairingType deserialize(JsonElement jsonElement, Type type,
                                   JsonDeserializationContext jsonDeserializationContext)
        throws JsonParseException {
      int pairingType = jsonElement.getAsInt();
      return PairingType.getEnum(pairingType);
    }

    @Override
    public JsonElement serialize(PairingType pairingType, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(pairingType.getInt());
    }
  }

  private static final class SearchTypeParser
      implements JsonSerializer<SearchType>, JsonDeserializer<SearchType> {

    @Override
    public SearchType deserialize(JsonElement jsonElement, Type type,
                                  JsonDeserializationContext jsonDeserializationContext)
        throws JsonParseException {
      return SearchType.getEnum(jsonElement.getAsInt());
    }

    @Override
    public JsonElement serialize(SearchType searchType, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(searchType.getInt());
    }
  }

  private static final class StringValueParser
      implements JsonSerializer<StringValue>, JsonDeserializer<StringValue> {

    @Override
    public JsonElement serialize(StringValue src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      JsonObject json = new JsonObject();
      json.addProperty(src.key, src.value);
      return json;
    }

    @Override
    public StringValue deserialize(JsonElement jsonElement, Type typeOfT,
                                   JsonDeserializationContext context) throws JsonParseException {
      JsonObject json = jsonElement.getAsJsonObject();
      String key;

      key = json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.USER_DEFINED.ordinal()])
          .getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.USERNAME.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.PASSWORD.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.PIN.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.OTP.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.INSTALL_CODE.ordinal()])
              .getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.CSA.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.DSK.ordinal()]).getAsString();
      key = (null != key) ? key :
          json.getAsJsonPrimitive(PARAM_KEY_LUT[Parameter.Key.RESOURCE_NAME.ordinal()])
              .getAsString();

      if (null == key) {
        return null;
      }

      String value = json.getAsJsonPrimitive(key).getAsString();

      return new StringValue(key, value);
    }
  }

  /**
   * An enum denoting possible keys for Info request and Info response.
   * NOTE: when adding a new key make sure that appropriate classes
   *          are added into InfoResponse, InfoRequest and those classes are added into
   *          {@link InfoRequest#PARAM_CLASS_LUT}
   *          {@link InfoResponse#PARAM_CLASS_LUT}.
   */
  public enum Key {
    USER_DEFINED,
    USERNAME,
    PASSWORD,
    PIN,
    OTP,
    INSTALL_CODE,
    CSA,
    DSK,
    RESOURCE_NAME,
    AVAIL_LISTEN_PORT,
    CONTENT_SEARCH,
    CONTENT_PLAYBACK,
    STREAM_BUFFER_SIZES,
    PAIRING_TYPE,
    TIMEZONE,
    REMOTE_PAIRING_CODES,
    REMOTE_PAIR_CAPABILITIES,
    REMOTE_BUTTON_INFO,
    REMOTE_SEARCH;

    static Key getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * An enum denoting type of pairing options available.
   */
  public enum PairingType {
    SMART_CONFIG,
    AP;

    static PairingType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * An enum denoting a filter for type of search.
   */
  public enum SearchType {
    BRAND;

    static SearchType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  interface Value {
  }

  /**
   * This class acts as value for corresponding key {@link Parameter.Key#STREAM_BUFFER_SIZES}.
   */
  public static class BufferSizeValue implements InfoResponse.Value, InfoRequest.Value {
    @SerializedName(Constants.PARAM_KEY_SEND_BUFFER_SIZE)
    public long sendBufferSize;
    @SerializedName(Constants.PARAM_KEY_RECEIVE_BUFFER_SIZE)
    public long receiveBufferSize;

    public BufferSizeValue(long sendBufferSize, long receiveBufferSize) {
      this.sendBufferSize = sendBufferSize;
      this.receiveBufferSize = receiveBufferSize;
    }
  }

  /**
   * A model class providing key and value pair feature.
   * Uses {@link StringValueParser} as adapter to GSON
   */
  public static class StringValue implements InfoResponse.Value, InfoRequest.Value {
    public String key;
    public String value;

    /**
     * A constructor for current class.
     *
     * @param key   Key parameter
     * @param value Value parameter
     */
    public StringValue(String key, String value) {
      this.key = key;
      this.value = value;

      if (null == this.key) {
        throw new IllegalArgumentException();
      }
    }

    public StringValue(Parameter.Key key, String value) {
      this(PARAM_KEY_LUT[key.ordinal()], value);
    }
  }

  /**
   * This class acts as value for corresponding key {@link Parameter.Key#PAIRING_TYPE}.
   */
  public static class PairingTypeValue implements InfoResponse.Value, InfoRequest.Value {
    @SerializedName(Constants.PARAM_KEY_PAIRING_TYPE)
    public PairingType pairingType;

    public PairingTypeValue(PairingType pairingType) {
      this.pairingType = pairingType;
    }
  }

  /**
   * This class acts as value for corresponding key {@link Parameter.Key#TIMEZONE}.
   */
  public static class TimeZoneValue implements InfoResponse.Value, InfoRequest.Value {
    @SerializedName(Constants.PARAM_KEY_TIME_ZONE)
    public String timezone;

    public TimeZoneValue(String timezone) {
      this.timezone = timezone;
    }

    public TimeZoneValue(ZoneId zoneId) {
      this.timezone = zoneId.getId();
    }
  }

  @SerializedName(Constants.PARAMETER_KEY)
  public Key key;

  @SerializedName(Constants.PARAMETER_VALUE)
  public T value;

  public Parameter(Key key, T value) {
    this.key = key;
    this.value = value;
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(Key.class, new KeyParser());
    Command.GSON_BUILDER.registerTypeAdapter(StringValue.class, new StringValueParser());
    Command.GSON_BUILDER.registerTypeAdapter(PairingType.class, new PairingTypeParser());
    Command.GSON_BUILDER.registerTypeAdapter(SearchType.class, new SearchTypeParser());
  }
}
