/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

/**
 * Resource class is blueprint of resource belonging to a device.
 * This class encapsulates all information about a resource such as id, name, manufacturer,
 * power source, protocol, parent device, parent zone and capability map etc.
 */
public class Resource implements Iterable<Capability> {
  /**
   * A class to serialize Resource object to its corresponding JSON.
   */
  private static class ResourceSerializer implements JsonSerializer<Resource> {
    @Override
    public JsonElement serialize(Resource src, Type typeOfSrc, JsonSerializationContext context) {
      // making JsonString manually to avoid unnecessary reflections.
      JsonObject jsonObject = new JsonObject();
      jsonObject.addProperty(Constants.RESOURCE_EUI, src.getId());
      jsonObject.addProperty(Constants.DEVICE_NODE_ID, src.getParent().getId());

      return jsonObject;
    }
  }

  @SerializedName(Constants.RESOURCE_EUI)
  private final String eui;
  @SerializedName(Constants.RESOURCE_NAME)
  private String name;
  @SerializedName(Constants.RESOURCE_METADATA)
  private String metadata;
  @SerializedName(Constants.MANUFACTURER)
  private String manufacturer;
  @SerializedName(Constants.MODEL)
  private String model;
  @SerializedName(Constants.FIRMWARE_VERSION)
  private String firmware;
  @SerializedName(Constants.POWER_SOURCE)
  private PowerSource powerSource;
  @SerializedName(Constants.RECEIVER_TYPE)
  private ReceiverType receiverType;
  @SerializedName(Constants.PROTOCOL_ID)
  private RadioProtocol protocol;

  private transient Device parentDevice;
  private transient Zone parentZone;
  private transient boolean ready = false;

  @SerializedName(Constants.RESOURCE_CAPABILITY_MAP)
  private Map<Capability.CapabilityId, Capability> capabilityMap;

  /**
   * The constructor for current class.
   */
  Resource() {
    this.eui = null;
    this.name = null;
    this.metadata = null;
    this.manufacturer = null;
    this.model = null;
    this.firmware = null;
    this.powerSource = null;
    this.receiverType = null;
    this.parentDevice = null;
  }

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(Resource.class, new ResourceSerializer());
  }

  /**
   * The constructor is protected. So, that the developer cannot create a resource for himself.
   *
   * @param eui          The unique Id of the resource.
   * @param parentDevice The parent device which contains this resource.
   */
  protected Resource(String eui, Device parentDevice, Zone parentZone) {
    this.eui = eui;
    this.parentDevice = parentDevice;
    this.parentZone = parentZone;
    this.powerSource = null;
    this.receiverType = null;
  }

  /**
   * This function is used to get the unique Id of the resource.
   *
   * @return String: The unique identifier of the resource. NOTE: This Id may
   *         collide with Ids from other devices.
   */
  public final String getId() {
    return this.eui;
  }

  /**
   * This function is used to get the name of the resource.
   *
   * @return String: The name of the Resource is returned.
   */
  public String getName() {
    return this.name;
  }

  /**
   * A function to set the name of the current resource.
   *
   * @param name    The name which has to be set.
   * @param listener The listener that will be triggered with callbacks.
   */
  public void setName(String name, Device.DeviceManagementStatusListener listener) {
    parentDevice.setResourceName(this, name, listener);
  }

  /**
   * This function is used to get the manufacturer name of the resource.
   *
   * @return String: The name of the Manufacturer is returned.
   */
  public String getManufacturer() {
    return this.manufacturer;
  }

  /**
   * This function is used to get the model name of the resource.
   *
   * @return String: The Model of the resource.
   */
  public String getModel() {
    return this.model;
  }

  /**
   * This function is used to get the firmware version of the current resource.
   *
   * @return String: The firmware of the resource.
   */
  public String getFirmware() {
    return this.firmware;
  }

  /**
   * This function is used to know how the resource is being powered.
   *
   * @return PowerSource: The enum indicating the power source of the resource.
   */
  public PowerSource getPowerSource() {
    return this.powerSource;
  }

  /**
   * This function is used to get the receiver properties of the resource.
   *
   * @return ReceiverType: The enum indicating the receiver type of the resource.
   */
  public ReceiverType getReceiverType() {
    return this.receiverType;
  }

  /**
   * This function is used to get the Parent device of the resource.
   *
   * @return Device: The parent device of the resource.
   */
  public final <T extends Device> T getParent() {
    return Utils.castUp(this.parentDevice);
  }

  /**
   * This function is used to get the Parent zone of the resource.
   *
   * @return Zone: The parent Zone of the resource.
   */
  public final <T extends Zone> T getParentZone() {
    return Utils.castUp(this.parentZone);
  }

  /**
   * A function to get the metadata of the resource.
   * Metadata is specific to the sub cluster of the application.
   * i.e.., Specific to the developer.
   *
   * @return String: Metadata of the resource
   */
  public String getMetadata() {
    return this.metadata;
  }

  /**
   * A function to set the name of the current resource.
   *
   * @param metadata The data which has to be stored.
   * @param listener  The listener that will be triggered with callbacks.
   */
  public void setMetadata(String metadata, Device.DeviceManagementStatusListener listener) {
    parentDevice.setResourceMetadata(this, metadata, listener);
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the CapabilityId to the Capability Objects.
   * This function can be used to databind the sdk to the application.
   * NOTE: This hashmap is for read-only any misuse may cause the program to
   *      misbehave and even crash.
   *
   * @return The key being the CapabilityId and the value being the Capability Object.
   */
  public Map<Capability.CapabilityId, Capability> getCapabilityMap() {
    if (null == capabilityMap) {
      this.capabilityMap = new EnumMap<>(Capability.CapabilityId.class);
    }

    return this.capabilityMap;
  }

  /**
   * This function is used to return the capability with the given capabilityId.
   *
   * @return capability of the resource with the given capabilityId
   */
  public <T extends Capability> T getCapability(Capability.CapabilityId capId) {
    return Utils.castUp(getCapabilityMap().get(capId));
  }

  /**
   * This function is used to return the capability with the given .
   *
   * @return Attribute of the resource with the given capabilityId and attributeId
   */
  public <T extends Attribute> T getAttribute(Capability.AttributeId attrId) {
    Capability cap = getCapability(attrId.getCapabilityId());
    return (null == cap) ? null : cap.getAttribute(attrId);
  }

  /**
   * This function assists in checking membership of capability in capability map.
   *
   * @param capabilityId identifier of the Capability
   * @return true if capability exists in memory.
   */
  public boolean containsCapability(Capability.CapabilityId capabilityId) {
    return getCapabilityMap().containsKey(capabilityId);
  }

  protected void internalAddCapability(Capability capability) {
    getCapabilityMap().put(capability.getId(), capability);
  }

  protected void internalRemoveCapability(Capability.CapabilityId capabilityId) {
    getCapabilityMap().remove(capabilityId);
  }

  /**
   * A function to suggest if the current resource is dummy or not.
   *
   * @return boolean: true if the object is *NOT* dummy, and the device is ready.
   */
  public boolean isReady() {
    return ready && parentDevice.isReady();
  }

  protected void internalSetName(String name) {
    this.name = name;
  }

  protected void internalSetMetadata(String metadata) {
    this.metadata = metadata;
  }

  protected void internalSetManufacturer(String manufacturer) {
    this.manufacturer = manufacturer;
  }

  protected void internalSetModel(String model) {
    this.model = model;
  }

  protected void internalSetFirmware(String firmware) {
    this.firmware = firmware;
  }

  protected void internalSetPowerSource(PowerSource powerSource) {
    this.powerSource = powerSource;
  }

  protected void internalSetReceiverType(ReceiverType receiverType) {
    this.receiverType = receiverType;
  }

  protected void internalSetParentDevice(Device parentDevice) {
    this.parentDevice = parentDevice;
  }

  protected void internalSetParentZone(Zone parentZone) {
    this.parentZone = parentZone;
  }

  protected void internalMarkAsReady() {
    this.ready = true;
  }

  protected void internalSetProtocol(RadioProtocol protocol) {
    // using this only in case of advertisements as of (6-10-2020)
    this.protocol = protocol;
  }

  protected void internalRemoveResource() {
    // intentionally empty
  }

  public RadioProtocol getProtocol() {
    // may not be set in some cases as of (6-10-2020)
    return protocol;
  }

  @Override
  public final int hashCode() {
    return Objects.hashCode(getId());
  }

  @Override
  public Iterator<Capability> iterator() {
    return getCapabilityMap().values().iterator();
  }

  @Override
  public String toString() {
    return "Resource{"
        + "eui='" + eui + '\''
        + ", name='" + name + '\''
        + ", metadata='" + metadata + '\''
        + ", manufacturer='" + manufacturer + '\''
        + ", model='" + model + '\''
        + ", firmware='" + firmware + '\''
        + ", powerSource=" + powerSource
        + ", receiverType=" + receiverType
        + ", protocol=" + protocol
        + ", ready=" + ready
        + '}';
  }
}
