/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;

/**
 * Resource action defines action to be taken on a resource,
 * which shall be altering attribute value of certain capability of that resource.
 */
public class ResourceAction {
  /**
   * A class to serialize ResourceAction objects to its corresponding JSON.
   */
  private static class ResourceActionParser implements JsonSerializer<ResourceAction>,
      JsonDeserializer<ResourceAction> {
    @Override
    public JsonElement serialize(ResourceAction resourceAction, Type typeOfSrc,
                                 JsonSerializationContext context) {
      JsonObject jsonObject = new JsonObject();

      Capability capability = resourceAction.capability;
      Resource resource = capability.getParent();
      Device device = resource.getParent();
      Network network = device.getParent();

      jsonObject.addProperty(Constants.CAPABILITY_ID, capability.getId().getInt());
      jsonObject.addProperty(Constants.RESOURCE_EUI, resource.getId());
      jsonObject.addProperty(Constants.DEVICE_NODE_ID, device.getId());
      jsonObject.addProperty(Constants.NETWORK_ID, network.getId());
      jsonObject.addProperty(Constants.COMMAND_ID, resourceAction.command.getCommandId().getInt());
      jsonObject.addProperty(Constants.RESOURCE_ACTION_ID, resourceAction.id);

      JsonObject commandJson = resourceAction.command.toJsonObject();
      if (0 < commandJson.entrySet().size()) {
        jsonObject.add(Constants.COMMAND_PARAMS, commandJson);
      }
      return jsonObject;
    }

    @Override

    public ResourceAction deserialize(JsonElement json, Type typeOfT,
                                      JsonDeserializationContext context) throws
        JsonParseException {
      JsonObject jsonObject = json.getAsJsonObject();

      int capabilityId = jsonObject.get(Constants.CAPABILITY_ID).getAsInt();
      String resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();
      long deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
      String networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();

      Capability capability =
          Utils.addMissingCapability(networkId, deviceNodeId, resourceEui, capabilityId, null);

      Command<? extends Capability.CommandId> command =
          capability.createCommand(jsonObject.get(Constants.COMMAND_ID).getAsInt(),
              jsonObject.get(Constants.COMMAND_PARAMS));

      int resourceActionId = jsonObject.get(Constants.RESOURCE_ACTION_ID).getAsInt();
      return new ResourceAction(resourceActionId, capability, command);
    }
  }

  public static final int DEFAULT_RESOURCE_ACTION_ID = 0;

  private final int id;
  private final Command<? extends Capability.CommandId> command;
  private final Capability capability;

  /**
   * This class is used to represent the action.
   *
   * @param capability The capability to which the command will be sent
   * @param command    The command which will get executed
   */
  public ResourceAction(Capability capability, Command<? extends Capability.CommandId> command) {
    this(DEFAULT_RESOURCE_ACTION_ID, capability, command);
  }

  /**
   * An initializer for the current class and adds serializers needed for this.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(ResourceAction.class, new ResourceActionParser());
  }

  /**
   * This class is used to represent the action.
   *
   * @param id         The id of the action
   * @param capability The capability to which the command will be sent
   * @param command    The command which will get executed
   */
  protected ResourceAction(int id, Capability capability,
                           Command<? extends Capability.CommandId> command) {
    this.id = id;
    this.capability = capability;
    this.command = command;
  }

  /**
   * A function to get the identifier of the current object.
   *
   * @return int: The identifier
   */
  public int getId() {
    return this.id;
  }

  /**
   * A function to get the command which get executed while executing the resourceAction.
   *
   * @return Command: The executing command
   */
  public Command<? extends Capability.CommandId> getCommand() {
    return this.command;
  }

  /**
   * A function to get the capability on which the command will be executed.
   *
   * @return Capability: The capability on which the command will be executed.
   */
  public Capability getCapability() {
    return this.capability;
  }

  /**
   * A function to make this ResourceAction to a serialized string.
   * NOTE: This mustn't be used to add/edit existing resourceActions.
   *
   * @return returns a String that represents the json format of the class
   */
  public String toJson() {
    return Command.GSON_BUILDER.create().toJson(this);
  }

  @Override
  public String toString() {
    return "ResourceAction{"
        + "id=" + id
        + ", command=" + Command.GSON_BUILDER.create().toJson(command)
        + ", capability=" + capability
        + '}';
  }

  /**
   * A function to create resource action from JSON.
   *
   * @param jsonAction The stringified Json format of the resourceAction
   * @return ResourceAction: the ResourceAction that has been recreated from the json string
   */
  public static ResourceAction fromJson(String jsonAction) {
    return Command.GSON_BUILDER.create().fromJson(jsonAction, ResourceAction.class);
  }
}
