/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;

/**
 * Resource condition encapsulates the information about the condition of a resource
 * upon which Scene and Rule can be triggered.
 */
public class ResourceCondition implements Rule.Condition {
  private static class ResourceConditionParser implements JsonSerializer<ResourceCondition>,
      JsonDeserializer<ResourceCondition> {
    @Override
    public ResourceCondition deserialize(JsonElement json, Type typeOfT,
                                         JsonDeserializationContext context)
        throws JsonParseException {

      JsonObject jsonObject = json.getAsJsonObject();
      int id;

      String networkId;
      long deviceNodeId;

      Object thresholdAttributeValue;
      JsonElement attributeValueJson = jsonObject.get(Constants.ATTRIBUTE_VALUE);

      id = jsonObject.get(Constants.RESOURCE_CONDITION_ID).getAsInt();
      networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
      deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
      String resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();
      int capabilityId = jsonObject.get(Constants.CAPABILITY_ID).getAsInt();

      Capability capability =
          Utils.addMissingCapability(networkId, deviceNodeId, resourceEui, capabilityId, null);

      Gson gson = Command.GSON_BUILDER.create();

      int attributeId = jsonObject.get(Constants.ATTRIBUTE_ID).getAsInt();
      int operatorType = jsonObject.get(Constants.RULE_OPERATOR_ID).getAsInt();
      long conditionDuration = jsonObject.get(Constants.RULE_CONDITION_DURATION).getAsLong();
      Attribute.DataType attributeDataType =
          gson.fromJson(jsonObject.get(Constants.RULE_VALUE_DATA_TYPE), Attribute.DataType.class);

      switch (attributeDataType) {
        case BOOLEAN:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, boolean.class);
          break;
        case BOOLEAN_ARR:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, boolean[].class);
          break;
        case STRING:
        case JSON:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, String.class);
          break;
        case STRING_ARR:
        case JSON_ARR:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, String[].class);
          break;
        case FLOAT:
        case DOUBLE:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, double.class);
          break;
        case FLOAT_ARR:
        case DOUBLE_ARR:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, double[].class);
          break;
        case UINT8:
        case UINT16:
        case UINT32:
        case UINT64:
        case INT8:
        case INT16:
        case INT32:
        case INT64:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, int.class);
          break;
        case UINT8_ARR:
        case UINT16_ARR:
        case UINT32_ARR:
        case UINT64_ARR:
        case INT8_ARR:
        case INT16_ARR:
        case INT32_ARR:
        case INT64_ARR:
          thresholdAttributeValue = gson.fromJson(attributeValueJson, int[].class);
          break;
        default:
          thresholdAttributeValue = null;
      }

      return new ResourceCondition(id, capability, attributeId, operatorType, conditionDuration,
          attributeDataType, thresholdAttributeValue);
    }

    @Override
    public JsonElement serialize(ResourceCondition condition, Type typeOfSrc,
                                 JsonSerializationContext context) {
      JsonObject jsonObject = new JsonObject();
      Gson gson = Command.GSON_BUILDER.create();

      // networkId need not be added
      jsonObject.addProperty(Constants.DEVICE_NODE_ID,
          condition.capability.getParent().getParent().getId());
      jsonObject.addProperty(Constants.RESOURCE_EUI, condition.capability.getParent().getId());
      jsonObject.addProperty(Constants.CAPABILITY_ID, condition.capability.getId().getInt());

      jsonObject.addProperty(Constants.RESOURCE_CONDITION_ID, condition.id);
      jsonObject.addProperty(Constants.ATTRIBUTE_ID, condition.attributeId);
      jsonObject.addProperty(Constants.RULE_OPERATOR_ID, condition.operatorType.getInt());
      jsonObject.addProperty(Constants.RULE_CONDITION_DURATION, condition.conditionDuration);

      jsonObject.add(Constants.RULE_VALUE_DATA_TYPE, gson.toJsonTree(condition.dataType));
      jsonObject.add(Constants.ATTRIBUTE_VALUE, gson.toJsonTree(condition.thresholdAttributeValue));

      return jsonObject;
    }
  }

  /**
   * An enum defining inequality operators applicable to decide the resource condition.
   */
  public enum OperatorType {
    EQUAL_TO,
    NOT_EQUAL_TO,
    GREATER_THAN,
    GREATER_THAN_EQUAL_TO,
    LESS_THAN,
    LESS_THAN_EQUAL_TO;

    static OperatorType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  private final int id;
  Capability capability;
  int attributeId;
  OperatorType operatorType;
  long conditionDuration; // value in millis
  Attribute.DataType dataType;
  Object thresholdAttributeValue;

  public static final int DEFAULT_RESOURCE_CONDITION_ID = 0;

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(ResourceCondition.class,
        new ResourceConditionParser());
  }

  protected ResourceCondition(int id, Capability capability, int attributeId, int operatorType,
                              long conditionDuration, Attribute.DataType dataType,
                              Object thresholdAttributeValue) {
    this.id = id;
    this.capability = capability;
    this.attributeId = attributeId;
    this.operatorType = OperatorType.getEnum(operatorType);
    this.conditionDuration = conditionDuration;

    // validation between dataType and threshold value instanceof
    this.dataType = dataType;
    this.thresholdAttributeValue = thresholdAttributeValue;
  }

  public ResourceCondition(Resource resource, Capability.AttributeId attributeId,
                           OperatorType operatorType,
                           long conditionDuration, Attribute.DataType dataType,
                           Object thresholdAttributeValue) {
    this(resource.getCapability(attributeId.getCapabilityId()), attributeId, operatorType,
        conditionDuration, dataType, thresholdAttributeValue);
  }

  public ResourceCondition(Capability capability, Capability.AttributeId attributeId,
                           OperatorType operatorType,
                           long conditionDuration, Attribute.DataType dataType,
                           Object thresholdAttributeValue) {
    this(DEFAULT_RESOURCE_CONDITION_ID, capability, attributeId.getInt(), operatorType.getInt(),
        conditionDuration, dataType, thresholdAttributeValue);
  }

  public int getId() {
    return this.id;
  }

  public Capability.AttributeId getAttributeId() {
    return this.capability.getId().getAttributeId(attributeId);
  }

  public Capability getParentCapability() {
    return this.capability;
  }

  public Attribute getAttribute() {
    Capability parent = getParentCapability();
    return (null == parent) ? null : capability.getAttribute(getAttributeId());
  }

  public OperatorType getOperatorType() {
    return this.operatorType;
  }

  public long getConditionDuration() {
    return this.conditionDuration;
  }

  public Attribute.DataType getAttributeDataType() {
    return this.dataType;
  }

  public Object getThresholdAttributeValue() {
    return this.thresholdAttributeValue;
  }

  @Override
  public ResourceCondition duplicate() {
    return new ResourceCondition(DEFAULT_RESOURCE_CONDITION_ID, capability, attributeId,
        operatorType.getInt(), conditionDuration, dataType, thresholdAttributeValue);
  }

  @Override
  public String toString() {
    return "ResourceCondition{"
        + "id=" + id
        + ", capability=" + capability
        + ", attributeId=" + attributeId
        + ", operatorType=" + operatorType
        + ", conditionDuration=" + conditionDuration
        + ", dataType=" + dataType
        + ", thresholdAttributeValue=" + thresholdAttributeValue
        + '}';
  }

  /**
   * This class serves a changeable version of Resource Action.
   */
  public static class EditableResourceCondition extends ResourceCondition {

    public EditableResourceCondition(Attribute attribute, OperatorType operatorType,
                                     long conditionDuration, Object thresholdAttributeValue) {
      super(attribute.getParent(), attribute.getId(), operatorType, conditionDuration,
          attribute.getDataType(), thresholdAttributeValue);
    }

    public void setCapability(Capability capability) {
      this.capability = capability;
    }

    public void setAttributeId(Capability.AttributeId attributeId) {
      this.attributeId = attributeId.getInt();
    }

    public void setOperatorType(OperatorType operatorType) {
      this.operatorType = operatorType;
    }

    public void setConditionDuration(long duration) {
      this.conditionDuration = duration;
    }

    public void setDataType(Attribute.DataType dataType) {
      this.dataType = dataType;
    }

    public void setValue(Object value) {
      this.thresholdAttributeValue = value;
    }
  }
}
