/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Rule class captures the rule defined by the user in a <b>network</b>, which when triggers
 * upon certain conditions({@link ScheduleCondition})
 * causes an action({@link ResourceAction}) defined on the resource.
 */
public class Rule {
  private static class RuleParser implements JsonSerializer<Rule>, JsonDeserializer<Rule> {
    @Override
    public Rule deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {

      JsonObject jsonObject = json.getAsJsonObject();
      Gson gson = Command.GSON_BUILDER.create();

      int id = jsonObject.get(Constants.RULE_ID).getAsInt();
      String name = jsonObject.get(Constants.RULE_NAME).getAsString();
      ArrayList<ResourceCondition> resourceConditions =
          gson.fromJson(jsonObject.get(Constants.RESOURCE_CONDITIONS),
              new TypeToken<ArrayList<ResourceCondition>>() {}.getType());
      ArrayList<ScheduleCondition> scheduleConditions =
          gson.fromJson(jsonObject.get(Constants.SCHEDULE_CONDITIONS),
              new TypeToken<ArrayList<ScheduleCondition>>() {}.getType());
      ArrayList<ResourceAction> resourceActions =
          gson.fromJson(jsonObject.get(Constants.RESOURCE_ACTIONS),
              new TypeToken<ArrayList<ResourceAction>>() {}.getType());
      ArrayList<Scene> sceneActions =
          gson.fromJson(jsonObject.get(Constants.SCENE_ACTIONS),
              new TypeToken<ArrayList<Scene>>(){}.getType());

      Rule rule = Factory.createRule(id, null);

      rule.internalSetName(name);
      rule.internalClearAddResourceConditions(resourceConditions);
      rule.internalClearAddScheduleConditions(scheduleConditions);
      rule.internalClearAddResourceActions(resourceActions);
      rule.internalClearAddSceneActions(sceneActions);

      return rule;
    }

    @Override
    public JsonElement serialize(Rule rule, Type typeOfSrc, JsonSerializationContext context) {
      Gson gson = Command.GSON_BUILDER.create();
      JsonObject jsonObject = new JsonObject();

      jsonObject.addProperty(Constants.RULE_ID, rule.id);
      jsonObject.addProperty(Constants.RULE_NAME, rule.name); // can be removed for speed up

      jsonObject.add(Constants.RESOURCE_ACTIONS, gson.toJsonTree(rule.getResourceActions()));
      jsonObject.add(Constants.SCENE_ACTIONS, gson.toJsonTree(rule.getSceneActions()));
      jsonObject.add(Constants.RESOURCE_CONDITIONS, gson.toJsonTree(rule.getResourceConditions()));
      jsonObject.add(Constants.SCHEDULE_CONDITIONS, gson.toJsonTree(rule.getScheduleConditions()));

      return jsonObject;
    }
  }

  private final int id;
  private List<ResourceCondition> resourceConditions;
  private List<ScheduleCondition> scheduleConditions;
  private List<ResourceAction> resourceActions;
  private List<Scene> sceneActions;
  private String name;

  private transient Network parent;
  private transient boolean ready = false;

  public static final int DEFAULT_RULE_ID = 0;

  protected Rule(int id, Network parent) {
    this.id = id;
    this.parent = parent;
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(Rule.class, new RuleParser());
  }

  public int getId() {
    return this.id;
  }

  public String getName() {
    return this.name;
  }

  public Network getParent() {
    return this.parent;
  }

  /**
   * A function to fetch resource conditions for this rule.
   *
   * @return list of Resource conditions defined by user
   */
  public List<ResourceCondition> getResourceConditions() {
    if (null == this.resourceConditions) {
      this.resourceConditions = Collections.synchronizedList(new ArrayList<>());
    }

    return this.resourceConditions;
  }

  /**
   * This function fetches the schedule conditions set for this rule.
   *
   * @return list of schedule conditions set for this rule
   */
  public List<ScheduleCondition> getScheduleConditions() {
    if (null == this.scheduleConditions) {
      this.scheduleConditions = Collections.synchronizedList(new ArrayList<>());
    }

    return this.scheduleConditions;
  }

  /**
   * This function fetches the resource actions set for this rule.
   *
   * @return list of resource actions set for this rule.
   */
  public List<ResourceAction> getResourceActions() {
    if (null == this.resourceActions) {
      this.resourceActions = Collections.synchronizedList(new ArrayList<>());
    }

    return this.resourceActions;
  }

  /**
   * This function fetches the scene actions for this rule.
   *
   * @return list of scene actions defined by the user for this rule
   */
  public List<Scene> getSceneActions() {
    if (null == this.sceneActions) {
      this.sceneActions = Collections.synchronizedList(new ArrayList<>());
    }

    return this.sceneActions;
  }

  /**
   * A function suggesting if the rule is made from {@link NativeCallbacks#ruleInfoCallback}
   * or because of state retrieval.
   *
   * @return boolean: state if the rule is dummy or not.
   */
  public boolean isReady() {
    return ready;
  }

  protected void internalClearAddResourceConditions(
      Collection<ResourceCondition> resourceConditions) {
    getResourceConditions().clear();
    getResourceConditions().addAll(resourceConditions);
  }

  protected void internalClearAddScheduleConditions(
      Collection<ScheduleCondition> scheduleConditions) {
    getScheduleConditions().clear();
    getScheduleConditions().addAll(scheduleConditions);
  }

  protected void internalClearAddResourceActions(Collection<ResourceAction> resourceActions) {
    getResourceActions().clear();
    getResourceActions().addAll(resourceActions);
  }

  protected void internalClearAddSceneActions(Collection<Scene> scenesActions) {
    getSceneActions().clear();
    getSceneActions().addAll(scenesActions);
  }

  protected void internalSetName(String name) {
    this.name = name;
  }

  protected void internalSetParent(Network parent) {
    this.parent = parent;
  }

  protected void internalMarkAsReady() {
    this.ready = true;
  }

  @Override
  public final int hashCode() {
    return getId();
  }

  @Override
  public String toString() {
    String networkId = (null == parent) ? "?" : parent.getId();

    return "Rule{"
        + "networkId=" + networkId
        + ", id=" + id
        + ", name='" + name + '\''
        + ", ready=" + ready
        + ", resourceConditions=" + resourceConditions
        + ", scheduleConditions=" + scheduleConditions
        + ", resourceActions=" + resourceActions
        + ", sceneActions=" + sceneActions
        + '}';
  }

  /**
   * An interface to be implemented by {@link ScheduleCondition} and {@link ResourceCondition}.
   */
  public interface Condition {
    Condition duplicate();
  }
}
