/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Scene class captures the scene defined by the user in a <b>network</b>, which when triggered
 * causes an action defined on the resource .
 */
public class Scene implements Iterable<ResourceAction> {

  /**
   * TODO: find if this parser can be deleted.
   *
   * @implNote This is *NOT* used by {@link NativeCallbacks#sceneInfoCallback}.
   */
  private static class SceneParser implements JsonSerializer<Scene>, JsonDeserializer<Scene> {
    @Override
    public JsonElement serialize(Scene scene, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(scene.sceneId);
    }

    @Override
    public Scene deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws
        JsonParseException {
      JsonObject jsonObject = json.getAsJsonObject();
      String networkId;
      int sceneId;

      // deserialization needs networkId along with sceneId
      networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
      sceneId = jsonObject.get(Constants.SCENE_ID).getAsInt();

      return Utils.addMissingScene(networkId, sceneId);
    }
  }

  public static final int DEFAULT_SCENE_ID = 0;

  private final int sceneId;
  private Map<Integer, ResourceAction> resourceActionMap;

  private String name;
  private String metadata;

  private transient Network parent;
  private transient boolean ready = false;

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(Scene.class, new SceneParser());
  }

  /**
   * The constructor for the current class.
   *
   * @param sceneId The unique identifier of the object
   * @param parent  The parent network of the scene
   */
  protected Scene(int sceneId, Network parent) {
    this.sceneId = sceneId;
    this.parent = parent;
  }

  /**
   * A function to get the Id of the scene.
   *
   * @return int: sceneId of the current scene
   */
  public int getId() {
    return this.sceneId;
  }

  /**
   * A function to get the name of the scene.
   *
   * @return String: The name of the current scene
   */
  public String getName() {
    return this.name;
  }

  /**
   * A function to set the name of the scene.
   *
   * @param name     The name that has to be assigned
   * @param listener The listener that will be triggered with
   *                 {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void setName(String name, Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(new Network.SetScene(this, name), listener);
  }

  public void addResourceActions(ResourceAction[] resourceActions,
                                 Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(new Network.AddSceneResourceAction(this, resourceActions),
        listener);
  }

  public void updateResourceActions(int[] resourceActionIds, ResourceAction[] resourceActions,
                                    Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(
        new Network.UpdateSceneResourceActions(this, resourceActionIds, resourceActions), listener);
  }

  public void saveResourceActions(ResourceAction[] resourceActions,
                                  Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(new Network.SaveSceneResourceActions(this, resourceActions),
        listener);
  }

  public void removeResourceActions(ResourceAction[] resourceActions,
                                    Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(
        new Network.RemoveSceneResourceActions(this, resourceActions), listener);
  }

  /**
   * A function to set the metadata of the current scene.
   *
   * @param metadata The metadata to be assigned
   * @param listener The listener that will be triggered with
   *                 {@link CallbacksInterface#networkManagementCommandCallback}
   */
  public void setMetadata(String metadata, Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(new Network.SetSceneMetadata(this, metadata), listener);
  }

  /**
   * A function to execute the scene.
   *
   * @throws RuntimeException on connection failures and other similar errors
   */
  public void execute() {
    CocoClient.getInstance().getNativeHandler().executeScene(this);
  }

  /**
   * A function to return the metadata of the scene.
   *
   * @return String: The metadata of the scene
   */
  public String getMetadata() {
    return metadata;
  }

  /**
   * A function to get the parent network of the current scene.
   *
   * @return Network: The parent network of the scene
   */
  public Network getParent() {
    return this.parent;
  }

  /**
   * A function to get the actions that get executed with the scene.
   *
   * @return The map containing the IDs and the actions of the scene
   */
  public Map<Integer, ResourceAction> getResourceActionMap() {
    if (null == this.resourceActionMap) {
      this.resourceActionMap = new ConcurrentHashMap<>();
    }

    return this.resourceActionMap;
  }

  public ResourceAction getResourceAction(int resourceActionId) {
    return getResourceActionMap().get(resourceActionId);
  }

  protected void internalAddResourceAction(ResourceAction resourceAction) {
    getResourceActionMap().put(resourceAction.getId(), resourceAction);
  }

  public boolean containsResourceAction(int resourceActionId) {
    return getResourceActionMap().containsKey(resourceActionId);
  }

  protected void internalUpdateResourceAction(ResourceAction resourceAction) {
    internalRemoveResourceAction(resourceAction.getId());
    internalAddResourceAction(resourceAction);
  }

  protected void internalRemoveResourceAction(int resourceActionId) {
    getResourceActionMap().remove(resourceActionId);
  }

  @Override
  public final int hashCode() {
    return getId();
  }

  /**
   * A function suggesting if the scene is made from {@link NativeCallbacks#sceneInfoCallback}
   * or because of state retrieval.
   *
   * @return boolean: state if the scene is dummy or not.
   */
  public boolean isReady() {
    return ready;
  }

  protected void internalSetName(String name) {
    this.name = name;
  }

  protected void internalSetMetadata(String metadata) {
    this.metadata = metadata;
  }

  protected void internalSetParent(Network parent) {
    this.parent = parent;
  }

  protected void internalMarkAsReady() {
    this.ready = true;
  }

  @Override
  public String toString() {
    String networkId = (null == parent) ? "?" : parent.getId();

    return "Scene{"
        + "networkId=" + networkId
        + ", sceneId=" + sceneId
        + ", name='" + name + '\''
        + ", metadata='" + metadata + '\''
        + ", ready=" + ready
        + ", resourceActionMap=" + resourceActionMap
        + '}';
  }

  @Override
  public Iterator<ResourceAction> iterator() {
    return getResourceActionMap().values().iterator();
  }
}
