/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;

/**
 * Schedule condition encapsulates the information about time and duration of a resource
 * upon which Scene and Rule can be triggered.
 */
public class ScheduleCondition implements Rule.Condition {

  private static final class ScheduleConditionParser
      implements JsonSerializer<ScheduleCondition>, JsonDeserializer<ScheduleCondition> {
    @Override
    public ScheduleCondition deserialize(JsonElement json, Type typeOfT,
                                         JsonDeserializationContext context)
        throws JsonParseException {
      JsonObject jsonObject = json.getAsJsonObject();
      int id = jsonObject.get(Constants.SCHEDULE_CONDITION_ID).getAsInt();
      ScheduleType scheduleType =
          ScheduleType.getEnum(jsonObject.get(Constants.SCHEDULE_TYPE_ID).getAsInt());
      Instant startTime =
          context.deserialize(jsonObject.get(Constants.SCHEDULE_START_TIME), Instant.class);
      Instant expiryTime =
          context.deserialize(jsonObject.get(Constants.SCHEDULE_EXPIRY_TIME), Instant.class);

      return new ScheduleCondition(id, startTime, expiryTime, scheduleType);
    }

    @Override
    public JsonElement serialize(ScheduleCondition condition, Type typeOfSrc,
                                 JsonSerializationContext context) {
      JsonObject json = new JsonObject();

      json.addProperty(Constants.SCHEDULE_CONDITION_ID, condition.getId());
      json.addProperty(Constants.SCHEDULE_TYPE_ID, condition.getScheduleType().getInt());

      json.add(Constants.SCHEDULE_EXPIRY_TIME, context.serialize(condition.expiryTime));
      json.add(Constants.SCHEDULE_START_TIME, context.serialize(condition.startTime));

      return json;
    }
  }

  /**
   * Type of scheduling options for a schedule condition that a user can set.
   */
  public enum ScheduleType {
    ONETIME,
    DAILY,
    WEEKLY,
    MONTHLY,
    YEARLY;

    static ScheduleType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  private final int id;
  private final Instant startTime;
  private final Instant expiryTime;
  private final ScheduleType scheduleType;

  public static final int DEFAULT_SCHEDULE_CONDITION_ID = 0;

  ScheduleCondition(int id, Instant startTime, Instant expiryTime, ScheduleType scheduleType) {
    this.id = id;
    this.startTime = startTime;
    this.expiryTime = expiryTime;
    this.scheduleType = scheduleType;
  }

  private ScheduleCondition(Instant startTime, LocalTime expiryTime, ScheduleType scheduleType) {
    this(DEFAULT_SCHEDULE_CONDITION_ID, startTime, getExpiryTimeInstant(startTime, expiryTime),
        scheduleType);
  }

  private static Instant getExpiryTimeInstant(Instant startTime, LocalTime expiryTime) {
    return LocalDateTime.of(startTime.atZone(ZoneOffset.UTC).toLocalDate(), expiryTime)
        .toInstant(ZoneOffset.UTC);
  }

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(ScheduleCondition.class,
        new ScheduleConditionParser());
  }

  public int getId() {
    return id;
  }

  public Instant getStartTime() {
    return startTime;
  }

  Instant getInstantExpiryTime() {
    return expiryTime;
  }

  public LocalTime getExpiryTime() {
    return ZonedDateTime.ofInstant(expiryTime, ZoneOffset.UTC).toLocalTime();
  }

  public ScheduleType getScheduleType() {
    return scheduleType;
  }

  @Override
  public ScheduleCondition duplicate() {
    return new ScheduleCondition(DEFAULT_SCHEDULE_CONDITION_ID, startTime, expiryTime,
        scheduleType);
  }

  @Override
  public String toString() {
    return "ScheduleCondition{"
        + "id=" + id
        + ", startTime=" + startTime
        + ", expiryTime=" + expiryTime
        + ", scheduleType=" + scheduleType
        + '}';
  }

  /**
   * Builder design pattern to create a Schedule Condition object.
   */
  public static final class Builder {
    private Instant startTime;
    private LocalTime expiryTime;
    private ScheduleCondition.ScheduleType scheduleType;

    /**
     * Constructor for the builder.
     */
    public Builder() {
      startTime = Instant.now();
      expiryTime = LocalTime.MAX;
      scheduleType = ScheduleType.ONETIME;
    }

    public ScheduleCondition build() {
      return new ScheduleCondition(startTime, expiryTime, scheduleType);
    }

    public Builder setStartTime(Instant startTime) {
      this.startTime = startTime;
      return this;
    }

    public Builder setExpiryTime(LocalTime expiryTime) {
      this.expiryTime = expiryTime;
      return this;
    }

    public Builder setScheduleType(ScheduleType scheduleType) {
      this.scheduleType = scheduleType;
      return this;
    }

    public Instant getStartTime() {
      return startTime;
    }

    public LocalTime getExpiryTime() {
      return expiryTime;
    }

    public ScheduleType getScheduleType() {
      return scheduleType;
    }
  }
}
