/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.Objects;

/**
 * This class represents a recording in a storage node and contains data about Storage.
 */
public class StorageContentMetadata {

  /**
   * An enum denoting type of content in the storage.
   */
  public enum ContentType {
    ALL,
    STREAM,
    ATTRIBUTE,
    FILE;

    private static final int offset = -1;

    static ContentType getEnum(int index) {
      return Utils.findEnum(index - offset, values());
    }

    int getInt() {
      return ordinal() + offset;
    }
  }

  /**
   * An enum denoting options for type for trigger for content.
   */
  public enum TriggerType {
    ALL,
    USER,
    ATTR,
    SCENE,
    RULE;

    private static final int offset = -1;

    static TriggerType getEnum(int index) {
      return Utils.findEnum(index - offset, values());
    }

    int getInt() {
      return ordinal() + offset;
    }
  }

  @SerializedName(Constants.CONTENT_ID)
  public final long contentId;
  @SerializedName(Constants.CONTENT_TYPE)
  public final ContentType contentType;
  @SerializedName(Constants.UPLOAD_TRIGGER_TYPE)
  public final TriggerType triggerType;
  @SerializedName(Constants.SIZE)
  public final long size;                  // in MB
  @SerializedName(Constants.STATIC_LINK)
  public final String staticLink;
  @SerializedName(Constants.USER_METADATA)
  public final String metadata;

  @SerializedName(Constants.SOURCE_URI)
  public final Identifier.SourceIdentifier sourceUri;
  @SerializedName(Constants.UPLOAD_TRIGGERED_URI)
  public final Identifier.TriggerIdentifier uploadTriggeredUri;

  // instant serialization already part of Command
  @SerializedName(Constants.CREATED_TIMESTAMP)
  public final Instant createdTimeStamp;

  @SerializedName(Constants.CONTENT_DURATION)
  public final long duration; // milli sec

  protected StorageContentMetadata(long contentId, Identifier.SourceIdentifier sourceUri,
                                   int contentType, int triggerType,
                                   Identifier.TriggerIdentifier triggerUri,
                                   long size, String staticLink, String metadata,
                                   long createdTimeStamp, long duration) {
    this.contentId = contentId;
    this.sourceUri = sourceUri;
    this.contentType = ContentType.getEnum(contentType);
    this.triggerType = TriggerType.getEnum(triggerType);
    this.uploadTriggeredUri = triggerUri;
    this.size = size;
    this.staticLink = staticLink;
    this.metadata = metadata;
    this.duration = duration;
    this.createdTimeStamp = Instant.ofEpochSecond(createdTimeStamp);
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(ContentType.class, new ContentTypeParser());
    Command.GSON_BUILDER.registerTypeAdapter(TriggerType.class, new TriggerTypeParser());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    StorageContentMetadata that = (StorageContentMetadata) o;

    return contentId == that.contentId
        && size == that.size
        && duration == that.duration
        && contentType == that.contentType
        && triggerType == that.triggerType
        && Objects.equals(staticLink, that.staticLink)
        && Objects.equals(metadata, that.metadata)
        && Objects.equals(sourceUri, that.sourceUri)
        && Objects.equals(uploadTriggeredUri, that.uploadTriggeredUri)
        && Objects.equals(createdTimeStamp, that.createdTimeStamp);
  }

  @Override
  public int hashCode() {
    return Objects.hash(contentId, contentType, triggerType, size, staticLink, metadata, sourceUri,
        uploadTriggeredUri, createdTimeStamp, duration);
  }

  @Override
  public String toString() {
    return "StorageContentMetadata{"
        + "contentId=" + contentId
        + ", contentType=" + contentType
        + ", triggerType=" + triggerType
        + ", size=" + size
        + ", staticLink='" + staticLink + '\''
        + ", metadata='" + metadata + '\''
        + ", sourceUri=" + sourceUri
        + ", uploadTriggeredUri=" + uploadTriggeredUri
        + ", createdTimeStamp=" + createdTimeStamp
        + ", duration=" + duration
        + '}';
  }

  private static final class ContentTypeParser implements JsonSerializer<ContentType>,
      JsonDeserializer<ContentType> {

    @Override
    public ContentType deserialize(JsonElement json, Type typeOfT,
                                   JsonDeserializationContext context) throws
        JsonParseException {
      return ContentType.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(ContentType src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  private static final class TriggerTypeParser
      implements JsonSerializer<TriggerType>, JsonDeserializer<TriggerType> {

    @Override
    public TriggerType deserialize(JsonElement json, Type typeOfT,
                                   JsonDeserializationContext context) throws JsonParseException {
      return TriggerType.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(TriggerType src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }
}
