/*
 * @author    Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.Map;

class Utils {

  @SuppressWarnings("unchecked")
  static <T, U extends T> U castUp(T t) {
    return (U) t;
  }

  /**
   * A function used to create dummy network objects if the device
   * points to a network which has not been created yet, in case of network
   * which is already connected it will be returned.
   *
   * @param networkId The networkId of the network.
   * @return The network object which the device should belong to.
   */
  static Network addMissingNetwork(String networkId) {
    if (!CocoClient.getInstance().containsNetwork(networkId)) {

      CocoClient.getInstance().internalAddNetwork(Factory.createNetwork(networkId));

      CallbackMultiplexer.getInstance().connectStatusCallback(CocoClient.getInstance()
          .getNetwork(networkId));
    }

    return CocoClient.getInstance().getNetwork(networkId);
  }

  /**
   * A function used to create dummy device objects, If the device is
   * already created and present in the map then it will be returned.
   *
   * @param networkId    The network Id to which the device should belong to
   * @param deviceNodeId The Id of the device.
   * @return The device object to which the resource should belong to.
   */
  static Device addMissingDevice(String networkId, long deviceNodeId) {
    Network network = addMissingNetwork(networkId);

    if (!network.containsDevice(deviceNodeId)) {
      Device device;

      network.internalAddDevice(device = Factory.createDevice(deviceNodeId, network));

      CallbackMultiplexer.getInstance().deviceInfoCallback(device);
    }

    return network.getDevice(deviceNodeId);
  }

  /**
   * A function used to create dummy resource objects, If the resource is
   * already present then the same object will be returned.
   *
   * @param networkId    The network Id to which the device should belong to
   * @param deviceNodeId The Id of the device.
   * @param resourceEui  The Eui of the resource.
   * @return The resource object to which the capability belongs.
   */
  static Resource addMissingResource(String networkId, long deviceNodeId,
                                     String resourceEui, Object developerContext) {

    Network parentNetwork = addMissingNetwork(networkId);
    Zone parentZone = parentNetwork.getZone(Zone.DEFAULT_ZONE_ID);

    return addMissingResource(networkId, deviceNodeId, resourceEui, parentZone, developerContext);
  }

  static Resource addMissingResource(String networkId, long deviceNodeId, String resourceEui,
                                     Zone parentZone, Object developerContext) {

    Device device = addMissingDevice(networkId, deviceNodeId);

    if (!device.containsResource(resourceEui)) {

      Resource resource = Factory.createResource(resourceEui, device, parentZone);

      device.internalAddResource(resource);
      parentZone.internalAddResource(resource);

      CallbackMultiplexer.getInstance().resourceCallback(resource);
    }

    return device.getResource(resourceEui);
  }

  /**
   * A function used to create dummy capability objects, If the capability
   * is already present then it will be returned.
   *
   * @param networkId    The network Id to which the device should belong to
   * @param deviceNodeId The Id of the device.
   * @param resourceEui  The Eui of the resource.
   * @param capabilityId The Id of the capability.
   * @return The capability object to which the attribute belongs
   */
  static Capability addMissingCapability(String networkId, long deviceNodeId, String resourceEui,
                                         int capabilityId, Object developerContext) {

    Capability capability;
    Resource resource = addMissingResource(networkId, deviceNodeId, resourceEui, developerContext);

    if (!resource.containsCapability(capIdConverter(capabilityId))) {
      capability = Factory.createCapability(capabilityId, resource);

      resource.internalAddCapability(capability);
      CallbackMultiplexer.getInstance().resourceCapabilityCallback(capability);
    } else {
      capability = resource.getCapability(capIdConverter(capabilityId));
    }

    return capability;
  }

  /**
   * A function to create dummy scenes, if the rule has come by and scene ain't.
   *
   * @param networkId The network Id to which the scene belongs to
   * @param sceneId   The id of the scene
   * @return The scene corresponding to the networkId and sceneId
   */
  static Scene addMissingScene(String networkId, int sceneId) {
    Scene scene;
    Network parent = addMissingNetwork(networkId);

    if (!parent.containsScene(sceneId)) {

      scene = Factory.createScene(sceneId, parent);

      parent.internalAddScene(scene);
    } else {
      scene = parent.getScene(sceneId);
    }

    return scene;
  }

  /**
   * A utility function to get scene specified by the networkId, sceneId.
   *
   * @param networkId A string that specifies the networkId
   * @param sceneId   An integer id of the scene
   * @return The scene object corresponding to the networkId and sceneId
   */
  static Scene getScene(String networkId, int sceneId) {
    Network parent = getNetwork(networkId);

    return (null == parent) ? null : parent.getScene(sceneId);
  }

  /**
   * A utility function to get network specified by the networkId, zoneId.
   *
   * @param networkId A string that specifies the networkId
   * @param zoneId    An integer id of the zone
   * @return The zone object corresponding to the networkId and zoneId
   */
  static Zone getZone(String networkId, int zoneId) {
    Network parent = getNetwork(networkId);

    return (null == parent) ? null : parent.getZone(zoneId);
  }

  /**
   * A utility function to get rule specified by the networkId.
   *
   * @param networkId A string that specifies the networkId
   * @param ruleId    An integer id of the rule
   * @return The rule object corresponding to the networkId and ruleId
   */
  static Rule getRule(String networkId, int ruleId) {
    Network parent = getNetwork(networkId);

    return (null == parent) ? null : parent.getRule(ruleId);
  }

  /**
   * A utility function to get network specified by the networkId.
   *
   * @param networkId A string that specifies the networkId
   * @return Network The network object corresponding to the networkId
   */
  static Network getNetwork(String networkId) {
    return getNetwork(CocoClient.getInstance().getNetworkMap(), networkId);
  }

  /**
   * A utility function to get network specified by the networkId.
   *
   * @param networkMap The map which is to be used for search
   * @param networkId  A string that specifies the networkId
   * @return Network The network object corresponding to the networkId
   */
  static Network getNetwork(Map<String, Network> networkMap, String networkId) {
    return networkMap.get(networkId);
  }

  /**
   * A utility function to get device specified by the corresponding identifiers.
   *
   * @param networkId    A string that specifies the networkId
   * @param deviceNodeId The id of the device
   * @return The Device object corresponding to the networkId & deviceNodeId
   */
  static Device getDevice(String networkId, long deviceNodeId) {
    return getDevice(CocoClient.getInstance().getNetworkMap(), networkId, deviceNodeId);
  }

  /**
   * A utility function to get device specified by the corresponding identifiers.
   *
   * @param networkMap   The map which is to be used for search
   * @param networkId    A string that specifies the networkId
   * @param deviceNodeId The id of the device
   * @return The Device object corresponding to the networkId & deviceNodeId
   */
  static Device getDevice(Map<String, Network> networkMap, String networkId, long deviceNodeId) {
    Network parentNetwork = getNetwork(networkMap, networkId);
    return (null == parentNetwork) ? null : parentNetwork.getDevice(deviceNodeId);
  }

  /**
   * A utility function to get resource specified by the corresponding identifiers.
   *
   * @param networkId    A string that specifies the networkId
   * @param deviceNodeId The id of the device
   * @param resourceEui  The id of the resource
   * @return The Resource object corresponding to the networkId & deviceNodeId & resourceEui
   */
  static Resource getResource(String networkId, long deviceNodeId, String resourceEui) {
    return getResource(CocoClient.getInstance().getNetworkMap(), networkId, deviceNodeId,
        resourceEui);
  }

  /**
   * A utility function to get resource specified by the corresponding identifiers.
   *
   * @param networkMap   The map which is to be used for search
   * @param networkId    A string that specifies the networkId
   * @param deviceNodeId The id of the device
   * @param resourceEui  The id of the resource
   * @return Network The Resource object corresponding to the networkId & deviceNodeId & resourceEui
   */
  static Resource getResource(Map<String, Network> networkMap, String networkId, long deviceNodeId,
                              String resourceEui) {
    Device parentDevice = getDevice(networkMap, networkId, deviceNodeId);
    return (null == parentDevice) ? null : parentDevice.getResource(resourceEui);
  }

  /**
   * A utility function to get capability specified by the corresponding identifiers.
   *
   * @param networkMap   The map which is to be used for search
   * @param networkId    A string that specifies the networkId
   * @param deviceNodeId The id of the device
   * @param resourceEui  The id of the resource
   * @param capabilityId The id of the capability
   * @return Network The Resource object corresponding to the networkId & deviceNodeId & resourceEui
   */
  static Capability getCapability(Map<String, Network> networkMap, String networkId,
                                  long deviceNodeId, String resourceEui, int capabilityId) {
    Resource parentResource = getResource(networkMap, networkId, deviceNodeId, resourceEui);
    return (null == parentResource) ? null :
        parentResource.getCapability(capIdConverter(capabilityId));
  }

  /**
   * A function to assist with capabilityId conversions.
   *
   * @param index The integer value of the capabilityID
   * @return CapabilityId: The capabilityId enum
   */
  private static Capability.CapabilityId capIdConverter(int index) {
    return Capability.CapabilityId.getEnum(index);
  }

  static <T extends Enum<T>> T findEnum(int index, T[] arr) {
    int size = arr.length;

    if (0 > index || index >= size) {
      throw new IllegalArgumentException("Requested index: " + index + ", Size of Enum: " + size);
    } else {
      return arr[index];
    }
  }
}
