/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Zone class is a blueprint of zone present in a network.
 * Zone is a container of resources added by user, this class also facilitates
 * movement of resources between different zones by implementing iterable.
 */
public class Zone implements Iterable<Resource> {
  public static String defaultZoneName = "default zone";
  public static final int DEFAULT_ZONE_ID = 0;

  private final int zoneId;
  private Set<Resource> resources;

  private String zoneName;
  private transient Network parent;
  private transient boolean ready = false;

  /**
   * The constructor of the current class.
   *
   * @param zoneId The unique id of the zone
   * @param parent The parent network of the zone
   */
  protected Zone(int zoneId, Network parent) {
    this.zoneId = zoneId;
    this.parent = parent;
  }

  /**
   * A function to get the zoneId of the zone.
   *
   * @return int: zoneId of the current zone
   */
  public int getId() {
    return this.zoneId;
  }

  /**
   * A function to get the name of the zone.
   *
   * @return String: The name of the zone
   */
  public String getName() {
    return this.zoneName;
  }

  /**
   * A function to get the set of resources of the current zone.
   *
   * @return The set of resources
   */
  public Set<Resource> getResources() {
    if (null == this.resources) {
      this.resources = Collections.synchronizedSet(new HashSet<>());
    }

    return this.resources;
  }

  /**
   * A function to get the parent Network of the zone.
   *
   * @return Network: The parent network of the current zone
   */
  public Network getParent() {
    return this.parent;
  }

  /**
   * A function to set the name of the current zone.
   *
   * @param name The name that has to be set
   */
  public void setName(String name, Network.NetworkManagementStatusListener listener) {
    parent.sendNetworkManagementCommand(new Network.SetZone(this, name), listener);
  }

  protected void internalAddResource(Resource resource) {
    getResources().add(resource);
  }

  protected void internalRemoveResource(Resource resource) {
    getResources().remove(resource);
  }

  protected boolean containsResource(Resource resource) {
    return getResources().contains(resource);
  }

  protected void internalSetName(String name) {
    this.zoneName = name;
  }

  protected void internalMarkAsReady() {
    this.ready = true;
  }

  protected void internalSetParent(Network parent) {
    this.parent = parent;
  }

  @Override
  public final int hashCode() {
    return getId();
  }

  /**
   * A function suggesting if the zone is made from {@link NativeCallbacks#zoneInfoCallback}
   * or because of state retrieval.
   *
   * @return boolean: states if the zone is dummy ot not.
   */
  public boolean isReady() {
    return ready;
  }

  @Override
  public Iterator<Resource> iterator() {
    return getResources().iterator();
  }

  @Override
  public String toString() {
    String networkId = (null == parent) ? "?" : parent.getId();

    return "Zone{"
        + "networkId=" + networkId
        + ", zoneId=" + zoneId
        + ", resources=" + resources
        + ", zoneName='" + zoneName + '\''
        + ", ready=" + ready
        + '}';
  }
}
