/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient;

import static com.abiquo.apiclient.domain.ApiPath.VIRTUALDATACENTERS_URL;
import static com.abiquo.apiclient.domain.Links.create;
import static com.abiquo.apiclient.domain.Links.editOrSelf;
import static com.abiquo.apiclient.domain.Links.isNic;
import static com.abiquo.apiclient.domain.Links.withRel;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.size;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.abiquo.apiclient.domain.ApiPath;
import com.abiquo.apiclient.domain.options.AllowedDatacenterListOptions;
import com.abiquo.apiclient.domain.options.AllowedPublicCloudRegionListOptions;
import com.abiquo.apiclient.domain.options.ExternalIpListOptions;
import com.abiquo.apiclient.domain.options.VirtualApplianceListOptions;
import com.abiquo.apiclient.domain.options.VirtualDatacenterListOptions;
import com.abiquo.apiclient.domain.options.VirtualMachineListOptions;
import com.abiquo.model.enumerator.NetworkType;
import com.abiquo.model.rest.RESTLink;
import com.abiquo.model.transport.AcceptedRequestDto;
import com.abiquo.model.transport.SingleResourceTransportDto;
import com.abiquo.server.core.appslibrary.VirtualMachineTemplateDto;
import com.abiquo.server.core.cloud.DeviceDto;
import com.abiquo.server.core.cloud.DeviceTypeDto;
import com.abiquo.server.core.cloud.DevicesDto;
import com.abiquo.server.core.cloud.FirewallPolicyDto;
import com.abiquo.server.core.cloud.FirewallRulesDto;
import com.abiquo.server.core.cloud.HardwareProfileDto;
import com.abiquo.server.core.cloud.HealthCheckDto;
import com.abiquo.server.core.cloud.HealthChecksDto;
import com.abiquo.server.core.cloud.LoadBalancerAddressDto;
import com.abiquo.server.core.cloud.LoadBalancerAddressesDto;
import com.abiquo.server.core.cloud.LoadBalancerDto;
import com.abiquo.server.core.cloud.LoadBalancersDto;
import com.abiquo.server.core.cloud.RoutingRuleDto;
import com.abiquo.server.core.cloud.RoutingRulesDto;
import com.abiquo.server.core.cloud.SSLCertificateDto;
import com.abiquo.server.core.cloud.VirtualApplianceDto;
import com.abiquo.server.core.cloud.VirtualApplianceState;
import com.abiquo.server.core.cloud.VirtualAppliancesDto;
import com.abiquo.server.core.cloud.VirtualDatacenterDto;
import com.abiquo.server.core.cloud.VirtualDatacentersDto;
import com.abiquo.server.core.cloud.VirtualMachineCloneOptionsDto;
import com.abiquo.server.core.cloud.VirtualMachineDto;
import com.abiquo.server.core.cloud.VirtualMachineState;
import com.abiquo.server.core.cloud.VirtualMachineStateDto;
import com.abiquo.server.core.cloud.VirtualMachineTaskDto;
import com.abiquo.server.core.cloud.VirtualMachinesDto;
import com.abiquo.server.core.enterprise.EnterpriseDto;
import com.abiquo.server.core.infrastructure.DatacenterDto;
import com.abiquo.server.core.infrastructure.DatacentersDto;
import com.abiquo.server.core.infrastructure.LocationDto;
import com.abiquo.server.core.infrastructure.PublicCloudRegionDto;
import com.abiquo.server.core.infrastructure.PublicCloudRegionsDto;
import com.abiquo.server.core.infrastructure.network.ExternalIpDto;
import com.abiquo.server.core.infrastructure.network.ExternalIpsDto;
import com.abiquo.server.core.infrastructure.network.NicDto;
import com.abiquo.server.core.infrastructure.network.NicsDto;
import com.abiquo.server.core.infrastructure.network.PublicIpDto;
import com.abiquo.server.core.infrastructure.network.VLANNetworkDto;
import com.abiquo.server.core.infrastructure.network.VMNetworkConfigurationDto;
import com.abiquo.server.core.infrastructure.network.VMNetworkConfigurationsDto;
import com.abiquo.server.core.infrastructure.storage.TierDto;
import com.abiquo.server.core.infrastructure.storage.TiersDto;
import com.abiquo.server.core.infrastructure.storage.VolumeManagementDto;
import com.abiquo.server.core.task.TaskDto;
import com.google.common.base.Optional;
import com.google.common.reflect.TypeToken;

public class CloudApi
{
    private final RestClient client;

    // Package private constructor to be used only by the ApiClient
    CloudApi(final RestClient client)
    {
        this.client = checkNotNull(client, "client cannot be null");
    }

    public VirtualDatacenterDto getVirtualDatacenter(final int id)
    {
        return client.get(VIRTUALDATACENTERS_URL + "/" + id, VirtualDatacenterDto.MEDIA_TYPE,
            VirtualDatacenterDto.class);
    }

    public VirtualDatacenterDto editVirtualDatacenter(final VirtualDatacenterDto vdc)
    {
        return client.edit(vdc);
    }

    public Iterable<DatacenterDto> listAllowedDatacenters()
    {
        return client.list(ApiPath.LOCATIONS_URL, DatacentersDto.MEDIA_TYPE, DatacentersDto.class);
    }

    public Iterable<DatacenterDto> listAllowedDatacenters(final AllowedDatacenterListOptions options)
    {
        return client.list(ApiPath.LOCATIONS_URL, options.queryParams(), DatacentersDto.MEDIA_TYPE,
            DatacentersDto.class);
    }

    public Iterable<PublicCloudRegionDto> listAllowedPublicCloudRegions()
    {
        return client.list(ApiPath.LOCATIONS_URL, PublicCloudRegionsDto.MEDIA_TYPE,
            PublicCloudRegionsDto.class);
    }

    public Iterable<PublicCloudRegionDto> listAllowedPublicCloudRegions(
        final AllowedPublicCloudRegionListOptions options)
    {
        return client.list(ApiPath.LOCATIONS_URL, options.queryParams(),
            PublicCloudRegionsDto.MEDIA_TYPE, PublicCloudRegionsDto.class);
    }

    public Iterable<VirtualDatacenterDto> listVirtualDatacenters()
    {
        return client.list(VIRTUALDATACENTERS_URL, VirtualDatacentersDto.MEDIA_TYPE,
            VirtualDatacentersDto.class);
    }

    public Iterable<VirtualDatacenterDto> listVirtualDatacenters(
        final VirtualDatacenterListOptions options)
    {
        return client.list(VIRTUALDATACENTERS_URL, options.queryParams(),
            VirtualDatacentersDto.MEDIA_TYPE, VirtualDatacentersDto.class);
    }

    public Iterable<ExternalIpDto> listExternalIps(final VirtualDatacenterDto vdc)
    {
        return client.list(vdc.searchLink("externalips").getHref(), ExternalIpsDto.MEDIA_TYPE,
            ExternalIpsDto.class);
    }

    public Iterable<ExternalIpDto> listExternalIps(final VirtualDatacenterDto vdc,
        final ExternalIpListOptions options)
    {
        return client.list(vdc.searchLink("externalips").getHref(), options.queryParams(),
            ExternalIpsDto.MEDIA_TYPE, ExternalIpsDto.class);
    }

    public Iterable<VirtualApplianceDto> listVirtualAppliances(final VirtualDatacenterDto vdc)
    {
        return client.list(vdc.searchLink("virtualappliances").getHref(),
            VirtualAppliancesDto.MEDIA_TYPE, VirtualAppliancesDto.class);
    }

    public Iterable<VirtualApplianceDto> listVirtualAppliances(final VirtualDatacenterDto vdc,
        final VirtualApplianceListOptions options)
    {
        return client.list(vdc.searchLink("virtualappliances").getHref(), options.queryParams(),
            VirtualAppliancesDto.MEDIA_TYPE, VirtualAppliancesDto.class);
    }

    public VirtualApplianceDto getVirtualAppliance(final int idVdc, final int idVapp)
    {
        return client.get(
            String.format("%s/%s/virtualappliances/%s", VIRTUALDATACENTERS_URL, idVdc, idVapp),
            VirtualApplianceDto.MEDIA_TYPE, VirtualApplianceDto.class);
    }

    public Iterable<VirtualMachineDto> listVirtualMachines(final VirtualApplianceDto vapp)
    {
        return client.list(vapp.searchLink("virtualmachines").getHref(),
            VirtualMachinesDto.MEDIA_TYPE, VirtualMachinesDto.class);
    }

    public Iterable<VirtualMachineDto> listVirtualMachines(final VirtualApplianceDto vapp,
        final VirtualMachineListOptions options)
    {
        return client.list(vapp.searchLink("virtualmachines").getHref(), options.queryParams(),
            VirtualMachinesDto.MEDIA_TYPE, VirtualMachinesDto.class);
    }

    public VLANNetworkDto getPrivateNetwork(final VirtualDatacenterDto vdc, final int idNetwork)
    {
        return client.get(vdc.searchLink("privatenetworks").getHref() + "/" + idNetwork,
            VLANNetworkDto.MEDIA_TYPE, VLANNetworkDto.class);
    }

    public Iterable<VMNetworkConfigurationDto> listNetworkConfigurations(final VirtualMachineDto vm)
    {
        return client.list(vm.searchLink("configurations").getHref(),
            VMNetworkConfigurationsDto.MEDIA_TYPE, VMNetworkConfigurationsDto.class);
    }

    public VirtualMachineDto getVirtualMachine(final VirtualApplianceDto vapp, final int idVm)
    {
        return client.get(vapp.searchLink("virtualmachines").getHref() + "/" + idVm,
            VirtualMachineDto.MEDIA_TYPE, VirtualMachineDto.class);
    }

    public VirtualDatacenterDto createVirtualDatacenter(final SingleResourceTransportDto location,
        final EnterpriseDto enterprise, final String name, final String type,
        final String vlanAddress, final String vlanGateway, final String vlanName)
    {
        VLANNetworkDto vlan = new VLANNetworkDto();
        vlan.setAddress(vlanAddress);
        vlan.setGateway(vlanGateway);
        vlan.setMask(24);
        vlan.setName(vlanName);
        vlan.setType(NetworkType.INTERNAL);

        return createVirtualDatacenter(location, enterprise, name, type, vlan);
    }

    public VirtualDatacenterDto createVirtualDatacenter(final SingleResourceTransportDto location,
        final EnterpriseDto enterprise, final String name, final String type,
        final VLANNetworkDto vlan)
    {
        checkArgument(location instanceof DatacenterDto || location instanceof PublicCloudRegionDto);
        String mt =
            location instanceof DatacenterDto ? DatacenterDto.SHORT_MEDIA_TYPE_JSON
                : PublicCloudRegionDto.SHORT_MEDIA_TYPE_JSON;

        VirtualDatacenterDto vdc = new VirtualDatacenterDto();
        vdc.setName(name);
        vdc.setHypervisorType(type);
        vdc.setVlan(vlan);

        vdc.addLink(create("enterprise", enterprise.getEditLink().getHref(),
            EnterpriseDto.SHORT_MEDIA_TYPE_JSON));
        vdc.addLink(create("location", location.searchLink("self").getHref(), mt));

        return client.post(VIRTUALDATACENTERS_URL, VirtualDatacenterDto.MEDIA_TYPE,
            VirtualDatacenterDto.MEDIA_TYPE, vdc, VirtualDatacenterDto.class);
    }

    public PublicIpDto createPublicIp(final PublicCloudRegionDto location)
    {
        return client.post(location.searchLink("ips").getHref(), PublicIpDto.MEDIA_TYPE,
            PublicIpDto.class);
    }

    public PublicIpDto addPublicIpToVirtualDatacenter(final PublicIpDto publicip,
        final VirtualDatacenterDto vdc)
    {
        return client.put(vdc.searchLink("purchased").getHref() + "/" + publicip.getId(),
            PublicIpDto.MEDIA_TYPE, PublicIpDto.class);
    }

    public PublicIpDto releasePublicIpFromVirtualDatacenter(final PublicIpDto publicip,
        final VirtualDatacenterDto vdc)
    {
        return client.put(vdc.searchLink("topurchase").getHref() + "/" + publicip.getId(),
            PublicIpDto.MEDIA_TYPE, PublicIpDto.class);
    }

    public VirtualMachineDto assignPublicIpToVirtualMachine(final PublicIpDto ip,
        final PublicCloudRegionDto location, final VirtualDatacenterDto vdc,
        final VirtualMachineDto vm)
    {
        int nics = size(filter(vm.getLinks(), isNic()));

        RESTLink nicLink =
            create(NicDto.REL_PREFIX + nics,
                vdc.searchLink("purchased").getHref() + "/" + ip.getId(),
                PublicIpDto.SHORT_MEDIA_TYPE_JSON);
        vm.addLink(nicLink);

        return editVirtualMachine(vm, 5, 300, TimeUnit.SECONDS);
    }

    public VirtualApplianceDto createVirtualAppliance(final VirtualDatacenterDto vdc,
        final String name)
    {
        VirtualApplianceDto vapp = new VirtualApplianceDto();
        vapp.setName(name);

        return client.post(vdc.searchLink("virtualappliances").getHref(),
            VirtualApplianceDto.MEDIA_TYPE, VirtualApplianceDto.MEDIA_TYPE, vapp,
            VirtualApplianceDto.class);
    }

    public VirtualMachineDto createVirtualMachine(final VirtualMachineTemplateDto template,
        final VirtualApplianceDto vapp)
    {
        VirtualMachineDto vm = new VirtualMachineDto();
        vm.setVdrpEnabled(Boolean.TRUE);
        vm.addLink(withRel("virtualmachinetemplate", template.getEditLink()));
        return client
            .post(vapp.searchLink("virtualmachines").getHref(), VirtualMachineDto.MEDIA_TYPE,
                VirtualMachineDto.MEDIA_TYPE, vm, VirtualMachineDto.class);
    }

    public VirtualMachineDto createVirtualMachine(final VirtualMachineTemplateDto template,
        final VirtualApplianceDto vapp, final HardwareProfileDto hp)
    {
        VirtualMachineDto vm = new VirtualMachineDto();
        vm.setVdrpEnabled(Boolean.TRUE);
        vm.addLink(withRel("virtualmachinetemplate", template.getEditLink()));
        vm.addLink(withRel("hardwareprofile", editOrSelf(hp)));
        return client
            .post(vapp.searchLink("virtualmachines").getHref(), VirtualMachineDto.MEDIA_TYPE,
                VirtualMachineDto.MEDIA_TYPE, vm, VirtualMachineDto.class);
    }

    public VirtualMachineDto deploy(final VirtualMachineDto vm, final int pollInterval,
        final int maxWait, final TimeUnit timeUnit)
    {
        return deploy(vm, false, pollInterval, maxWait, timeUnit);
    }

    public VirtualMachineDto deploy(final VirtualMachineDto vm, final boolean forceDeploy,
        final int pollInterval, final int maxWait, final TimeUnit timeUnit)
    {
        client.post(vm.searchLink("deploy").getHref() + "?force=" + forceDeploy,
            AcceptedRequestDto.MEDIA_TYPE, new TypeToken<AcceptedRequestDto<String>>()
            {
                private static final long serialVersionUID = -6348281615419377868L;
            });

        VirtualMachineDto refreshed = client.waitUntilUnlocked(vm, pollInterval, maxWait, timeUnit);
        if (!refreshed.getState().isDeployed())
        {
            throw new RuntimeException("Deploy virtual machine operation failed");
        }

        return refreshed;
    }

    public VirtualApplianceDto deploy(final VirtualApplianceDto vapp, final int pollInterval,
        final int maxWait, final TimeUnit timeUnit)
    {
        return deploy(vapp, false, pollInterval, maxWait, timeUnit);
    }

    public VirtualApplianceDto deploy(final VirtualApplianceDto vapp, final boolean forceDeploy,
        final int pollInterval, final int maxWait, final TimeUnit timeUnit)
    {
        client.post(vapp.searchLink("deploy").getHref() + "?force=" + forceDeploy,
            AcceptedRequestDto.MEDIA_TYPE, new TypeToken<AcceptedRequestDto<String>>()
            {
                private static final long serialVersionUID = -6348281615419377868L;
            });

        VirtualApplianceDto refreshed =
            client.waitUntilUnlocked(vapp, pollInterval, maxWait, timeUnit);
        if (VirtualApplianceState.DEPLOYED != refreshed.getState())
        {
            throw new RuntimeException("Deploy virtual appliance operation failed");
        }

        return refreshed;
    }

    public VirtualMachineDto undeploy(final VirtualMachineDto vm, final boolean forceUndeploy,
        final int pollInterval, final int maxWait, final TimeUnit timeUnit)
    {
        VirtualMachineTaskDto virtualMachineTask = new VirtualMachineTaskDto();
        virtualMachineTask.setForceUndeploy(forceUndeploy);

        client.post(vm.searchLink("undeploy").getHref(), AcceptedRequestDto.MEDIA_TYPE,
            VirtualMachineTaskDto.MEDIA_TYPE, virtualMachineTask,
            new TypeToken<AcceptedRequestDto<String>>()
            {
                private static final long serialVersionUID = -6348281615419377868L;
            });

        VirtualMachineDto refreshed = client.waitUntilUnlocked(vm, pollInterval, maxWait, timeUnit);
        if (refreshed.getState().isDeployed())
        {
            throw new RuntimeException("Undeploy virtual machine operation failed");
        }

        return refreshed;
    }

    public VirtualMachineDto undeploy(final VirtualMachineDto vm, final int pollInterval,
        final int maxWait, final TimeUnit timeUnit)
    {
        return undeploy(vm, false, pollInterval, maxWait, timeUnit);
    }

    public VirtualApplianceDto undeploy(final VirtualApplianceDto vapp,
        final boolean forceUndeploy, final int pollInterval, final int maxWait,
        final TimeUnit timeUnit)
    {
        VirtualMachineTaskDto virtualMachineTask = new VirtualMachineTaskDto();
        virtualMachineTask.setForceUndeploy(forceUndeploy);

        client.post(vapp.searchLink("undeploy").getHref(), AcceptedRequestDto.MEDIA_TYPE,
            VirtualMachineTaskDto.MEDIA_TYPE, virtualMachineTask,
            new TypeToken<AcceptedRequestDto<String>>()
            {
                private static final long serialVersionUID = -6348281615419377868L;
            });

        VirtualApplianceDto refreshed =
            client.waitUntilUnlocked(vapp, pollInterval, maxWait, timeUnit);
        if (VirtualApplianceState.NOT_DEPLOYED != refreshed.getState())
        {
            throw new RuntimeException("Undeploy virtual appliance operation failed");
        }

        return refreshed;
    }

    public VirtualApplianceDto undeploy(final VirtualApplianceDto vapp, final int pollInterval,
        final int maxWait, final TimeUnit timeUnit)
    {
        return undeploy(vapp, false, pollInterval, maxWait, timeUnit);
    }

    public VirtualMachineDto powerState(final VirtualMachineDto vm,
        final VirtualMachineState state, final int pollInterval, final int maxWait,
        final TimeUnit timeUnit)
    {
        VirtualMachineStateDto vmState = new VirtualMachineStateDto();
        vmState.setState(state);

        client.put(vm.searchLink("state").getHref(), AcceptedRequestDto.MEDIA_TYPE,
            VirtualMachineStateDto.MEDIA_TYPE, vmState, new TypeToken<AcceptedRequestDto<String>>()
            {
                private static final long serialVersionUID = -6348281615419377868L;
            });

        VirtualMachineDto refreshed = client.waitUntilUnlocked(vm, pollInterval, maxWait, timeUnit);
        if (state != refreshed.getState())
        {
            throw new RuntimeException("Virtual machine power state '" + state.name()
                + "' operation failed");
        }

        return refreshed;
    }

    public VirtualMachineDto editVirtualMachine(final VirtualMachineDto vm, final int pollInterval,
        final int maxWait, final TimeUnit timeUnit)
    {
        VirtualMachineDto refreshed = null;

        if (vm.getState().isDeployed())
        {
            client.put(vm.getEditLink().getHref(), AcceptedRequestDto.MEDIA_TYPE,
                VirtualMachineDto.MEDIA_TYPE, vm, new TypeToken<AcceptedRequestDto<String>>()
                {
                    private static final long serialVersionUID = -6348281615419377868L;
                });

            refreshed = client.waitUntilUnlocked(vm, pollInterval, maxWait, timeUnit);
            if (VirtualMachineState.OFF != refreshed.getState())
            {
                throw new RuntimeException("Virtual machine reconfigure operation failed");
            }
        }
        else
        {
            client.put(vm.getEditLink().getHref(), AcceptedRequestDto.MEDIA_TYPE,
                VirtualMachineDto.MEDIA_TYPE, vm);
            refreshed = client.refresh(vm);
        }

        return refreshed;
    }

    public VolumeManagementDto getVolume(final VirtualDatacenterDto vdc, final int idVolume)
    {
        return client.get(vdc.searchLink("volumes").getHref() + "/" + idVolume,
            VolumeManagementDto.MEDIA_TYPE, VolumeManagementDto.class);
    }

    public VolumeManagementDto createVolume(final VirtualDatacenterDto vdc, final String name,
        final long sizeInMb, final TierDto tier)
    {
        VolumeManagementDto dto = new VolumeManagementDto();
        dto.setName(name);
        dto.setSizeInMB(sizeInMb);
        dto.addLink(create("tier", tier.searchLink("self").getHref(), TierDto.SHORT_MEDIA_TYPE_JSON));

        return client.post(vdc.searchLink("volumes").getHref(), VolumeManagementDto.MEDIA_TYPE,
            VolumeManagementDto.MEDIA_TYPE, dto, VolumeManagementDto.class);
    }

    public TaskDto getTask(final VirtualMachineDto vm, final String idTask)
    {
        return client.get(vm.searchLink("tasks").getHref() + "/" + idTask, TaskDto.MEDIA_TYPE,
            TaskDto.class);
    }

    public Iterable<TierDto> listTiers(final VirtualDatacenterDto vdc)
    {
        return client.list(vdc.searchLink("tiers").getHref(), TiersDto.MEDIA_TYPE, TiersDto.class);
    }

    public Iterable<DeviceDto> listDevices(final LocationDto location)
    {
        for (RESTLink link : location.getLinks())
        {
            if (link.getRel().equals("devices") && link.getType().equals(DevicesDto.MEDIA_TYPE))
            {
                return client.list(link.getHref(), DevicesDto.MEDIA_TYPE, DevicesDto.class);
            }
        }
        return Collections.emptyList();
    }

    public Iterable<LoadBalancerDto> listLoadBalancers(final VirtualDatacenterDto vdc)
    {
        return client.list(vdc.searchLink("loadbalancers").getHref(), LoadBalancersDto.MEDIA_TYPE,
            LoadBalancersDto.class);
    }

    public Iterable<LoadBalancerDto> listLoadBalancers(final DeviceDto lbd)
    {
        return client.list(lbd.searchLink("loadbalancers").getHref(), LoadBalancersDto.MEDIA_TYPE,
            LoadBalancersDto.class);
    }

    public LoadBalancerDto getLoadBalacer(final DeviceDto device, final int idLoadBalancer)
    {
        return client.get(device.searchLink("loadbalancers").getHref() + "/" + idLoadBalancer,
            LoadBalancerDto.MEDIA_TYPE, LoadBalancerDto.class);
    }

    public void deleteLoadBalancer(final LoadBalancerDto lbd)
    {
        client.delete(lbd);
    }

    public LoadBalancerDto editLoadBalancer(final LoadBalancerDto lbd)
    {
        return client.edit(lbd);
    }

    public LoadBalancerDto createLoadBalancer(final DeviceDto device, final String name,
        final String algorithm, final List<RoutingRuleDto> routingRules,
        final List<HealthCheckDto> healthChecks, final List<LoadBalancerAddressDto> lbAddresses,
        final List<FirewallPolicyDto> firewalls, final Optional<VirtualDatacenterDto> vdc,
        final Optional<VLANNetworkDto> network)
    {
        LoadBalancerDto lbd = new LoadBalancerDto();
        lbd.setName(name);
        lbd.setAlgorithm(algorithm);

        lbd.setHealthChecks(new HealthChecksDto());
        lbd.setRoutingRules(new RoutingRulesDto());
        lbd.setLoadBalancerAddresses(new LoadBalancerAddressesDto());
        lbd.getHealthChecks().addAll(healthChecks);
        lbd.getRoutingRules().addAll(routingRules);
        lbd.getLoadBalancerAddresses().addAll(lbAddresses);

        if (vdc.isPresent())
        {
            lbd.addLink(withRel("virtualdatacenter", editOrSelf(vdc.get())));
        }
        if (network.isPresent())
        {
            checkArgument(network.get().getType() == NetworkType.INTERNAL,
                "Only internal networks can be assigned to a load balancer");
            lbd.addLink(withRel("privatenetwork", editOrSelf(network.get())));
        }

        for (FirewallPolicyDto fw : firewalls)
        {
            lbd.addLink(withRel("firewall", editOrSelf(fw)));
        }

        return client.post(device.searchLink("loadbalancers").getHref(),
            LoadBalancerDto.MEDIA_TYPE, LoadBalancerDto.MEDIA_TYPE, lbd, LoadBalancerDto.class);
    }

    public HealthCheckDto createHealthCheck(final LoadBalancerDto lbd, final String name,
        final String protocol, final long intervalInMs, final long timeoutInMs)
    {
        return createHealthCheck(lbd, name, protocol, intervalInMs, timeoutInMs, null, null, null);
    }

    public HealthCheckDto createHealthCheck(final LoadBalancerDto lbd, final String name,
        final String protocol, final long intervalInMs, final long timeoutInMs,
        final Integer attemps, final Integer port, final String path)
    {

        HealthCheckDto healthCheck = new HealthCheckDto();
        healthCheck.setName(name);
        healthCheck.setProtocol(protocol);
        healthCheck.setIntervalInMs(intervalInMs);
        healthCheck.setTimeoutInMs(timeoutInMs);
        healthCheck.setAttempts(attemps);
        healthCheck.setPort(port);
        healthCheck.setPath(path);
        return client.post(lbd.searchLink("healthchecks").getHref(), HealthCheckDto.MEDIA_TYPE,
            HealthCheckDto.MEDIA_TYPE, healthCheck, HealthCheckDto.class);
    }

    public RoutingRuleDto createRoutingRule(final LoadBalancerDto lbd, final String protocolIn,
        final String protocolOut, final int portIn, final int portOut,
        final SSLCertificateDto sslCertificate)
    {
        RoutingRuleDto routingRule = new RoutingRuleDto();
        routingRule.setPortIn(portIn);
        routingRule.setPortOut(portOut);
        routingRule.setProtocolIn(protocolIn);
        routingRule.setProtocolOut(protocolOut);
        if (sslCertificate != null)
        {
            routingRule.setSslCertificate(sslCertificate);
        }
        return client.post(lbd.searchLink("routingrules").getHref(), RoutingRuleDto.MEDIA_TYPE,
            RoutingRuleDto.MEDIA_TYPE, routingRule, RoutingRuleDto.class);
    }

    public Iterable<RoutingRuleDto> listRoutingRules(final LoadBalancerDto lbd)
    {
        return client.list(lbd.searchLink("routingrules").getHref(), RoutingRulesDto.MEDIA_TYPE,
            RoutingRulesDto.class);
    }

    public Iterable<HealthCheckDto> listHealthChecks(final LoadBalancerDto lbd)
    {
        return client.list(lbd.searchLink("healthchecks").getHref(), HealthChecksDto.MEDIA_TYPE,
            HealthChecksDto.class);
    }

    public RoutingRulesDto editRoutingRules(final RoutingRulesDto routingRules)
    {
        for (RoutingRuleDto routingRule : routingRules.getCollection())
        {
            editRoutingRule(routingRule);
        }
        return routingRules;
    }

    public HealthChecksDto editHealthChecks(final HealthChecksDto healthChecks)
    {
        for (HealthCheckDto healthCheck : healthChecks.getCollection())
        {
            editHealthCheck(healthCheck);
        }
        return healthChecks;
    }

    public void deleteRoutingRule(final RoutingRuleDto routingRule)
    {
        client.delete(routingRule);
    }

    public void deleteHealthCheck(final HealthCheckDto healthCheck)
    {
        client.delete(healthCheck);
    }

    public RoutingRuleDto editRoutingRule(final RoutingRuleDto routingRule)
    {
        return client.edit(routingRule);
    }

    public HealthCheckDto editHealthCheck(final HealthCheckDto healthCheck)
    {

        return client.edit(healthCheck);
    }

    public RoutingRuleDto getRoutingRule(final LoadBalancerDto lbd, final int idRoutingRule)
    {
        return client.get(lbd.searchLink("routingrules").getHref() + "/" + idRoutingRule,
            RoutingRuleDto.MEDIA_TYPE, RoutingRuleDto.class);
    }

    public HealthCheckDto getHealthCheck(final LoadBalancerDto lbd, final int idHealthCheck)
    {
        return client.get(lbd.searchLink("healthchecks").getHref() + "/" + idHealthCheck,
            HealthCheckDto.MEDIA_TYPE, HealthCheckDto.class);
    }

    /**
     * @param location {@link PublicCloudRegionDto} or {@link DatacenterDto}
     */

    public DeviceDto createDevice(final LocationDto location, final DeviceTypeDto deviceType,
        final String name, final boolean vdcDefault)
    {
        return createDevice(location, deviceType, name, null, null, null, null, vdcDefault);
    }

    public DeviceDto createDevice(final LocationDto location, final DeviceTypeDto deviceType,
        final String name, final String endpoint, final String user, final String password,
        final VirtualDatacenterDto virtualDatacenter, final boolean vdcDefault)
    {
        DeviceDto device = new DeviceDto();
        device.addLink(withRel("devicetype", editOrSelf(deviceType)));
        device.addLink(withRel("virtualdatacenter", editOrSelf(virtualDatacenter)));
        device.setName(name);
        device.setEndpoint(endpoint);
        device.setUser(user);
        device.setPassword(password);
        device.setVdcDefault(vdcDefault);

        return client.post(location.searchLink("devices", "devices").getHref(),//
            DeviceDto.MEDIA_TYPE, DeviceDto.MEDIA_TYPE, device, DeviceDto.class);
    }

    public FirewallPolicyDto createFirewallPolicy(final String name, final String description,
        final DeviceDto firewallDevice, final Optional<VirtualDatacenterDto> optVdc)
    {
        FirewallPolicyDto firewall = new FirewallPolicyDto();
        firewall.setName(name);
        firewall.setDescription(description);
        if (optVdc.isPresent())
        {
            firewall.addLink(withRel("virtualdatacenter", editOrSelf(optVdc.get())));
        }

        return client.post(firewallDevice.searchLink("firewalls").getHref(),
            FirewallPolicyDto.MEDIA_TYPE, FirewallPolicyDto.MEDIA_TYPE, firewall,
            FirewallPolicyDto.class);
    }

    public FirewallRulesDto addFirewallRules(final FirewallPolicyDto firewall,
        final FirewallRulesDto rules)
    {
        return client.put(firewall.searchLink("rules").getHref(), FirewallRulesDto.MEDIA_TYPE,
            FirewallRulesDto.MEDIA_TYPE, rules, FirewallRulesDto.class);
    }

    public void addFirewallToVirtualMachine(final FirewallPolicyDto firewall,
        final VirtualMachineDto vm)
    {
        vm.addLink(withRel("firewall", editOrSelf(firewall)));

        client.put(vm.getEditLink().getHref(), AcceptedRequestDto.MEDIA_TYPE,
            VirtualMachineDto.MEDIA_TYPE, vm);
    }

    public NicsDto getNics(final VirtualMachineDto vm)
    {
        return client.get(vm.searchLink("nics").getHref(), NicsDto.MEDIA_TYPE, NicsDto.class);
    }

    public VirtualMachineDto cloneVirtualMachine(final VirtualMachineDto vm)
    {
        return client.post(vm.searchLink("clone").getHref(), VirtualMachineDto.MEDIA_TYPE,
            VirtualMachineCloneOptionsDto.MEDIA_TYPE, vm, VirtualMachineDto.class);
    }

    // /curl -uadmin:xabiquo -X POST
    // http://localhost/api/cloud/virtualdatacenters/1/virtualappliances/1/virtualmachines/1/action/clone
    // -H"Content-type:application/vnd.abiquo.virtualmachinecloneoptions+json"
    // -H"Accept:application/vnd.abiquo.virtualmachine+json" -v
}
