/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient;

import static com.abiquo.apiclient.domain.ApiPath.ENTERPRISES_URL;
import static com.abiquo.apiclient.domain.ApiPath.LOGIN_URL;
import static com.abiquo.apiclient.domain.ApiPath.PRIVILEGES_URL;
import static com.abiquo.apiclient.domain.ApiPath.ROLES_URL;
import static com.abiquo.apiclient.domain.ApiPath.SCOPES_URL;
import static com.abiquo.apiclient.domain.LinkRel.DATACENTERREPOSITORIES;
import static com.abiquo.apiclient.domain.Links.create;
import static com.abiquo.apiclient.domain.Links.editOrSelf;
import static com.abiquo.apiclient.domain.Links.withRel;
import static com.google.common.base.Preconditions.checkNotNull;

import java.util.List;
import java.util.Optional;

import com.abiquo.apiclient.domain.LinkRel;
import com.abiquo.apiclient.domain.Links;
import com.abiquo.apiclient.domain.options.EnterpriseListOptions;
import com.abiquo.apiclient.domain.options.UserListOptions;
import com.abiquo.model.enumerator.AuthType;
import com.abiquo.model.rest.RESTLink;
import com.abiquo.server.core.appslibrary.DatacenterRepositoriesDto;
import com.abiquo.server.core.appslibrary.DatacenterRepositoryDto;
import com.abiquo.server.core.cloud.HypervisorTypeDto;
import com.abiquo.server.core.cloud.VirtualDatacenterDto;
import com.abiquo.server.core.cloud.VirtualDatacentersDto;
import com.abiquo.server.core.enterprise.EnterpriseDto;
import com.abiquo.server.core.enterprise.EnterprisePropertiesDto;
import com.abiquo.server.core.enterprise.EnterprisePropertiesDto.Property;
import com.abiquo.server.core.enterprise.EnterprisesDto;
import com.abiquo.server.core.enterprise.PrivilegeDto;
import com.abiquo.server.core.enterprise.PrivilegesDto;
import com.abiquo.server.core.enterprise.RoleDto;
import com.abiquo.server.core.enterprise.RolesDto;
import com.abiquo.server.core.enterprise.ScopeDto;
import com.abiquo.server.core.enterprise.ScopesDto;
import com.abiquo.server.core.enterprise.UserDto;
import com.abiquo.server.core.enterprise.UsersDto;
import com.abiquo.server.core.infrastructure.PublicCloudCredentialsDto;
import com.abiquo.server.core.infrastructure.PublicCloudCredentialsListDto;
import com.google.common.base.Joiner;

public class EnterpriseApi
{
    private final RestClient client;

    // Package private constructor to be used only by the ApiClient
    EnterpriseApi(final RestClient client)
    {
        this.client = checkNotNull(client, "client cannot be null");
    }

    public EnterpriseDto createEnterprise(final String name)
    {
        EnterpriseDto enterprise = new EnterpriseDto();
        enterprise.setName(name);
        return client.post(ENTERPRISES_URL, EnterpriseDto.MEDIA_TYPE, EnterpriseDto.MEDIA_TYPE,
            enterprise, EnterpriseDto.class);
    }

    public EnterpriseDto editEnterprise(final EnterpriseDto enterprise)
    {
        return client.put(enterprise.getEditLink().getHref(), EnterpriseDto.MEDIA_TYPE,
            EnterpriseDto.MEDIA_TYPE, enterprise, EnterpriseDto.class);
    }

    public PublicCloudCredentialsDto addPublicCloudCredentials(final EnterpriseDto enterprise,
        final PublicCloudCredentialsDto credentials)
    {

        return client.post(String.format("%s/%s/credentials/", ENTERPRISES_URL, enterprise.getId()),
            PublicCloudCredentialsDto.MEDIA_TYPE, PublicCloudCredentialsDto.MEDIA_TYPE, credentials,
            PublicCloudCredentialsDto.class);
    }

    public PublicCloudCredentialsDto addPublicCloudCredentials(final ApiClient api,
        final EnterpriseDto enterprise, final String type, final String access, final String key)
    {
        HypervisorTypeDto hypervisorType = api.getConfigApi().getHypervisorType(type);

        PublicCloudCredentialsDto publicCloudCredentials = new PublicCloudCredentialsDto();
        publicCloudCredentials.setAccess(access);
        publicCloudCredentials.setKey(key);
        publicCloudCredentials.addLink(withRel("hypervisortype", editOrSelf(hypervisorType)));

        return api.getEnterpriseApi().addPublicCloudCredentials(enterprise, publicCloudCredentials);
    }

    public PublicCloudCredentialsListDto listPublicCloudCredentials(final EnterpriseDto enterprise)
    {
        return client.get(enterprise.searchLink(LinkRel.CREDENTIALS),
            PublicCloudCredentialsListDto.class);

    }

    public EnterpriseDto getEnterprise(final String id)
    {
        return client.get(ENTERPRISES_URL + "/" + id, EnterpriseDto.MEDIA_TYPE,
            EnterpriseDto.class);
    }

    public Iterable<EnterpriseDto> listEnterprises()
    {
        return client.list(ENTERPRISES_URL, EnterprisesDto.MEDIA_TYPE, EnterprisesDto.class);
    }

    public Iterable<EnterpriseDto> listEnterprises(final EnterpriseListOptions options)
    {
        return client.list(ENTERPRISES_URL, options.queryParams(), EnterprisesDto.MEDIA_TYPE,
            EnterprisesDto.class);
    }

    public ScopeDto createScope(final String name)
    {
        ScopeDto scope = new ScopeDto();
        scope.setName(name);
        scope.setAutomaticAddDatacenter(true);
        scope.setAutomaticAddEnterprise(true);
        return client.post(SCOPES_URL, ScopeDto.MEDIA_TYPE, ScopeDto.MEDIA_TYPE, scope,
            ScopeDto.class);
    }

    public void deleteUser(final UserDto user)
    {
        client.delete(user);
    }

    public void deleteScope(final ScopeDto scope)
    {
        client.delete(scope);
    }

    public UserDto editUser(final UserDto user)
    {
        return client.edit(user);
    }

    public UserDto createUser(final String name, final String surname, final String nick,
        final String password, final String email, final String description, final boolean active,
        final String locale, final AuthType authType, final String publicSshKey,
        final List<Integer> availableVirtualDatacentersIds, final EnterpriseDto enterprise,
        final RoleDto role, final Optional<ScopeDto> scope)
    {
        UserDto user = new UserDto();
        user.setName(name);
        user.setSurname(surname);
        user.setNick(nick);
        user.setPassword(password);
        user.setEmail(email);
        user.setDescription(description);
        user.setActive(active);
        user.setLocale(locale);
        user.setAuthType(authType.toString());
        user.setPublicSshKey(publicSshKey);
        user.setAvailableVirtualDatacenters(
            Joiner.on(",").skipNulls().join(availableVirtualDatacentersIds));
        user.addLink(create("role", role.getEditLink().getHref(), role.getEditLink().getType()));
        if (scope.isPresent())
        {
            user.addLink(Links.withRel("scope", scope.get().getEditLink()));
        }
        return client.post(enterprise.searchLink("users").getHref(), UserDto.MEDIA_TYPE,
            UserDto.MEDIA_TYPE, user, UserDto.class);
    }

    public UserDto getCurrentUser()
    {
        return client.get(LOGIN_URL, UserDto.MEDIA_TYPE, UserDto.class);
    }

    public Iterable<UserDto> listUsers(final EnterpriseDto enterprise)
    {
        return client.list(enterprise.searchLink("users").getHref(),
            enterprise.searchLink("users").getType(), UsersDto.class);
    }

    public Iterable<UserDto> listUsers(final EnterpriseDto enterprise,
        final UserListOptions options)
    {
        return client.list(enterprise.searchLink("users").getHref(), options.queryParams(),
            enterprise.searchLink("users").getType(), UsersDto.class);
    }

    public Iterable<RoleDto> listRoles()
    {
        return client.list(ROLES_URL, RolesDto.MEDIA_TYPE, RolesDto.class);
    }

    public Iterable<PrivilegeDto> listPrivileges(final RoleDto role)
    {
        return client.list(role.searchLink(LinkRel.PRIVILEGES), PrivilegesDto.class);
    }

    public Iterable<ScopeDto> listScopes()
    {
        return client.list(SCOPES_URL, ScopesDto.MEDIA_TYPE, ScopesDto.class);
    }

    public Iterable<DatacenterRepositoryDto> listDatacenterRepositories(
        final EnterpriseDto enterprise)
    {
        return client.list(enterprise.searchLink(DATACENTERREPOSITORIES),
            DatacenterRepositoriesDto.class);
    }

    public EnterprisePropertiesDto getEnterpriseProperties(final EnterpriseDto enterprise)
    {
        return client.get(enterprise.searchLink("properties").getHref(),
            EnterprisePropertiesDto.MEDIA_TYPE, EnterprisePropertiesDto.class);
    }

    public String getEnterpriseProperty(final EnterpriseDto enterprise, final String key)
    {
        EnterprisePropertiesDto entProp = client.get(enterprise.searchLink("properties").getHref(),
            EnterprisePropertiesDto.MEDIA_TYPE_JSON, EnterprisePropertiesDto.class);

        return entProp.getProperties().stream().filter(p -> key.equals(p.getPropertyKey()))
            .findAny().map(Property::getPropertyValue).orElse(null);
    }

    public EnterprisePropertiesDto addEnterpriseProperties(final EnterpriseDto enterprise,
        final EnterprisePropertiesDto properties)
    {
        EnterprisePropertiesDto prop = getEnterpriseProperties(enterprise);
        prop.getProperties().addAll(properties.getProperties());
        return client.put(prop.searchLink("edit").getHref(), EnterprisePropertiesDto.MEDIA_TYPE,
            EnterprisePropertiesDto.MEDIA_TYPE, prop, EnterprisePropertiesDto.class);
    }

    public EnterprisePropertiesDto deleteEnterpriseProperties(final EnterpriseDto enterprise,
        final List<String> keys)
    {
        EnterprisePropertiesDto prop = getEnterpriseProperties(enterprise);
        for (String key : keys)
        {
            prop.getProperties().removeIf(p -> key.equals(p.getPropertyKey()));
        }

        return client.put(prop.searchLink("edit").getHref(),
            EnterprisePropertiesDto.MEDIA_TYPE_JSON, EnterprisePropertiesDto.MEDIA_TYPE_JSON, prop,
            EnterprisePropertiesDto.class);
    }

    public Iterable<PrivilegeDto> listAllPrivileges()
    {
        return client.list(PRIVILEGES_URL, PrivilegesDto.MEDIA_TYPE, PrivilegesDto.class);
    }

    public Iterable<RoleDto> listRolesFromEnterprise(final EnterpriseDto ent)
    {
        Integer entId = ent.getId();
        return client.list(ROLES_URL + "?identerprise=" + entId, RolesDto.MEDIA_TYPE,
            RolesDto.class);
    }

    public Iterable<VirtualDatacenterDto> listVirtualDatacenters(final EnterpriseDto enterprise)
    {
        RESTLink vdcsLink = enterprise.searchLink(LinkRel.CLOUD_VIRTUALDATACENTERS);
        return client.list(vdcsLink.getHref(), VirtualDatacentersDto.MEDIA_TYPE,
            VirtualDatacentersDto.class);
    }

}
