/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient;

import static com.abiquo.apiclient.domain.ApiPath.COSTCODES_URL;
import static com.abiquo.apiclient.domain.ApiPath.CURRENCIES_URL;
import static com.abiquo.apiclient.domain.ApiPath.DEVICETYPES_URL;
import static com.abiquo.apiclient.domain.ApiPath.HYPERVISORTYPES_URL;
import static com.abiquo.apiclient.domain.ApiPath.PRICINGTEMPLATES_URL;
import static com.abiquo.apiclient.domain.ApiPath.PRIVILEGES_URL;
import static com.google.common.base.Preconditions.checkNotNull;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.StreamSupport;

import com.abiquo.apiclient.domain.LinkRel;
import com.abiquo.apiclient.domain.options.ListOptions;
import com.abiquo.server.core.cloud.DeviceTypeDto;
import com.abiquo.server.core.cloud.DeviceTypesDto;
import com.abiquo.server.core.cloud.HypervisorTypeDto;
import com.abiquo.server.core.cloud.HypervisorTypesDto;
import com.abiquo.server.core.cloud.RegionDto;
import com.abiquo.server.core.cloud.RegionsDto;
import com.abiquo.server.core.enterprise.PrivilegeDto;
import com.abiquo.server.core.enterprise.PrivilegesDto;
import com.abiquo.server.core.infrastructure.LocationDto;
import com.abiquo.server.core.pricing.CostCodeDto;
import com.abiquo.server.core.pricing.CostCodesDto;
import com.abiquo.server.core.pricing.CurrenciesDto;
import com.abiquo.server.core.pricing.CurrencyDto;
import com.abiquo.server.core.pricing.PricingTemplateDto;
import com.abiquo.server.core.pricing.PricingTemplatesDto;

public class ConfigApi
{
    private final RestClient client;

    // Package private constructor to be used only by the ApiClient
    ConfigApi(final RestClient client)
    {
        this.client = checkNotNull(client, "client cannot be null");
    }

    public HypervisorTypeDto getHypervisorType(final String type)
    {
        return client.get(String.format("%s/%s", HYPERVISORTYPES_URL, type),
            HypervisorTypeDto.MEDIA_TYPE, HypervisorTypeDto.class);
    }

    public HypervisorTypeDto getHypervisorType(final LocationDto location)
    {
        return client.get(location.searchLink(LinkRel.HYPERVISORTYPE).getHref(),
            HypervisorTypeDto.MEDIA_TYPE, HypervisorTypeDto.class);
    }

    public Iterable<HypervisorTypeDto> getHypervisorTypes()
    {
        return client.list(HYPERVISORTYPES_URL, HypervisorTypesDto.MEDIA_TYPE,
            HypervisorTypesDto.class);
    }

    public DeviceTypeDto getDeviceType(final String type)
    {
        return client.get(String.format("%s/%s", DEVICETYPES_URL, type), DeviceTypeDto.MEDIA_TYPE,
            DeviceTypeDto.class);
    }

    public Iterable<DeviceTypeDto> getDeviceTypes()
    {
        return client.list(DEVICETYPES_URL, DeviceTypesDto.MEDIA_TYPE, DeviceTypesDto.class);
    }

    public Iterable<RegionDto> getRegions(final String hypervisorType)
    {
        return client.list(getHypervisorType(hypervisorType).searchLink("regions").getHref(),
            RegionsDto.MEDIA_TYPE, RegionsDto.class);
    }

    public Optional<RegionDto> getRegion(final String hypervisorType, final String region)
    {
        return StreamSupport.stream(getRegions(hypervisorType).spliterator(), false)
            .filter(r -> r.getProviderId().equals(region)).findFirst();
    }

    public Iterable<RegionDto> getDynamicRegions(final String hypervisorType, final String identity,
        final String credential, final String endpoint)
    {
        Map<String, String> headers = new HashMap<>();
        headers.put("X-Abiquo-PCR-Identity", identity);
        headers.put("X-Abiquo-PCR-Credential", credential);
        headers.put("X-Abiquo-PCR-Endpoint", endpoint);

        return client.list(getHypervisorType(hypervisorType).searchLink("regions").getHref(),
            RegionsDto.MEDIA_TYPE, RegionsDto.class, headers);
    }

    public Optional<RegionDto> getDynamicRegion(final String hypervisorType, final String region,
        final String identity, final String credential, final String endpoint)
    {

        return StreamSupport
            .stream(getDynamicRegions(hypervisorType, identity, credential, endpoint).spliterator(),
                false)
            .filter(r -> r.getName().equals(region)).findFirst();
    }

    public CostCodeDto createCostcode(final CostCodeDto costCode)
    {
        return client.post(COSTCODES_URL, CostCodeDto.MEDIA_TYPE_JSON, CostCodeDto.MEDIA_TYPE_JSON,
            costCode, CostCodeDto.class);
    }

    public Iterable<CostCodeDto> listCostcodes(final ListOptions options)
    {
        return client.list(COSTCODES_URL, options.queryParams(), CostCodesDto.MEDIA_TYPE,
            CostCodesDto.class);
    }

    public Iterable<PrivilegeDto> listPrivileges()
    {
        return client.list(PRIVILEGES_URL, PrivilegesDto.MEDIA_TYPE, PrivilegesDto.class);
    }

    public PricingTemplateDto createPricingTemplate(final PricingTemplateDto pricingTemplate)
    {
        return client.post(PRICINGTEMPLATES_URL, PricingTemplateDto.MEDIA_TYPE,
            PricingTemplateDto.MEDIA_TYPE, pricingTemplate, PricingTemplateDto.class);
    }

    public Iterable<PricingTemplateDto> listPricingTemplates()
    {
        return client.list(PRICINGTEMPLATES_URL, PricingTemplatesDto.MEDIA_TYPE,
            PricingTemplatesDto.class);
    }

    public Iterable<CurrencyDto> listCurrencies()
    {
        return client.list(CURRENCIES_URL, CurrenciesDto.MEDIA_TYPE, CurrenciesDto.class);
    }

}
