/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static java.net.HttpURLConnection.HTTP_BAD_REQUEST;

import java.time.Period;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.abiquo.apiclient.domain.LinkRel;
import com.abiquo.apiclient.domain.exception.HttpException;
import com.abiquo.model.rest.RESTLink;
import com.abiquo.model.transport.SingleResourceTransportDto;
import com.abiquo.server.core.cloud.metric.MetricDto;
import com.abiquo.server.core.cloud.metric.MetricMetadataDto;
import com.abiquo.server.core.cloud.metric.MetricsMetadataDto;

public class MetricsApi
{
    private static final String StartRelativeQueryParam = "startrelative";

    private static final String StartAbsoluteQueryParam = "startabsolute";

    private static final String EndAbsoluteQueryParam = "endabsolute";

    private static final String GranularityQueryParam = "granularity";

    private static final String StatisticQueryParam = "statistic";

    public static enum StatisticType
    {
        Average("average"), Maximum("maximum"), Minimum("minimum"), Sum("sum"), Count("count"), //
        Dev("dev");

        private final String queryParameterValue;

        StatisticType(final String queryParameterValue)
        {
            this.queryParameterValue = queryParameterValue;
        }

        public String getParameter()
        {
            return queryParameterValue;
        }
    };

    public static class QueryMetric
    {
        private final Optional<Long> granularitySeconds;

        private final Optional<StatisticType> statisticType;

        private final Optional<String> startRelative;

        private final Optional<Long> startAbsolute;

        private final Optional<Long> endAbsolute;

        private final Map<String, String> dimensions;

        QueryMetric(final StatisticType statisticType, final Map<String, String> dimensions,
            final Period startRelative, final Date startAbsolute, final Date endAbsolute,
            final Long granularity)
        {
            this.statisticType = Optional.ofNullable(statisticType);
            this.dimensions = dimensions;
            this.granularitySeconds = Optional.ofNullable(granularity);
            this.startRelative = Optional.ofNullable(startRelative).map(Period::toString);
            this.startAbsolute = Optional.ofNullable(startAbsolute).map(Date::getTime);
            this.endAbsolute = Optional.ofNullable(endAbsolute).map(Date::getTime);
        }

        public Optional<Long> getGranularitySeconds()
        {
            return granularitySeconds;
        }

        public Map<String, String> getDimensions()
        {
            return dimensions;
        }

        public Optional<StatisticType> getStatisticType()
        {
            return statisticType;
        }

        public Optional<String> getStartRelative()
        {
            return startRelative;
        }

        public Optional<Long> getStartAbsolute()
        {
            return startAbsolute;
        }

        public Optional<Long> getEndAbsolute()
        {
            return endAbsolute;
        }
    }

    public static class QueryMetricBuilder
    {
        private Long granularityAmount = null;

        private TimeUnit granularityUnit = null;

        private StatisticType statisticType = null;

        private Period startRelative = null;

        private Date startAbsolute = null;

        private Date endAbsolute = null;

        private Map<String, String> dimensions = new HashMap<>();

        public QueryMetric build()
        {
            Long granularityInSeconds = null;
            if (granularityAmount != null && granularityUnit != null)
            {
                granularityInSeconds = granularityUnit.toSeconds(granularityAmount);
            }

            dimensions.remove(StartRelativeQueryParam);
            dimensions.remove(StartAbsoluteQueryParam);
            dimensions.remove(EndAbsoluteQueryParam);
            dimensions.remove(GranularityQueryParam);
            dimensions.remove(StatisticQueryParam);

            return new QueryMetric(statisticType, dimensions, startRelative, startAbsolute,
                endAbsolute, granularityInSeconds);
        }

        public QueryMetricBuilder withStatisticType(final StatisticType statisticType)
        {
            this.statisticType = statisticType;
            return this;
        }

        public QueryMetricBuilder withGranularity(final long amount, final TimeUnit unit)
        {
            if (unit.toSeconds(amount) < 60)
            {
                throw new IllegalArgumentException(
                    "Aggregator granularity should be >= 60 seconds");
            }

            this.granularityAmount = amount;
            this.granularityUnit = unit;
            return this;
        }

        public QueryMetricBuilder withDimensions(final Map<String, String> dimensions)
        {
            this.dimensions.putAll(dimensions);
            return this;
        }

        public QueryMetricBuilder withRelativeStartTime(final Period period)
        {
            this.startRelative = period;
            this.startAbsolute = null;
            this.endAbsolute = null;
            return this;
        }

        public QueryMetricBuilder withAbsoluteStartTime(final Date date)
        {
            this.startAbsolute = date;
            this.endAbsolute = null;
            this.startRelative = null;
            return this;
        }

        public QueryMetricBuilder withAbsoluteTimeRange(final Date start, final Date end)
        {
            this.startAbsolute = start;
            this.endAbsolute = end;
            this.startRelative = null;
            return this;
        }
    }

    private final RestClient client;

    // Package private constructor to be used only by the ApiClient
    MetricsApi(final RestClient client)
    {
        this.client = checkNotNull(client, "client cannot be null");
    }

    public List<MetricMetadataDto> listMetricMetadata(final SingleResourceTransportDto dto)
    {
        final RESTLink link = dto.searchLink(LinkRel.METRICSMETADATA);
        if (link == null)
        {
            return Collections.emptyList();
        }

        return client.get(link, MetricsMetadataDto.class).getCollection();
    }

    public Optional<MetricMetadataDto> findMetricMetadata(final SingleResourceTransportDto dto,
        final String metricName)
    {
        checkArgument(!isNullOrEmpty(metricName), "metric name cannot be null or empty");

        final RESTLink metricsMetadataLink = dto.searchLink(LinkRel.METRICSMETADATA);
        if (metricsMetadataLink == null)
        {
            return Optional.empty();
        }

        try
        {
            final RESTLink metadataLink = new RESTLink();
            metadataLink.setType(MetricMetadataDto.MEDIA_TYPE);
            metadataLink.setHref(AppendSegmentToPath(metricsMetadataLink.getHref(), metricName));
            return Optional.ofNullable(client.get(metadataLink, MetricMetadataDto.class));
        }
        catch (HttpException httpException)
        {
            if (HTTP_BAD_REQUEST == httpException.getCode())
            {
                return Optional.empty();
            }

            throw httpException;
        }
    }

    public MetricDto getMetricStatistic(final MetricMetadataDto metricMetadataDto)
    {
        return getMetricStatistic(metricMetadataDto, new QueryMetricBuilder().build());
    }

    public MetricDto getMetricStatistic(final MetricMetadataDto metricMetadataDto,
        final QueryMetric query)
    {
        final Map<String, Object> params = new HashMap<>(query.getDimensions());

        query.getStatisticType().ifPresent( //
            statistic -> params.put(StatisticQueryParam, statistic.getParameter()));

        query.getGranularitySeconds().ifPresent( //
            granularity -> params.put(GranularityQueryParam, String.valueOf(granularity)));

        query.getStartRelative().ifPresent( //
            duration -> params.put(StartRelativeQueryParam, duration));

        query.getStartAbsolute().ifPresent(start -> params.put(StartAbsoluteQueryParam, start));

        query.getEndAbsolute().ifPresent(end -> params.put(EndAbsoluteQueryParam, end));

        final RESTLink link = metricMetadataDto.searchLink(LinkRel.METRIC);
        return client.get(link.getHref(), params, link.getType(), MetricDto.class);
    }

    private static String AppendSegmentToPath(final String path, final String segment)
    {
        checkArgument(!isNullOrEmpty(path), "path cannot be null or empty");

        if (path.charAt(path.length() - 1) == '/')
        {
            return path + segment;
        }

        return path + "/" + segment;
    }
}
