/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient.interceptors;

import static com.abiquo.apiclient.domain.ApiPath.API_URL;
import static com.abiquo.apiclient.domain.ApiPath.LOGIN_URL;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.IOException;

import com.abiquo.apiclient.RestClient;
import com.abiquo.apiclient.auth.Authentication;
import com.abiquo.apiclient.auth.BasicAuthentication;
import com.abiquo.apiclient.auth.TokenAuthentication;
import com.abiquo.server.core.enterprise.UserDto;
import com.squareup.okhttp.Interceptor;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

/**
 * Authenticates the requests using the configured {@link Authentication} strategy.
 *
 * @author Ignasi Barrera
 * @author Sergi Castro
 */
public class AuthenticationInterceptor implements Interceptor
{
    private Authentication authentication;

    private final RestClient client;

    public AuthenticationInterceptor(final Authentication authentication, final RestClient client)
    {
        this.authentication = checkNotNull(authentication, "authentication cannot be null");
        this.client = checkNotNull(client, "the same instance of client cannot be null");
    }

    @Override
    public Response intercept(final Chain chain) throws IOException
    {
        if (authentication.allowAbiquoToken()
            && BasicAuthentication.class.isInstance(authentication)
            && !chain.request().url().getPath().equals(API_URL + LOGIN_URL))
        {
            client.get(LOGIN_URL, UserDto.MEDIA_TYPE, UserDto.class);
        }

        Request authenticated = authentication.authenticate(chain.request());
        Response response = chain.proceed(authenticated);

        if (authentication.allowAbiquoToken())
        {
            String token = response.header(TokenAuthentication.AUTH_HEADER);
            if (token != null)
            {
                authentication = TokenAuthentication.token(token);
            }
        }

        return response;
    }

}
