/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient;

import static com.abiquo.apiclient.domain.ApiPath.DATACENTERS_URL;
import static com.abiquo.apiclient.domain.ApiPath.LOADLEVELRULES_URL;
import static com.abiquo.apiclient.domain.ApiPath.PUBLIC_CLOUD_REGIONS_URL;
import static com.abiquo.apiclient.domain.ApiPath.REMOTE_SERVICES_URL;
import static com.abiquo.apiclient.domain.ApiPath.ROLES_URL;
import static com.abiquo.apiclient.domain.Links.create;
import static com.abiquo.apiclient.domain.Links.editOrSelf;
import static com.abiquo.apiclient.domain.Links.withRel;
import static java.util.Objects.requireNonNull;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.abiquo.apiclient.domain.LinkRel;
import com.abiquo.apiclient.domain.exception.HttpException;
import com.abiquo.apiclient.domain.options.DatacenterListOptions;
import com.abiquo.apiclient.domain.options.NatIPsListOptions;
import com.abiquo.apiclient.domain.options.PublicCloudRegionListOptions;
import com.abiquo.apiclient.domain.options.VirtualMachineInfrastructureListOptions;
import com.abiquo.model.enumerator.NetworkType;
import com.abiquo.model.rest.RESTLink;
import com.abiquo.model.transport.LinksDto;
import com.abiquo.server.core.cloud.DeviceDto;
import com.abiquo.server.core.cloud.DeviceTypeDto;
import com.abiquo.server.core.cloud.HardwareProfileDto;
import com.abiquo.server.core.cloud.HardwareProfilesDto;
import com.abiquo.server.core.cloud.NatIpDto;
import com.abiquo.server.core.cloud.NatIpsDto;
import com.abiquo.server.core.cloud.NatNetworkDto;
import com.abiquo.server.core.cloud.NatNetworksDto;
import com.abiquo.server.core.cloud.RegionDto;
import com.abiquo.server.core.cloud.VirtualMachineDto;
import com.abiquo.server.core.cloud.VirtualMachineFlatDto;
import com.abiquo.server.core.cloud.VirtualMachinesDto;
import com.abiquo.server.core.enterprise.DatacenterLimitsDto;
import com.abiquo.server.core.enterprise.DatacentersLimitsDto;
import com.abiquo.server.core.enterprise.EnterpriseDto;
import com.abiquo.server.core.enterprise.PrivilegeDto;
import com.abiquo.server.core.enterprise.PrivilegesDto;
import com.abiquo.server.core.enterprise.RoleDto;
import com.abiquo.server.core.enterprise.RolesDto;
import com.abiquo.server.core.infrastructure.ClusterDto;
import com.abiquo.server.core.infrastructure.ClustersDto;
import com.abiquo.server.core.infrastructure.DatacenterDto;
import com.abiquo.server.core.infrastructure.DatacentersDto;
import com.abiquo.server.core.infrastructure.DatastoreDto;
import com.abiquo.server.core.infrastructure.DatastoreTierDto;
import com.abiquo.server.core.infrastructure.DatastoreTiersDto;
import com.abiquo.server.core.infrastructure.LocationDto;
import com.abiquo.server.core.infrastructure.MachineDto;
import com.abiquo.server.core.infrastructure.MachinesDto;
import com.abiquo.server.core.infrastructure.PublicCloudRegionDto;
import com.abiquo.server.core.infrastructure.PublicCloudRegionsDto;
import com.abiquo.server.core.infrastructure.RackDto;
import com.abiquo.server.core.infrastructure.RacksDto;
import com.abiquo.server.core.infrastructure.RemoteServiceDto;
import com.abiquo.server.core.infrastructure.RemoteServicesDto;
import com.abiquo.server.core.infrastructure.network.IpsBulkCreationDto;
import com.abiquo.server.core.infrastructure.network.NetworkServiceTypeDto;
import com.abiquo.server.core.infrastructure.network.NetworkServiceTypesDto;
import com.abiquo.server.core.infrastructure.network.VLANNetworkDto;
import com.abiquo.server.core.infrastructure.network.VLANNetworksDto;
import com.abiquo.server.core.infrastructure.storage.StorageDeviceDto;
import com.abiquo.server.core.infrastructure.storage.StorageDevicesDto;
import com.abiquo.server.core.infrastructure.storage.StoragePoolDto;
import com.abiquo.server.core.infrastructure.storage.StoragePoolsDto;
import com.abiquo.server.core.infrastructure.storage.TierDto;
import com.abiquo.server.core.infrastructure.storage.TiersDto;
import com.abiquo.server.core.scheduler.DatastoreLoadRuleDto;
import com.abiquo.server.core.scheduler.MachineLoadRuleDto;

public class InfrastructureApi
{
    private final RestClient client;

    // Package private constructor to be used only by the ApiClient
    InfrastructureApi(final RestClient client)
    {
        this.client = requireNonNull(client, "client cannot be null");
    }

    public Stream<DatacenterDto> listDatacenters()
    {
        return client.list(DATACENTERS_URL, DatacentersDto.MEDIA_TYPE, DatacentersDto.class);
    }

    public Stream<DatacenterDto> listDatacenters(final DatacenterListOptions options)
    {
        return client.list(DATACENTERS_URL, options.queryParams(), DatacentersDto.MEDIA_TYPE,
            DatacentersDto.class);
    }

    public Stream<RackDto> listRacks(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink("racks").getHref(), RacksDto.MEDIA_TYPE,
            RacksDto.class);
    }

    public Stream<DatacenterLimitsDto> listLimits(final EnterpriseDto enterprise)
    {
        return client.list(enterprise.searchLink("limits"), DatacentersLimitsDto.class);
    }

    public DatacenterLimitsDto getEnterpriseLimitsForDatacenter(final EnterpriseDto enterprise,
        final DatacenterDto datacenter)
    {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("datacenter", datacenter.getId());
        return client.get(enterprise.searchLink("limits").getHref(), params,
            DatacentersLimitsDto.MEDIA_TYPE, DatacentersLimitsDto.class).getCollection().get(0);
    }

    public Stream<VLANNetworkDto> listExternalNetworks(final DatacenterLimitsDto limits)
    {
        return client.list(limits.searchLink("externalnetworks").getHref(),
            VLANNetworksDto.MEDIA_TYPE, VLANNetworksDto.class);
    }

    /**
     * Creates a new datacenter using new remote services. If any remote service in remoteServices
     * exists, it is reused for the new datacenter.
     *
     * @param name The new datacenter name
     * @param location The new datacenter location
     * @param remoteServices The remote services the new datacenter needs
     * @return The new datacenter dto
     */
    public DatacenterDto createDatacenter(final String name, final String location,
        final List<RemoteServiceDto> remoteServices)
    {
        DatacenterDto datacenter = new DatacenterDto();
        List<RemoteServiceDto> existentRSs = listRemoteServices().collect(Collectors.toList());

        RemoteServicesDto remoteServicesDto = new RemoteServicesDto();
        remoteServicesDto.addAll(remoteServices);

        for (final RemoteServiceDto rs : remoteServices)
        {
            Optional<RemoteServiceDto> exRS = existentRSs.stream().filter(
                remserv -> remserv.getType().equals(rs.getType()) && remserv.getType().isReusable())
                .findFirst();

            if (exRS.isPresent())
            {
                datacenter.addLink(withRel("remoteservice", editOrSelf(exRS.get())));
                remoteServicesDto.getCollection().remove(rs);
            }
        }

        datacenter.setName(name);
        datacenter.setLocation(location);
        datacenter.setRemoteServices(remoteServicesDto);

        return client.post(DATACENTERS_URL, DatacenterDto.MEDIA_TYPE, DatacenterDto.MEDIA_TYPE,
            datacenter, DatacenterDto.class);
    }

    /**
     * Creates a new public cloud region. If any remote service in remoteServices exists, it is
     * reused for the new public cloud region.
     *
     * @param name The new public cloud region name
     * @param region The new public cloud region provider's region
     * @param type The new public cloud region type
     * @param remoteServices The remote services the new public cloud region needs
     * @return The new public cloud region dto
     */
    public PublicCloudRegionDto createPublicCloudRegion(final String name, final RegionDto region,
        final List<RemoteServiceDto> remoteServices)
    {
        PublicCloudRegionDto publicCloudRegion = new PublicCloudRegionDto();
        List<RemoteServiceDto> existentRSs = listRemoteServices().collect(Collectors.toList());

        RemoteServicesDto remoteServicesDto = new RemoteServicesDto();
        remoteServicesDto.addAll(remoteServices);

        for (final RemoteServiceDto rs : remoteServices)
        {

            Optional<RemoteServiceDto> exRS = existentRSs.stream()
                .filter(remserv -> remserv.getType().equals(rs.getType())).findFirst();

            if (exRS.isPresent())
            {
                publicCloudRegion.addLink(withRel("remoteservice", editOrSelf(exRS.get())));
                remoteServicesDto.getCollection().remove(rs);
            }
        }

        publicCloudRegion.setName(name);
        publicCloudRegion.setRemoteServices(remoteServicesDto);
        publicCloudRegion.addLink(withRel("region", editOrSelf(region)));

        return client.post(PUBLIC_CLOUD_REGIONS_URL, PublicCloudRegionDto.MEDIA_TYPE,
            PublicCloudRegionDto.MEDIA_TYPE, publicCloudRegion, PublicCloudRegionDto.class);
    }

    /**
     * Creates a new public cloud region. If any remote service in remoteServices exists, it is
     * reused for the new public cloud region.
     *
     * @param name The new public cloud region name
     * @param region The new public cloud region provider's region
     * @param type The new public cloud region type
     * @param remoteServices The remote services the new public cloud region needs
     * @param identity X-Abiquo-PCR-Identity header
     * @param credential X-Abiquo-PCR-Credential header
     * @param endpoint X-Abiquo-PCR-Endpoint header
     * @return The new public cloud region dto
     */
    public PublicCloudRegionDto createPublicCloudDynamicRegion(final String name,
        final RegionDto region, final List<RemoteServiceDto> remoteServices, final String identity,
        final String credential, final String endpoint)
    {
        PublicCloudRegionDto publicCloudRegion = new PublicCloudRegionDto();
        List<RemoteServiceDto> existentRSs = listRemoteServices().collect(Collectors.toList());

        RemoteServicesDto remoteServicesDto = new RemoteServicesDto();
        remoteServicesDto.addAll(remoteServices);

        for (final RemoteServiceDto rs : remoteServices)
        {
            Optional<RemoteServiceDto> exRS = existentRSs.stream()
                .filter(remserv -> remserv.getType().equals(rs.getType())).findFirst();

            if (exRS.isPresent())
            {
                publicCloudRegion.addLink(withRel("remoteservice", editOrSelf(exRS.get())));
                remoteServicesDto.getCollection().remove(rs);
            }
        }

        publicCloudRegion.setName(name);
        publicCloudRegion.setRemoteServices(remoteServicesDto);
        publicCloudRegion.addLink(withRel("region", editOrSelf(region)));

        Map<String, String> headers = new HashMap<>();
        headers.put("X-Abiquo-PCR-Identity", identity);
        headers.put("X-Abiquo-PCR-Credential", credential);
        headers.put("X-Abiquo-PCR-Endpoint", endpoint);

        return client.post(PUBLIC_CLOUD_REGIONS_URL, PublicCloudRegionDto.MEDIA_TYPE,
            PublicCloudRegionDto.MEDIA_TYPE, publicCloudRegion, PublicCloudRegionDto.class,
            headers);
    }

    public Stream<RemoteServiceDto> listRemoteServices(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink("remoteservices"), RemoteServicesDto.class);
    }

    public Stream<RemoteServiceDto> listRemoteServices()
    {
        return client.list(REMOTE_SERVICES_URL, RemoteServicesDto.MEDIA_TYPE,
            RemoteServicesDto.class);
    }

    public RackDto createRack(final DatacenterDto datacenter, final String name)
    {
        RackDto rack = new RackDto();
        rack.setName(name);
        return client.post(datacenter.searchLink("racks").getHref(), RackDto.MEDIA_TYPE,
            RackDto.MEDIA_TYPE, rack, RackDto.class);
    }

    /** addLocationToEnterprise */
    @Deprecated
    public void addDatacenterToEnterprise(final EnterpriseDto enterprise,
        final DatacenterDto datacenter)
    {
        addLocationToEnterprise(enterprise, datacenter);
    }

    /** addLocationToEnterprise */
    @Deprecated
    public void addPublicCloudRegionToEnterprise(final EnterpriseDto enterprise,
        final PublicCloudRegionDto pcr)
    {
        addLocationToEnterprise(enterprise, pcr);
    }

    public DatacenterLimitsDto addLocationToEnterprise(final EnterpriseDto enterprise,
        final LocationDto location)
    {
        RESTLink editlink = location.getEditLink();
        DatacenterLimitsDto limits = new DatacenterLimitsDto();
        limits.addLink(create("location", editlink.getHref(), editlink.getType()));

        return client.post(enterprise.searchLink("limits").getHref(),
            DatacenterLimitsDto.MEDIA_TYPE, DatacenterLimitsDto.MEDIA_TYPE, limits,
            DatacenterLimitsDto.class);
    }

    public MachinesDto discoverMachines(final DatacenterDto datacenter, final String type,
        final String ip, final String user, final String password)
    {
        Map<String, Object> queryParams = new HashMap<String, Object>();
        queryParams.put("hypervisor", type);
        queryParams.put("ip", ip);
        queryParams.put("user", user);
        queryParams.put("password", password);

        return client.get(datacenter.searchLink("discover").getHref(), queryParams,
            MachinesDto.MEDIA_TYPE, MachinesDto.class);
    }

    public MachinesDto discoverManagedMachines(final DatacenterDto datacenter, final String type,
        final String ip, final String managerIp, final String managerUser,
        final String managerPassword)
    {
        Map<String, Object> queryParams = new HashMap<String, Object>();
        queryParams.put("hypervisor", type);
        queryParams.put("ip", ip);
        queryParams.put("managerip", managerIp);
        queryParams.put("manageruser", managerUser);
        queryParams.put("managerpassword", managerPassword);

        return client.get(datacenter.searchLink("discover").getHref(), queryParams,
            MachinesDto.MEDIA_TYPE, MachinesDto.class);
    }

    public MachineDto createMachine(final RackDto rack, final MachineDto machine)
    {
        return client.post(rack.searchLink("machines").getHref(), MachineDto.MEDIA_TYPE,
            MachineDto.MEDIA_TYPE, machine, MachineDto.class);
    }

    public Stream<NetworkServiceTypeDto> listNetworkServiceTypes(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink("networkservicetypes").getHref(),
            NetworkServiceTypesDto.MEDIA_TYPE, NetworkServiceTypesDto.class);
    }

    public MachineLoadRuleDto createDatacenterLoadLevelRule(final DatacenterDto datacenter,
        final int cpuLoadPercentage, final int ramLoadPercentage)
    {
        return createLoadLevelRule(create("datacenter", datacenter.getEditLink().getHref(),
            datacenter.getEditLink().getType()), cpuLoadPercentage, ramLoadPercentage);
    }

    public MachineLoadRuleDto createRackLoadLevelRule(final RackDto rack,
        final int cpuLoadPercentage, final int ramLoadPercentage)
    {
        return createLoadLevelRule(
            create("rack", rack.getEditLink().getHref(), rack.getEditLink().getType()),
            cpuLoadPercentage, ramLoadPercentage);
    }

    public MachineLoadRuleDto createMachineLoadLevelRule(final MachineDto machine,
        final int cpuLoadPercentage, final int ramLoadPercentage)
    {
        return createLoadLevelRule(
            create("machine", machine.getEditLink().getHref(), machine.getEditLink().getType()),
            cpuLoadPercentage, ramLoadPercentage);
    }

    private MachineLoadRuleDto createLoadLevelRule(final RESTLink targetLink,
        final int cpuLoadPercentage, final int ramLoadPercentage)
    {
        MachineLoadRuleDto rule = new MachineLoadRuleDto();
        rule.setCpuLoadPercentage(cpuLoadPercentage);
        rule.setRamLoadPercentage(ramLoadPercentage);
        rule.addLink(targetLink);

        return client.post(LOADLEVELRULES_URL + "/machineLoadLevel", MachineLoadRuleDto.MEDIA_TYPE,
            MachineLoadRuleDto.MEDIA_TYPE, rule, MachineLoadRuleDto.class);
    }

    public DatastoreLoadRuleDto createDatacenterLoadLevelStorageRule(final DatacenterDto datacenter,
        final int storageLoadPercentage)
    {
        return createLoadLevelRule(create("datacenter", datacenter.getEditLink().getHref(),
            datacenter.getEditLink().getType()), storageLoadPercentage);
    }

    public DatastoreLoadRuleDto createDatastoreTierLoadLevelStorageRule(
        final DatastoreTierDto datastoreTier, final int storageLoadPercentage)
    {
        return createLoadLevelRule(create("datastoretier", datastoreTier.getEditLink().getHref(),
            datastoreTier.getEditLink().getType()), storageLoadPercentage);
    }

    public DatastoreLoadRuleDto createDatastoreLoadLevelStorageRule(final DatastoreDto datastore,
        final int storageLoadPercentage)
    {
        return createLoadLevelRule(create("datastore", datastore.getEditLink().getHref(),
            datastore.getEditLink().getType()), storageLoadPercentage);
    }

    private DatastoreLoadRuleDto createLoadLevelRule(final RESTLink targetLink,
        final int storageLoadPercentage)
    {
        DatastoreLoadRuleDto rule = new DatastoreLoadRuleDto();
        rule.setStorageLoadPercentage(storageLoadPercentage);
        rule.addLink(targetLink);
        return client.post(LOADLEVELRULES_URL + "/datastoreloadlevel",
            DatastoreLoadRuleDto.MEDIA_TYPE, DatastoreLoadRuleDto.MEDIA_TYPE, rule,
            DatastoreLoadRuleDto.class);
    }

    public Stream<StorageDeviceDto> listDevices(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink("devices").getHref(), StorageDevicesDto.MEDIA_TYPE,
            StorageDevicesDto.class);
    }

    public Stream<StoragePoolDto> listPools(final StorageDeviceDto device)
    {
        return client.list(device.searchLink("pools").getHref(), StoragePoolsDto.MEDIA_TYPE,
            StoragePoolsDto.class);
    }

    public Stream<StoragePoolDto> listRemotePools(final StorageDeviceDto device)
    {
        Map<String, Object> queryParams = new HashMap<String, Object>();
        queryParams.put("sync", true);

        return client.list(device.searchLink("pools").getHref(), queryParams,
            StoragePoolsDto.MEDIA_TYPE, StoragePoolsDto.class);
    }

    public StorageDeviceDto createStorageDevice(final DatacenterDto datacenter, final String name,
        final String technology, final String managementIp, final int managementPort,
        final String serviceIp, final int servicePort, final String username, final String password)
    {
        StorageDeviceDto device = new StorageDeviceDto();
        device.addLink(create("datacenter", datacenter.getEditLink().getHref(),
            datacenter.getEditLink().getType()));
        device.setName(name);
        device.setStorageTechnology(technology);
        device.setManagementIp(managementIp);
        device.setManagementPort(managementPort);
        device.setServiceIp(serviceIp);
        device.setServicePort(servicePort);
        device.setUsername(username);
        device.setPassword(password);

        return client.post(datacenter.searchLink("devices").getHref(), StorageDeviceDto.MEDIA_TYPE,
            StorageDeviceDto.MEDIA_TYPE, device, StorageDeviceDto.class);
    }

    public StoragePoolDto createPool(final DatacenterDto datacenter,
        final StorageDeviceDto storageDevice, final String pool, final String tierName)
    {
        StoragePoolDto storagePool =
            listRemotePools(storageDevice).filter(stPool -> stPool.getName().equals(pool))
                .findFirst().orElseThrow(() -> new HttpException(404, "Storage pool not found"));

        TierDto tier = listTiers(datacenter).filter(t -> t.getName().equals(tierName)).findFirst()
            .orElseThrow(() -> new HttpException(404, "Tier not found"));

        storagePool.setEnabled(true);
        storagePool
            .addLink(create("tier", tier.getEditLink().getHref(), tier.getEditLink().getType()));

        return client.post(storageDevice.searchLink("pools").getHref(), StoragePoolDto.MEDIA_TYPE,
            StoragePoolDto.MEDIA_TYPE, storagePool, StoragePoolDto.class);
    }

    public VLANNetworkDto createExternalNetwork(final DatacenterDto datacenter,
        final NetworkServiceTypeDto nst, final EnterpriseDto enterprise, final String name,
        final String address, final String gateway, final int mask, final int tag,
        final Optional<DeviceDto> optDevice)
    {
        return createInfrastructureNetwork(datacenter, nst, enterprise, name, address, gateway,
            mask, tag, optDevice, NetworkType.EXTERNAL);
    }

    public VLANNetworkDto createUnmanagedNetwork(final DatacenterDto datacenter,
        final NetworkServiceTypeDto nst, final EnterpriseDto enterprise, final String name,
        final String address, final String gateway, final int mask, final int tag,
        final Optional<DeviceDto> optDevice)
    {
        return createInfrastructureNetwork(datacenter, nst, enterprise, name, address, gateway,
            mask, tag, optDevice, NetworkType.UNMANAGED);
    }

    private VLANNetworkDto createInfrastructureNetwork(final DatacenterDto datacenter,
        final NetworkServiceTypeDto nst, final EnterpriseDto enterprise, final String name,
        final String address, final String gateway, final int mask, final int tag,
        final Optional<DeviceDto> optDevice, final NetworkType type)
    {
        VLANNetworkDto vlan = new VLANNetworkDto();
        vlan.addLink(create("enterprise", enterprise.getEditLink().getHref(),
            enterprise.getEditLink().getType()));
        vlan.addLink(
            create("networkservicetype", nst.getEditLink().getHref(), nst.getEditLink().getType()));
        vlan.setAddress(address);
        vlan.setName(name);
        vlan.setType(type);
        vlan.setMask(mask);
        vlan.setTag(tag);
        vlan.setGateway(gateway);
        vlan.setUnmanaged(NetworkType.UNMANAGED == type);

        String url = datacenter.searchLink("network").getHref();
        if (optDevice.isPresent())
        {
            RESTLink deviceEdit = optDevice.get().getEditLink();
            vlan.addLink(create("device", deviceEdit.getHref(), deviceEdit.getType()));
        }

        return client.post(url, VLANNetworkDto.MEDIA_TYPE, VLANNetworkDto.MEDIA_TYPE, vlan,
            VLANNetworkDto.class);
    }

    public Stream<TierDto> listTiers(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink("tiers").getHref(), TiersDto.MEDIA_TYPE,
            TiersDto.class);
    }

    public Stream<PublicCloudRegionDto> listPublicCloudRegions(
        final PublicCloudRegionListOptions options)
    {
        return client.list(PUBLIC_CLOUD_REGIONS_URL, options.queryParams(),
            PublicCloudRegionsDto.MEDIA_TYPE, PublicCloudRegionsDto.class);
    }

    public Stream<PublicCloudRegionDto> listPublicCloudRegions()
    {
        return client.list(PUBLIC_CLOUD_REGIONS_URL, PublicCloudRegionsDto.MEDIA_TYPE,
            PublicCloudRegionsDto.class);
    }

    public Stream<MachineDto> listMachines(final RackDto rack)
    {
        return client.list(rack.searchLink("machines").getHref(), MachinesDto.MEDIA_TYPE,
            MachinesDto.class);
    }

    public Stream<VirtualMachineDto> listInfrastructureVirtualMachines(final MachineDto machine)
    {
        return client.list(machine.searchLink(LinkRel.VIRTUALMACHINES).getHref(),
            VirtualMachinesDto.MEDIA_TYPE, VirtualMachinesDto.class);
    }

    public Stream<VirtualMachineDto> listInfrastructureVirtualMachines(final MachineDto machine,
        final VirtualMachineInfrastructureListOptions options)
    {
        return client.list(machine.searchLink(LinkRel.VIRTUALMACHINES).getHref(),
            options.queryParams(), VirtualMachinesDto.MEDIA_TYPE, VirtualMachinesDto.class);
    }

    public VirtualMachineFlatDto getVirtualMachineFlat(final VirtualMachineDto vm,
        final VirtualMachineInfrastructureListOptions options)
    {
        return client.get(vm.searchLink(LinkRel.VIRTUALMACHINE).getHref(), options.queryParams(),
            VirtualMachineFlatDto.MEDIA_TYPE, VirtualMachineFlatDto.class);
    }

    public Stream<ClusterDto> listClusters(final RackDto rack)
    {
        return client.list(rack.searchLink("clusters").getHref(), ClustersDto.MEDIA_TYPE,
            ClustersDto.class);
    }

    public DeviceDto createDevice(final DatacenterDto dc, final DeviceTypeDto deviceType,
        final String name, final String endpoint, final String user, final String password,
        final boolean vdcDefault)
    {
        DeviceDto device = new DeviceDto();
        device.addLink(withRel("devicetype", editOrSelf(deviceType)));
        device.setName(name);
        device.setEndpoint(endpoint);
        device.setUser(user);
        device.setPassword(password);
        device.setVdcDefault(vdcDefault);

        return client.post(dc.searchLink("devices", "devices").getHref(), //
            DeviceDto.MEDIA_TYPE, DeviceDto.MEDIA_TYPE, device, DeviceDto.class);
    }

    public NatNetworkDto createNatNetwork(final DatacenterDto dc, final String name,
        final String cidr, final DeviceDto device)
    {
        NatNetworkDto nat = new NatNetworkDto();
        nat.setName(name);
        nat.setCidr(cidr);
        nat.addLink(
            create("device", device.getEditLink().getHref(), device.getEditLink().getType()));

        return client.post(dc.searchLink("natnetworks").getHref(), NatNetworkDto.MEDIA_TYPE,
            NatNetworkDto.MEDIA_TYPE, nat, NatNetworkDto.class);
    }

    public Stream<NatNetworkDto> listNatNetworks(final DatacenterDto dc)
    {
        return client.list(dc.searchLink("natnetworks").getHref(), NatNetworksDto.MEDIA_TYPE,
            NatNetworksDto.class);
    }

    public NatNetworkDto editNatNetwork(final NatNetworkDto natNw)
    {
        return client.edit(natNw);
    }

    public void deleteNatNetwork(final NatNetworkDto natNw)
    {
        client.delete(natNw);
    }

    public RoleDto createRole(final String name, final boolean blocked,
        final String... externalRoles)
    {
        RoleDto role = new RoleDto();
        Arrays.stream(externalRoles).forEach(role.getExternalRoles()::add);
        role.setName(name);
        role.setBlocked(blocked);
        return client.post(ROLES_URL, RoleDto.MEDIA_TYPE, RoleDto.MEDIA_TYPE, role, RoleDto.class);
    }

    public Stream<RoleDto> listRoles()
    {
        return client.list(ROLES_URL, RolesDto.MEDIA_TYPE, RolesDto.class);
    }

    public void deleteRole(final RoleDto role)
    {
        client.delete(role);
    }

    public RoleDto getRole(final int roleId)
    {
        return client.get(ROLES_URL + "/" + roleId, RoleDto.MEDIA_TYPE, RoleDto.class);
    }

    public RoleDto editRole(final RoleDto role)
    {
        return client.edit(role);
    }

    public LinksDto listPrivilegeLinksOfARole(final RoleDto role)
    {
        return client.get(role.searchLink("privileges").getHref(), LinksDto.MEDIA_TYPE,
            LinksDto.class);
    }

    public Stream<PrivilegeDto> listPrivilegesOfARole(final RoleDto role)
    {
        return client.list(role.searchLink("privileges").getHref(), PrivilegesDto.MEDIA_TYPE,
            PrivilegesDto.class);
    }

    public NatIpDto createNatIp(final NatNetworkDto natNetwork, final String ip)
    {
        NatIpDto natIP = new NatIpDto();
        natIP.setIp(ip);
        return client.post(natNetwork.searchLink("ips").getHref(), NatIpDto.MEDIA_TYPE,
            NatIpDto.MEDIA_TYPE, natIP, NatIpDto.class);
    }

    public IpsBulkCreationDto createMultipleNatIps(final NatNetworkDto natNetwork,
        final String startingIP, final int numips)
    {
        IpsBulkCreationDto natIPs = new IpsBulkCreationDto();
        natIPs.setIp(startingIP);
        natIPs.setNumips(numips);
        return client.post(natNetwork.searchLink("ips").getHref(), NatIpsDto.MEDIA_TYPE,
            IpsBulkCreationDto.MEDIA_TYPE, natIPs, IpsBulkCreationDto.class);
    }

    public Stream<NatIpDto> listNatIPs(final NatNetworkDto natNw)
    {
        return client.list(natNw.searchLink("ips").getHref(), NatIpsDto.MEDIA_TYPE,
            NatIpsDto.class);
    }

    public Stream<NatIpDto> listNatIPs(final NatNetworkDto natNw, final NatIPsListOptions options)
    {
        return client.list(natNw.searchLink("ips").getHref(), options.queryParams(),
            NatIpsDto.MEDIA_TYPE, NatIpsDto.class);
    }

    public void deleteNatIp(final NatIpDto natIP)
    {
        client.delete(natIP);
    }

    public HardwareProfileDto createHardwareProfile(final DatacenterDto datacenter,
        final HardwareProfileDto hardwareProfile)
    {
        return client.post(datacenter.searchLink(LinkRel.HARDWAREPROFILES).getHref(),
            HardwareProfileDto.MEDIA_TYPE, HardwareProfileDto.MEDIA_TYPE, hardwareProfile,
            HardwareProfileDto.class);
    }

    public Stream<HardwareProfileDto> listHardwareProfiles(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink(LinkRel.HARDWAREPROFILES),
            HardwareProfilesDto.class);
    }

    public Stream<DatastoreTierDto> listDatastoreTiers(final DatacenterDto datacenter)
    {
        return client.list(datacenter.searchLink(LinkRel.DATASTORETIERS), DatastoreTiersDto.class);
    }

}
