/**
 * Copyright (C) 2008 Abiquo Holdings S.L.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.abiquo.apiclient.stream;

import static com.google.common.util.concurrent.Uninterruptibles.sleepUninterruptibly;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.stream.Collectors.joining;
import static org.atmosphere.wasync.Event.CLOSE;
import static org.atmosphere.wasync.Event.MESSAGE;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;

import org.atmosphere.wasync.ClientFactory;
import org.atmosphere.wasync.Request.METHOD;
import org.atmosphere.wasync.Request.TRANSPORT;
import org.atmosphere.wasync.Socket;
import org.atmosphere.wasync.Socket.STATUS;
import org.atmosphere.wasync.impl.AtmosphereClient;
import org.atmosphere.wasync.impl.AtmosphereRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.abiquo.tracing.model.Trace;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ning.http.client.AsyncHttpClient;
import com.ning.http.client.AsyncHttpClientConfig;
import com.ning.http.client.Param;
import com.ning.http.client.Realm;
import com.ning.http.client.Response;
import com.ning.http.client.oauth.ConsumerKey;
import com.ning.http.client.oauth.OAuthSignatureCalculator;
import com.ning.http.client.oauth.RequestToken;
import com.ning.http.client.uri.Uri;

public class StreamClient implements AutoCloseable
{
    private static final Logger LOGGER = LoggerFactory.getLogger("abiquo.stream");

    private final String mEndpoint;

    private final String username;

    private final String password;

    private final String apiKey;

    private final String apiSecret;

    private final String token;

    private final String tokenSecret;

    private final String apiEndpoint;

    private final SSLConfiguration sslConfiguration;

    private final List<Consumer<Trace>> consumers;

    public final String filters;

    public Long lastEventProcessed = System.currentTimeMillis();

    private final boolean reconnect;

    private final int reconnectAttempts;

    private final int pauseBeforeReconnectInSeconds;

    private final Consumer<StreamClient> beforeReconnection;

    private final Consumer<StreamClient> afterReconnection;

    private final Consumer<StreamClient> defaultReconnectCallback =
        this::getEventsFromApiAndProcessCallback;

    public final static ObjectMapper json = new ObjectMapper()//
        .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

    public final static DateFormat dateFormat =
        new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");

    static
    {
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    public AsyncHttpClient httpClient;

    private Socket socket;

    private ExecutorService asyncHttpexecutor;

    private AtomicBoolean manuallyClosing = new AtomicBoolean(false);

    private AtomicBoolean reconnecting = new AtomicBoolean(false);

    private AtomicBoolean beforeReconnectExecuted = new AtomicBoolean(false);

    private ExecutorService reconnectionExecutor = Executors.newSingleThreadExecutor();

    // Do not use directly. Use the builder.
    private StreamClient(final String mEndpoint, final String username, final String password,
        final String apiKey, final String apiSecret, final String token, final String tokenSecret,
        final String apiEndpoint, final SSLConfiguration sslConfiguration,
        final List<Consumer<Trace>> consumers, final Consumer<StreamClient> beforeReconnection,
        final Consumer<StreamClient> afterReconnection, final boolean reconnect,
        final int reconnectAttempts, final int pauseBeforeReconnectInSeconds, final String filters)
    {
        this.mEndpoint = mEndpoint;
        this.username = username;
        this.password = password;
        this.apiKey = apiKey;
        this.apiSecret = apiSecret;
        this.token = token;
        this.tokenSecret = tokenSecret;
        this.apiEndpoint = apiEndpoint;
        this.consumers = consumers;
        this.sslConfiguration = sslConfiguration;
        this.reconnect = reconnect;
        this.filters = filters;
        this.reconnectAttempts = reconnectAttempts;
        this.pauseBeforeReconnectInSeconds = pauseBeforeReconnectInSeconds;
        this.beforeReconnection = beforeReconnection != null ? beforeReconnection //
            : defaultReconnectCallback;
        this.afterReconnection = afterReconnection != null ? afterReconnection //
            : defaultReconnectCallback;

    }

    public void connect() throws IOException
    {
        if (socket != null)
        {
            throw new IllegalStateException("already listening to events");
        }

        beforeReconnectExecuted.set(false);
        reconnecting.set(false);

        // httpClient its also used to request to the API in reconnect, so need +1 thread
        asyncHttpexecutor = Executors.newCachedThreadPool();

        AsyncHttpClientConfig.Builder config = new AsyncHttpClientConfig.Builder();
        config.setRequestTimeout(-1);
        config.setReadTimeout(-1);
        config.setExecutorService(asyncHttpexecutor);

        if (sslConfiguration != null)
        {
            config.setHostnameVerifier(this.sslConfiguration.hostnameVerifier());
            config.setSSLContext(this.sslConfiguration.sslContext());
        }

        if (username != null)
        {
            config.setRealm(new Realm.RealmBuilder() //
                .setPrincipal(this.username) //
                .setPassword(this.password) //
                .setUsePreemptiveAuth(true) //
                .setScheme(Realm.AuthScheme.BASIC) //
                .build());
        }

        httpClient = new AsyncHttpClient(config.build());

        if (apiKey != null)
        {
            httpClient.setSignatureCalculator(//
                new AbiquoAPIOAuth(new ConsumerKey(this.apiKey, this.apiSecret),
                    new RequestToken(this.token, this.tokenSecret), this.apiEndpoint));
        }

        LOGGER.debug("Connecting to {} with filters  {} ", mEndpoint, filters);

        AtmosphereClient client = ClientFactory.getDefault().newClient(AtmosphereClient.class);
        AtmosphereRequest request = client.newRequestBuilder() //
            .method(METHOD.GET) //
            .uri(mEndpoint + "/stream?Content-Type=application/json" + filters) //
            .transport(TRANSPORT.SSE) //
            .build();

        socket = client.create(client.newOptionsBuilder().runtime(httpClient).build());
        socket.open(request);
        socket//
            .on(MESSAGE, (final String a) -> onMessage(a))//
            .on(CLOSE, (final String a) -> onClose());

        LOGGER.debug("Connected!");
    }

    private void onMessage(final String rawEvent)
    {
        if (!beforeReconnectExecuted.get())
        {
            try
            {
                afterReconnection.accept(this);
            }
            catch (Throwable e)
            {
                LOGGER.error("Cannot run after reconnection logic", e);
            }
            beforeReconnectExecuted.set(true);
        }

        Trace trace;
        try
        {
            trace = json.readValue(rawEvent, Trace.class);
        }
        catch (IOException ex)
        {
            LOGGER.warn("Cannot bind to Trace : " + rawEvent, ex);
            return;
        }

        processTrace(trace);
    }

    protected void processTrace(final Trace trace)
    {
        lastEventProcessed = trace.getTimestamp();

        for (Consumer<Trace> consumer : consumers)
        {
            try
            {
                consumer.accept(trace);
            }
            catch (Throwable e)
            {
                String traceAsString = "";
                try
                {
                    traceAsString = json.writeValueAsString(trace);
                }
                catch (JsonProcessingException e1)
                {
                }
                LOGGER.error("Cannot process consumer for message : " + traceAsString, e);
            }
        }
    }

    private synchronized void onClose()
    {
        if (!manuallyClosing.get() && !reconnecting.get() && reconnect)
        {
            reconnecting.set(true);
            reconnectionExecutor.execute(() -> {
                try
                {
                    beforeReconnection.accept(this);
                }
                catch (Throwable e)
                {
                    LOGGER.error("Cannot run before reconnection logic", e);
                }

                LOGGER.warn("Connection lost, going to reconnect");
                reconnect();
            });
        }
        else
        {
            throw new IllegalStateException("Connection closed");
        }
    }

    private void reconnect()
    {
        Throwable lastE = null;
        int retry = 0;
        while (retry < reconnectAttempts)
        {
            try
            {
                LOGGER.warn("Attempting to reconnect number : " + (retry + 1));
                closeConnection();
                connect();
                return;
            }
            catch (Throwable e)
            {
                lastE = e;
                retry++;

                sleepUninterruptibly(pauseBeforeReconnectInSeconds, TimeUnit.SECONDS);
            }
        }
        LOGGER.warn("Cannot reconnect after " + reconnectAttempts + " attempts", lastE);
        closeConnection();
    }

    @Override
    public synchronized void close() throws IOException
    {
        try
        {
            manuallyClosing.set(true);
            closeConnection();
            reconnectionExecutor.shutdownNow();
        }
        finally
        {
            manuallyClosing.set(false);
        }
    }

    private synchronized void closeConnection()
    {
        LOGGER.debug("Disconnecting...");

        if (httpClient != null)
        {
            try
            {
                httpClient.close();
            }
            catch (Throwable e)
            {
                LOGGER.debug("Failed to close async-http-client", e);
            }
            httpClient = null;
        }

        asyncHttpexecutor.shutdownNow();
        try
        {
            asyncHttpexecutor.awaitTermination(1, TimeUnit.SECONDS);
        }
        catch (Throwable e)
        {
            LOGGER.error("Cannot stop executor service", e);
        }

        // ConcurrentModificationException asyncHttpClient -> wasync
        sleepUninterruptibly(1, TimeUnit.SECONDS);

        if (socket != null)
        {
            try
            {
                if (socket.status() != STATUS.CLOSE)
                {
                    socket.close();

                    while (socket.status() != STATUS.CLOSE)
                    {
                        LOGGER.warn("Waiting socket close");
                        sleepUninterruptibly(100, MILLISECONDS);
                    }
                }
            }
            catch (Throwable e)
            {
                LOGGER.debug("Failed to socket", e);
            }
            socket = null;
        }
        LOGGER.debug("Disconnected!");
    }

    public List<Trace> getEventsFromApiAndProcessCallback(final StreamClient client)
    {
        String apiEvents = String.format("%s/events?datefrom=%s%s&asc=true&limit=0", apiEndpoint,
            StreamClient.dateFormat.format(client.lastEventProcessed), client.filters);

        Response response = null;
        try
        {
            response = client.httpClient.prepareGet(apiEvents).execute().get();
        }
        catch (Exception e)
        {
            throw new RuntimeException("Cannot GET : " + apiEvents, e);
        }

        if (response.getStatusCode() != 200)
        {
            throw new RuntimeException(
                "Failed GET (" + response.getStatusCode() + "): " + apiEvents);
        }

        List<Trace> traces;
        try
        {
            traces = StreamClient.json.readValue(response.getResponseBodyAsStream(), Traces.class)
                .getCollection();
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot serialize events MT");
        }

        if (traces != null)
        {
            for (Trace trace : traces)
            {
                processTrace(trace);
            }
        }
        return traces;
    }

    public static Builder builder(final String endpointToM, final String endpointToApi)
    {
        return new Builder().endpoints(endpointToM, endpointToApi);
    }

    public static class Builder
    {
        private String mEndpoint;

        private String username;

        private String password;

        private SSLConfiguration sslConfiguration;

        private List<Consumer<Trace>> consumers = new ArrayList<>();

        private boolean reconnect = false;

        private int reconnectAttempts = 10;

        private int pauseBeforeReconnectInSeconds = 5;

        private Consumer<StreamClient> beforeReconnect;

        private Consumer<StreamClient> afterReconnect;

        private String apiKey;

        private String apiSecret;

        private String token;

        private String tokenSecret;

        private String apiEndpoint;

        private Set<String> severityFilters = new HashSet<>();

        private Set<String> entityFilters = new HashSet<>();

        private Set<String> actionFilters = new HashSet<>();

        private Set<String> userFilters = new HashSet<>();

        private Set<String> enterpriseFilters = new HashSet<>();

        private Builder()
        {

        }

        /**
         * @param endpointToApi should be the same configured ''abiquo.server.api.location'' in
         *            abiquo.properties.
         */
        private Builder endpoints(final String endpointToM, final String endpointToApi)
        {
            this.mEndpoint = requireNonNull(endpointToM, "endpointToM cannot be null");
            this.apiEndpoint = requireNonNull(endpointToApi, "endpointToApi cannot be null");
            return this;
        }

        /** Authenticate using user and password */
        public Builder basicAuth(final String username, final String password)
        {
            if (apiKey != null)
            {
                throw new IllegalStateException("oauth already configured");
            }
            this.username = requireNonNull(username, "username cannot be null");
            this.password = requireNonNull(password, "password cannot be null");
            return this;
        }

        /**
         * Authenticate using an authorized OAuth application
         */
        public Builder oauth(final String apiKey, final String apiSecret, final String token,
            final String tokenSecret)
        {
            if (username != null)
            {
                throw new IllegalStateException("basic auth already configured");
            }
            this.apiKey = requireNonNull(apiKey, "apiKey cannot be null");
            this.apiSecret = requireNonNull(apiSecret, "apiSecret cannot be null");
            this.token = requireNonNull(token, "token cannot be null");
            this.tokenSecret = requireNonNull(tokenSecret, "tokenSecret cannot be null");
            return this;
        }

        /** Custom SSL configuration */
        public Builder sslConfiguration(final SSLConfiguration sslConfiguration)
        {
            this.sslConfiguration = sslConfiguration;
            return this;
        }

        /**
         * Filter by severity
         *
         * @param severity : INFO, WARN, ERROR
         */
        public Builder addFilterBySeverity(final String severity)
        {
            this.severityFilters.add(requireNonNull(severity, "severity cannot be null"));
            return this;
        }

        /**
         * Filter by type of entity
         *
         * @param entity see
         *            https://wiki.abiquo.com/api/latest/EventsResource.html#list-events-types
         */
        public Builder addFilterByEntity(final String entity)
        {
            this.entityFilters.add(requireNonNull(entity, "entity cannot be null"));
            return this;
        }

        /**
         * Filter by type of action
         *
         * @param entity see
         *            https://wiki.abiquo.com/api/latest/EventsResource.html#list-events-types
         */
        public Builder addFilterByAction(final String action)
        {
            this.actionFilters.add(requireNonNull(action, "action cannot be null"));
            return this;
        }

        /**
         * Filter by user ID
         *
         * @param userId identifier of the user
         */
        public Builder addFilterByUser(final Integer userId)
        {
            this.userFilters.add(requireNonNull(userId, "userId cannot be null").toString());
            return this;
        }

        /**
         * Filter by enterprise ID
         *
         * @param enterpriseId identifier of the enterprise
         */
        public Builder addFilterByEnterprise(final Integer enterpriseId)
        {
            this.enterpriseFilters
                .add(requireNonNull(enterpriseId, "enterpriseId cannot be null").toString());
            return this;
        }

        /**
         * Add a reaction to trace events
         */
        public Builder addCallback(final Consumer<Trace> callback)
        {
            this.consumers.add(requireNonNull(callback, "callback cannot be null"));
            return this;
        }

        /**
         * Activate reconnection (default false)
         */
        public Builder reconnect()
        {
            this.reconnect = true;
            return this;
        }

        /**
         * Number of reconnection attempts (default 10)
         */
        public Builder reconnectAttempts(final int attempts)
        {
            this.reconnectAttempts = attempts;
            return this;
        }

        /**
         * Seconds before attempt reconnect (default 5 seconds)
         */
        public Builder pauseBeforeReconnectInSeconds(final int seconds)
        {
            this.pauseBeforeReconnectInSeconds = seconds;
            return this;
        }

        /**
         * Run before attempt reconnection
         */
        public Builder beforeReconnect(final Consumer<StreamClient> beforeReconnect)
        {
            this.beforeReconnect = beforeReconnect;
            return this;
        }

        /**
         * Run after reconnection succeed
         */
        public Builder afterReconnect(final Consumer<StreamClient> afterReconnect)
        {
            this.afterReconnect = afterReconnect;
            return this;
        }

        public StreamClient build()
        {
            String filters = new String();
            if (!severityFilters.isEmpty())
            {
                filters += "&severity=" + severityFilters.stream().collect(joining(","));
            }
            if (!entityFilters.isEmpty())
            {
                filters += "&entity=" + entityFilters.stream().collect(joining(","));
            }
            if (!actionFilters.isEmpty())
            {
                filters += "&action=" + actionFilters.stream().collect(joining(","));
            }
            if (!userFilters.isEmpty())
            {
                filters += "&user=" + userFilters.stream().collect(joining(","));
            }
            if (!enterpriseFilters.isEmpty())
            {
                filters += "&enterprise=" + enterpriseFilters.stream().collect(joining(","));
            }
            if (reconnect && reconnectAttempts <= 0)
            {
                throw new IllegalStateException(
                    ".reconnect() requires a positive 'reconnectAttempts'");
            }
            if (consumers.isEmpty())
            {
                throw new IllegalStateException("need to add some callback, use .addCallback()");
            }
            if (username == null && apiKey == null)
            {
                throw new IllegalStateException(
                    "need to add some authorization mechanism, use .basicAuth(...) or .oauth(...)");
            }

            return new StreamClient(mEndpoint, username, password, apiKey, apiSecret, token,
                tokenSecret, apiEndpoint, sslConfiguration, consumers, beforeReconnect,
                afterReconnect, reconnect, reconnectAttempts, pauseBeforeReconnectInSeconds,
                filters);
        }
    }

    public static interface SSLConfiguration
    {
        /**
         * Provides the SSLContext to be used in the SSL sessions.
         */
        public SSLContext sslContext();

        /**
         * Provides the hostname verifier to be used in the SSL sessions.
         */
        public HostnameVerifier hostnameVerifier();
    }

    public static class AbiquoAPIOAuth extends OAuthSignatureCalculator
    {
        private final String endpointApi;

        public AbiquoAPIOAuth(final ConsumerKey consumerAuth, final RequestToken userAuth,
            final String endpointApi)
        {
            super(consumerAuth, userAuth);
            this.endpointApi = endpointApi + "/login";
        }

        @Override
        public String calculateSignature(final String method, final Uri uri,
            final long oauthTimestamp, final String nonce, final List<Param> formParams,
            final List<Param> queryParams)
        {
            return super.calculateSignature(method, Uri.create(endpointApi), oauthTimestamp, nonce,
                formParams, Collections.singletonList(new Param("expand", "privileges")));
        }
    }

    public static class Traces
    {
        List<Trace> collection = new ArrayList<>();

        public List<Trace> getCollection()
        {
            return collection;
        }

        public void setCollection(final List<Trace> collection)
        {
            this.collection = collection;
        }
    }
}
