/*
 * Decompiled with CFR 0.152.
 */
package com.eventstore.dbclient;

import com.eventstore.dbclient.ConnectionShutdownException;
import com.eventstore.dbclient.Endpoint;
import com.eventstore.dbclient.EventStoreDBClientSettings;
import com.eventstore.dbclient.NotLeaderException;
import com.eventstore.dbclient.OptionsBase;
import com.eventstore.dbclient.ServerFeatures;
import com.eventstore.dbclient.ServerInfo;
import com.eventstore.dbclient.ServerVersion;
import io.grpc.ManagedChannel;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.SslContext;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class GrpcClient {
    private static final int MAX_INBOUND_MESSAGE_LENGTH = 0x1100000;
    protected final EventStoreDBClientSettings settings;
    protected final SslContext sslContext;
    private final Logger logger = LoggerFactory.getLogger(GrpcClient.class);
    private final LinkedBlockingQueue<Msg> messages;
    protected ManagedChannel channel;
    protected Endpoint endpoint;
    protected Exception lastException;
    protected UUID currentChannelId;
    protected Optional<ServerInfo> serverInfo = Optional.empty();
    protected volatile boolean shutdownRequested = false;
    protected volatile boolean shutdown = false;

    protected GrpcClient(EventStoreDBClientSettings settings, SslContext sslContext) {
        this.settings = settings;
        this.sslContext = sslContext;
        this.messages = new LinkedBlockingQueue();
        this.currentChannelId = UUID.randomUUID();
    }

    protected void startConnectionLoop() {
        this.pushMsg(new CreateChannel(this.currentChannelId));
        CompletableFuture.runAsync(this::messageLoop, this.createConnectionLoopExecutor());
    }

    protected Executor createConnectionLoopExecutor() {
        return Executors.newSingleThreadExecutor(r -> {
            Thread thread = new Thread(r, "esdb-client-" + this.currentChannelId);
            thread.setDaemon(true);
            return thread;
        });
    }

    protected abstract boolean doConnect();

    protected void pushMsg(Msg msg) {
        try {
            if (this.shutdown) {
                if (msg instanceof RunWorkItem) {
                    RunWorkItem args = (RunWorkItem)msg;
                    args.reportError(new ConnectionShutdownException());
                }
                if (msg instanceof Shutdown) {
                    ((Shutdown)msg).completed.accept(42);
                }
                return;
            }
            this.messages.put(msg);
            this.logger.debug("Scheduled msg: {}", (Object)msg);
        }
        catch (InterruptedException e) {
            this.logger.error("Unexpected exception occurred when pushing a new message", (Throwable)e);
        }
    }

    public EventStoreDBClientSettings getSettings() {
        return this.settings;
    }

    public <A> CompletableFuture<A> runWithArgs(Function<WorkItemArgs, CompletableFuture<A>> action) {
        CompletableFuture result = new CompletableFuture();
        GrpcClient self = this;
        String msgId = UUID.randomUUID().toString();
        this.pushMsg(new RunWorkItem(msgId, (args, fatalError) -> {
            if (fatalError != null) {
                result.completeExceptionally(fatalError);
                return;
            }
            ((CompletableFuture)action.apply(args)).whenComplete((outcome, error) -> {
                StatusRuntimeException ex;
                if (outcome != null) {
                    result.complete(outcome);
                    return;
                }
                if (error instanceof NotLeaderException) {
                    NotLeaderException ex2 = (NotLeaderException)error;
                    result.completeExceptionally(ex2);
                    self.pushMsg(new CreateChannel(args.id, ex2.getLeaderEndpoint()));
                    return;
                }
                if (error instanceof StatusRuntimeException && (ex = (StatusRuntimeException)((Object)((Object)error))).getStatus().getCode().equals((Object)Status.Code.UNAVAILABLE)) {
                    self.pushMsg(new CreateChannel(args.id));
                }
                this.logger.debug("RunWorkItem[{}] completed exceptionally: {}", (Object)msgId, (Object)error.toString());
                result.completeExceptionally((Throwable)error);
            });
        }));
        return result;
    }

    public <A> CompletableFuture<A> run(Function<ManagedChannel, CompletableFuture<A>> action) {
        return this.runWithArgs(args -> (CompletableFuture)action.apply(args.getChannel()));
    }

    public CompletableFuture<Endpoint> getCurrentEndpoint() {
        return this.runWithArgs(args -> CompletableFuture.completedFuture(((WorkItemArgs)args).endpoint));
    }

    private boolean discover(UUID previousId, Optional<Endpoint> candidate) {
        long attempts = 1L;
        if (!this.currentChannelId.equals(previousId)) {
            this.logger.debug("Skipping connection attempt as new connection to endpoint [{}] has already been created.", (Object)this.endpoint);
            return true;
        }
        if (candidate.isPresent()) {
            this.closeChannel();
            this.endpoint = candidate.get();
            this.channel = this.createChannel(this.endpoint);
            this.logger.debug("Prepared channel to proposed leader candidate [{}]", (Object)this.endpoint);
            try {
                if (this.loadServerFeatures()) {
                    this.currentChannelId = UUID.randomUUID();
                    this.logger.info("Connection to proposed leader candidate [{}] created successfully", (Object)this.endpoint);
                    return true;
                }
            }
            catch (Exception e) {
                this.logger.error("A fatal exception happened when fetching server supported features", (Throwable)e);
            }
            this.logger.warn("Failed connection to proposed leader candidate [{}]. Retrying with default leader discovery.", (Object)this.endpoint);
        }
        while (true) {
            this.logger.debug("Start connection attempt ({}/{})", (Object)attempts, (Object)this.settings.getMaxDiscoverAttempts());
            this.closeChannel();
            if (this.doConnect()) {
                this.logger.debug("Prepared channel to endpoint [{}]", (Object)this.endpoint);
                try {
                    if (this.loadServerFeatures()) {
                        this.currentChannelId = UUID.randomUUID();
                        this.logger.info("Connection to endpoint [{}] created successfully", (Object)this.endpoint);
                        return true;
                    }
                }
                catch (Exception e) {
                    this.logger.error("A fatal exception happened when fetching server supported features from endpoint [{}]", (Object)this.endpoint, (Object)e);
                    return false;
                }
            }
            if (++attempts > (long)this.settings.getMaxDiscoverAttempts()) {
                this.logger.error("Maximum discovery attempt count reached: {}", (Object)this.settings.getMaxDiscoverAttempts());
                return false;
            }
            this.logger.warn("Unable to find a node. Retrying... ({}/{})", (Object)attempts, (Object)this.settings.getMaxDiscoverAttempts());
            this.sleep(this.settings.getDiscoveryInterval());
        }
    }

    private boolean loadServerFeatures() {
        try {
            this.serverInfo = ServerFeatures.getSupportedFeatures(this.settings, this.channel);
            return true;
        }
        catch (ServerFeatures.RetryableException e) {
            this.logger.warn("An exception happened when fetching server supported features. Retrying connection attempt.", (Throwable)e);
            return false;
        }
    }

    public CompletableFuture<Optional<ServerVersion>> getServerVersion() {
        return this.runWithArgs(args -> CompletableFuture.completedFuture(((WorkItemArgs)args).info.map(ServerInfo::getServerVersion)));
    }

    protected void sleep(long millis) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException e) {
            this.logger.error("Thread is interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private boolean handleMsg(Msg msg) {
        boolean result = false;
        if (msg instanceof CreateChannel) {
            if (!this.shutdown) {
                CreateChannel args = (CreateChannel)msg;
                result = this.discover(args.previousId, args.channel);
            } else {
                this.logger.warn("Channel creation request ignored, the connection to endpoint [{}] is already closed", (Object)this.endpoint);
            }
        } else if (msg instanceof RunWorkItem) {
            RunWorkItem args = (RunWorkItem)msg;
            if (this.shutdown) {
                Exception e = this.lastException != null ? this.lastException : new ConnectionShutdownException();
                this.logger.warn("Receive an command request but the connection to endpoint [{}] is already closed", (Object)this.endpoint, (Object)e);
                args.item.accept(null, e);
            } else if (this.channel == null) {
                try {
                    this.messages.put(new RunWorkItem(args.msgId, args.item));
                    this.logger.debug("Channel is not resolved yet, parking current work item");
                }
                catch (InterruptedException e) {
                    this.logger.error("Exception occurred when parking a work item", (Throwable)e);
                    args.item.accept(null, e);
                }
            } else {
                WorkItemArgs workItemArgs = new WorkItemArgs(this.currentChannelId, this.channel, this.endpoint, this.serverInfo);
                args.item.accept(workItemArgs, null);
            }
            result = true;
        } else if (msg instanceof Shutdown) {
            if (!this.shutdown) {
                this.logger.info("Received a shutdown request, closing connection to endpoint [{}]", (Object)this.endpoint);
                this.closeChannel();
                result = false;
                this.logger.info("Connection to endpoint [{}] was closed successfully", (Object)this.endpoint);
            } else {
                ((Shutdown)msg).completed.accept(42);
                this.logger.info("Shutdown request ignored, connection is already closed");
            }
        }
        return result;
    }

    private void messageLoop() {
        Consumer<Object> shutdownCompleted = null;
        try {
            Msg msg;
            do {
                msg = this.messages.take();
                this.logger.debug("Current msg: {}", (Object)msg);
            } while (this.handleMsg(msg));
            this.shutdown = true;
            if (msg instanceof Shutdown) {
                shutdownCompleted = ((Shutdown)msg).completed;
            }
        }
        catch (InterruptedException e) {
            this.lastException = e;
            this.shutdown = true;
        }
        this.logger.debug("Client has been shutdown. Draining pending requests...");
        ArrayList msgs = new ArrayList();
        this.messages.drainTo(msgs);
        for (Msg msg : msgs) {
            this.handleMsg(msg);
        }
        if (shutdownCompleted != null) {
            shutdownCompleted.accept(42);
        }
        this.logger.debug("Drainage completed successfully");
    }

    private void closeChannel() {
        if (this.channel != null) {
            try {
                this.logger.trace("Shutting down existing gRPC channel [{}]", (Object)this.channel);
                boolean terminated = this.channel.shutdown().awaitTermination(5L, TimeUnit.SECONDS);
                if (!terminated) {
                    this.channel.shutdownNow().awaitTermination(5L, TimeUnit.SECONDS);
                }
                this.logger.trace("Successful shutdown of gRPC channel [{}]", (Object)this.channel);
            }
            catch (InterruptedException e) {
                this.logger.error("Error when closing gRPC channel", (Throwable)e);
            }
            finally {
                this.channel = null;
            }
        }
    }

    protected ManagedChannel createChannel(Endpoint endpoint) {
        NettyChannelBuilder builder = NettyChannelBuilder.forAddress((String)endpoint.getHostname(), (int)endpoint.getPort()).maxInboundMessageSize(0x1100000);
        if (this.sslContext == null) {
            builder.usePlaintext();
        } else {
            builder.sslContext(this.sslContext);
        }
        if (this.settings.getKeepAliveTimeout() <= 0L) {
            builder.keepAliveTimeout(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
        } else {
            builder.keepAliveTimeout(this.settings.getKeepAliveTimeout(), TimeUnit.MILLISECONDS);
        }
        if (this.settings.getKeepAliveInterval() <= 0L) {
            builder.keepAliveTime(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
        } else {
            builder.keepAliveTime(this.settings.getKeepAliveInterval(), TimeUnit.MILLISECONDS);
        }
        return builder.build();
    }

    public void shutdown() throws ExecutionException, InterruptedException {
        this.shutdownRequested = true;
        CompletableFuture completion = new CompletableFuture();
        this.pushMsg(new Shutdown(completion::complete));
        completion.get();
    }

    public boolean isShutdown() {
        return this.shutdown || this.shutdownRequested;
    }

    static class Shutdown
    implements Msg {
        final Consumer<Object> completed;

        Shutdown(Consumer<Object> completed) {
            this.completed = completed;
        }

        public String toString() {
            return "Shutdown";
        }
    }

    static class RunWorkItem
    implements Msg {
        final String msgId;
        final WorkItem item;

        RunWorkItem(String msgId, WorkItem item) {
            this.msgId = msgId;
            this.item = item;
        }

        void reportError(Exception e) {
            this.item.accept(null, e);
        }

        public String toString() {
            return "RunWorkItem[" + this.msgId + "]";
        }
    }

    static class CreateChannel
    implements Msg {
        final Optional<Endpoint> channel;
        final UUID previousId;

        CreateChannel(UUID previousId) {
            this.channel = Optional.empty();
            this.previousId = previousId;
        }

        CreateChannel(UUID previousId, Endpoint endpoint) {
            this.channel = Optional.of(endpoint);
            this.previousId = previousId;
        }

        public String toString() {
            return new StringJoiner(", ", CreateChannel.class.getSimpleName() + "[", "]").add("endpoint=" + this.channel.map(Endpoint::toString).orElse("NOT_SET")).toString();
        }
    }

    private static interface Msg {
    }

    @FunctionalInterface
    static interface WorkItem {
        public void accept(WorkItemArgs var1, Exception var2);
    }

    class WorkItemArgs {
        private final UUID id;
        private final ManagedChannel channel;
        private final Endpoint endpoint;
        private final Optional<ServerInfo> info;

        public WorkItemArgs(UUID id, ManagedChannel channel, Endpoint endpoint, Optional<ServerInfo> info) {
            this.id = id;
            this.channel = channel;
            this.endpoint = endpoint;
            this.info = info;
        }

        public UUID getId() {
            return this.id;
        }

        public ManagedChannel getChannel() {
            return this.channel;
        }

        public Endpoint getEndpoint() {
            return this.endpoint;
        }

        public boolean supportFeature(int feature) {
            return this.info.map(value -> value.supportFeature(feature)).orElse(false);
        }

        public <A> HttpURLConnection getHttpConnection(OptionsBase<A> options, EventStoreDBClientSettings settings, String path) {
            try {
                HttpURLConnection conn = (HttpURLConnection)this.endpoint.getURL(settings.isTls(), path).openConnection();
                conn.setRequestProperty("Accept", "application/json");
                String creds = options.getHttpCredentialString();
                if (creds == null && settings.getDefaultCredentials() != null) {
                    creds = settings.getDefaultCredentials().basicAuthHeader();
                }
                if (creds != null) {
                    conn.setRequestProperty("Authorization", creds);
                }
                return conn;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}

