/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client;

import org.apache.http.HttpStatus;

import com.ibm.optim.oaas.client.impl.MessageHandler;

/**
 * Base class for OaaS runtime exceptions.
 */
public class OaasRuntimeException extends RuntimeException {
  /**
   * Default.
   */
  private static final long serialVersionUID = 1L;

  private final Object[] parameters;
  private MessageHandler messageHandler;

  /**
   * Creates an exception.
   * 
   * @param message
   *            The exception message.
   * @param cause
   *            The cause of the exception.
   * @param parameters
   *            The optional parameters for the message.
   */
  public OaasRuntimeException(String message, Throwable cause, Object... parameters) {
    super(message, cause);
    this.parameters = parameters;
  }

  /**
   * Creates an exception.
   * 
   * @param message
   *            The exception message.
   * @param parameters
   *            The optional parameters for the message.
   */
  public OaasRuntimeException(String message, Object... parameters) {
    super(message);
    this.parameters = parameters;
  }

  /**
   * Creates an exception.
   * 
   * @param cause
   *            The cause of the exception.
   */
  public OaasRuntimeException(Throwable cause) {
    super(cause);
    this.parameters = null;
  }

  /**
   * Gets the parameters, if any, that were set in the constructor.
   * 
   * @return The parameters.
   */
  public Object[] getParameters() {
    return parameters;
  }

  /**
   * Gets the message handler.
   * 
   * @return The message handler or <code>null</code> if no bundle name is
   *         set.
   */
  private synchronized MessageHandler getMessageHandler() {
    if (messageHandler == null) {
      String bundle = getResourceBundleName();
      if (bundle != null) {
        // by default we create the message handler in safe mode
        messageHandler = new MessageHandler(bundle, Boolean.TRUE, getClassLoader());
      }
    }
    return messageHandler;
  }

  /**
   * Gets the localized message using a given locale.
   * 
   * @param localeName
   *            The name of the locale for the message.
   * @return The localized message.
   */
  public String getLocalizedMessage(String localeName) {
    MessageHandler handler = getMessageHandler();
    if (handler != null) {
      return handler.getLocalizedMessageWithLocale(localeName, super.getMessage(), getParameters());
    }
    return super.getMessage();
  }

  /**
   * Gets the localized message.
   * 
   * @return The localized message using the default locale.
   */
  @Override
  public String getLocalizedMessage() {
    return getLocalizedMessage(null);
  }

  /**
   * Uses localized messages if you have them.
   */
  @Override
  public String getMessage() {
    return getLocalizedMessage();
  }

  /**
   * This must be overridden in order to provide localized messages.
   * <p>
   * Return the resource bundle name used to create the resource bundle (if
   * any) when localizing messages. By default this method returns
   * <code>null</code>.
   * </p>
   * 
   * @return The resource bundle name or <code>null</code> if no bundles.
   */
  protected String getResourceBundleName() {
    return null;
  }

  /**
   * This can be overridden to provide the class loader that will be used to
   * look up the resource bundle.
   * <p>
   * By default this returns the class loader of the exception class.
   * </p>
   * 
   * @return The class loader used to look up the resource bundle.
   */
  protected ClassLoader getClassLoader() {
    return this.getClass().getClassLoader();
  }

  /**
   * Gets the real (deepest) root cause of an exception.
   * 
   * @param throwable
   *            the exception.
   * @return The deepest non-null exception (cause).
   */
  public static Throwable getRootCause(Throwable throwable) {
    Throwable deepestError = throwable;
    while ((deepestError.getCause() != null)
        && (deepestError != deepestError.getCause())) {
      deepestError = deepestError.getCause();
    }
    return deepestError;
  }

  /**
   * Gets the REST status code. This method must be overridden to provide the
   * appropriate value.
   * 
   * @return The status code.
   */
  public int getRestStatusCode() {
    return HttpStatus.SC_INTERNAL_SERVER_ERROR;
  }

  /**
   * Returns the message code.
   * 
   * @return The message code.
   */
  public String getMessageCode() {
    return super.getMessage();
  }
}