/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client;

import com.ibm.optim.oaas.client.impl.ClientMessageCodes;

/**
 * Exception raised when accessing a remote resource using the HTTP protocol.
 *
 */
public class OperationException extends OaasException {
  private static final long serialVersionUID = 1L;

  private static final String BUNDLE_NAME = ClientMessageCodes.class.getName();

  private String _uri;
  private int _code;
  private String _operation;
  private ServerExceptionData _restException;

  /**
   * Creates a client exception.
   * 
   * @param operation
   *            The HTTP operation.
   * @param uri
   *            The accessed URI.
   * @param code
   *            The code returned by the server.
   */
  public OperationException(String operation, String uri, int code) {
    super(ClientMessageCodes.AKCJC5001E_OPERATION_EXCEPTION.name(), code,
          operation, uri);
    _uri = uri;
    _code = code;
    _operation = operation;
  }

  /**
   * Creates a client exception with additional server information
   * 
   * @param operation
   *            The HTTP operation.
   * @param uri
   *            The accessed URI.
   * @param code
   *            The code returned by the server.
   * @param data
   *            Additional data sent by the server.
   */
  public OperationException(String operation, String uri, int code,
                            ServerExceptionData data) {
    super(ClientMessageCodes.AKCJC5001E_OPERATION_EXCEPTION.name(), code,
          operation, uri);
    _uri = uri;
    _code = code;
    _operation = operation;
    _restException = data;
  }

  /**
   * Creates a client exception with another exception as the source.
   * 
   * @param operation
   *            The HTTP operation.
   * @param uri
   *            The accessed URI.
   * @param code
   *            The code returned by the server.
   * @param e
   *            The source exception.
   */
  public OperationException(String operation, String uri, int code,
                            Exception e) {
    super(ClientMessageCodes.AKCJC5001E_OPERATION_EXCEPTION.name(), e,
          code, operation, uri);
    _uri = uri;
    _code = code;
    _operation = operation;
  }

  public OperationException(String operation, String uri, int code,
                            String gatewayMessage) {
    super(ClientMessageCodes.AKCJC5010E_OPERATION_GATEWAY_EXCEPTION.name(), 
          code, operation, uri, gatewayMessage);
    _uri = uri;
    _code = code;
    _operation = operation;
  }

  /**
   * Returns the server URI if defined.
   * 
   * @return The server URI or <code>null</code> if not defined.
   */
  public String getURI() {
    return _uri;
  }

  /**
   * Returns the server returned code if defined.
   * 
   * @return The code or <code>0</code> if not defined.
   */
  public int getCode() {
    return _code;
  }

  /**
   * Returns the operations if defined. The operations are HTTP operations
   * such as GET, PUT, POST, DELETE.
   * 
   * @return The operation or <code>null</code> if not defined.
   */
  public String getOperation() {
    return _operation;
  }

  /**
   * Returns the server side data returned for this exception if defined
   * 
   * @return The server exception data of <code>null</code> if not defined.
   */
  public ServerExceptionData getServerExceptionData() {
    return _restException;
  }

  @Override
  protected String getResourceBundleName() {
    return BUNDLE_NAME;
  }
}