/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.http;

import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;

/**
 * Factory class to create HTTP clients. HTTP clients can be created directly
 * with the HTTP client builder, but this factory offers some useful shortcuts
 * and predefined configuration.
 * 
 * @see org.apache.http.impl.client.HttpClients
 */
public class HttpClientFactory {
  /**
   * The default socket timeout.
   */
  public static final int DEFAULT_SOCKET_TIMEOUT = 120000;

  /**
   * The default connection timeout.
   */
  public static final int DEFAULT_CONNECTION_TIMEOUT = 120000;

  /**
   * The default maximum number of connections.
   */
  public static final int DEFAULT_MAX_CONNECTIONS = 30;

  /**
   * Creates a standard HTTP client with the default socket timeout,
   * connection timeout and maximum connections.
   * 
   * @return The HTTP client.
   */
  public static CloseableHttpClient createDefault() {
    return create(DEFAULT_CONNECTION_TIMEOUT, DEFAULT_MAX_CONNECTIONS, DEFAULT_SOCKET_TIMEOUT);
  }

  /**
   * Creates a standard HTTP client with specific values of socket timeout,
   * connection timeout and maximum number of connections.
   * 
   * @param connectionTimeout
   *            The connection timeout.
   * @param maxConnections
   *            The maximum number of connections.
   * @param socketTimeout
   *            The socket timeout.
   * @return The HTTP client.
   */
  public static CloseableHttpClient create(int connectionTimeout,
                                           int maxConnections,
                                           int socketTimeout) {
    RequestConfig requestConfig = RequestConfig.custom()
        .setSocketTimeout(socketTimeout)
        .setConnectTimeout(connectionTimeout).build();
    CloseableHttpClient client = HttpClients.custom()
        .setMaxConnTotal(maxConnections)
        .setMaxConnPerRoute(maxConnections)
        .setSSLSocketFactory(SNISSLConnectionSocketFactory.getSSLSocketFactory())
        .setDefaultRequestConfig(requestConfig).build();
    return client;
  }

  /**
   * Creates a builder to configure a custom HTTP client.
   * 
   * @return The builder.
   */
  public static HttpClientBuilder custom() {
    return HttpClients.custom();
  }
}