/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.http;

import java.util.concurrent.TimeUnit;

import org.apache.http.impl.client.CloseableHttpClient;

/**
 * This monitor thread can be used to evict expired and idle connections.
 *
 */
public class HttpIdleConnectionMonitorThread extends Thread {
  /**
   * Default idle timeout in seconds.
   */
  public static final int DEFAULT_IDLE_TIMEOUT = 120;

  /**
   * Default time interval to check for expired and idle connections.
   */
  public static final int DEFAULT_INTERVAL = 5;

  private final CloseableHttpClient client;
  private volatile boolean shutdown;
  private int interval = DEFAULT_INTERVAL;
  private int idleTimeout = DEFAULT_IDLE_TIMEOUT;

  /**
   * Creates a monitor thread for the given client with default values.
   * 
   * @param client
   *            The HTTP client.
   */
  public HttpIdleConnectionMonitorThread(CloseableHttpClient client) {
    super();
    this.client = client;
  }

  /**
   * Creates a monitor thread for the given client with specific values.
   * 
   * @param client
   *            The HTTP client.
   * @param interval
   *            The time interval in seconds.
   * @param idleTimeout
   *            The idle timeout in seconds.
   */
  public HttpIdleConnectionMonitorThread(CloseableHttpClient client,
                                         int interval,
                                         int idleTimeout) {
    super();
    this.client = client;
    this.interval = interval;
    this.idleTimeout = idleTimeout;
  }

  @SuppressWarnings("deprecation")
  @Override
  public void run() {
    try {
      while (!shutdown) {
        synchronized (this) {
          wait(interval * 1000);
          client.getConnectionManager().closeExpiredConnections();
          client.getConnectionManager().closeIdleConnections(idleTimeout, TimeUnit.SECONDS);
        }
      }
    } catch (InterruptedException ex) {
      // ignore
    }
  }

  /**
   * Shutdown the monitor thread.
   */
  public void shutdown() {
    shutdown = true;
    synchronized (this) {
      notifyAll();
    }
  }
}