/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.http;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;

import org.apache.http.HttpHost;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.protocol.HttpContext;

/**
 * This class provides a socket factory to handle SNI (Server Name Identification)
 * with the IBM JDK and also perform some retries when it is difficult to connect.
 * The socket factory is automatically added when using the default HTTP client.
 * @see HttpClientFactory#createDefault()
 */
public final class SNISSLConnectionSocketFactory {
  private static Logger LOG = Logger.getLogger(SNISSLConnectionSocketFactory.class.getName());

  private static final SNISSLConnectionSocketFactory singleton = new SNISSLConnectionSocketFactory();

  private LayeredConnectionSocketFactory sslSocketFactory;

  private SNISSLConnectionSocketFactory() {
    SSLContext sslcontext = SSLContexts.createSystemDefault();
    sslSocketFactory = new SSLConnectionSocketFactory(sslcontext) {
      @Override
      public Socket connectSocket(final int connectTimeout,
                                  final Socket socket,
                                  final HttpHost host,
                                  final InetSocketAddress remoteAddress,
                                  final InetSocketAddress localAddress,
                                  final HttpContext context) throws IOException {
        if (socket instanceof SSLSocket) {
          try {
            // THIS ONLY WORKS FOR IBM JDK 7
            Method method = socket.getClass().getMethod("a", String.class);
            method.invoke(socket, host.getHostName());
          } catch (Exception e) {
            // if we are not with IBM JDK we just ignore
            LOG.log(Level.FINE,"not an IBM JDK 7, ignoring SNI workaround");
          }
        }

        // see RTC 25530: implement some retries
        ConnectException ex=null;
        for (int i = 1; i < 4; i++) {
          try {
            return super.connectSocket(connectTimeout, socket, host, remoteAddress, localAddress, context);
          } catch (ConnectException ex1) {
            ex=ex1;
            LOG.log(Level.WARNING, "Exception connecting to DoCloud: '{0}'. Retrying ({1})...", new Object[]{ex.getMessage(),i});
          }
        }
        throw ex;
      }
    };
  }

  public static LayeredConnectionSocketFactory getSSLSocketFactory() {
    return singleton.sslSocketFactory;
  }
}