/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.impl;

import java.util.Locale;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ibm.icu.text.MessageFormat;
import com.ibm.icu.util.ULocale;

/**
 * Simple class for handling localized messages and message prefixes.
 * <p>
 * Note that this class will never throw an exception for a missing bundle or
 * missing resource key if running in safe mode. In this case it is the caller's
 * responsibility to ensure that the bundle is on the classpath and that the
 * resource key exists in the bundle.
 * </p>
 * <p>
 * If the message prefix ({@link #MESSAGE_PREFIX}) exists in the resource bundle
 * then it will be prefixed to the message.
 * </p>
 * <p>
 * If the message code ({@link #KEY_CODE_SUFFIX}) exists in the resource bundle
 * then it will also be prefixed to the message. The {@link #KEY_CODE_SUFFIX
 * code suffix} will be looked up as the resource key plus the
 * {@link #KEY_CODE_SUFFIX}.
 * </p>
 * <p>
 * This class is thread-safe.
 * </p>
 * 
 * @author Julian Payne
 */
public class MessageHandler {
  /**
   * This is the property ({@value}) that should exist in the resource bundle
   * if the message prefix is to be used with the messages in this resource
   * bundle.
   */
  public static final String MESSAGE_PREFIX = "message.prefix";

  /**
   * This is the property ({@value}) that will be appended to the resource
   * key in order to get the message code for this resource key.
   */
  public static final String KEY_CODE_SUFFIX = ".code";

  /**
   * This is the property ({@value}) that will be appended to the resource
   * key in order to get the message description for this resource key. This
   * description is only used when building the tables of message codes and is
   * not displayed in the localized message.
   */
  public static final String KEY_DESCRIPTION_SUFFIX = ".description";

  private static final Logger logger = Logger.getLogger(MessageHandler.class.getName());

  private final String bundleName;
  private final boolean safeMode;
  private final ClassLoader loader;

  /**
   * Create an instance of a message handler for a resource bundle in safe
   * mode.
   * 
   * @param bundleName
   *            The bundle name.
   */
  public MessageHandler(String bundleName) {
    this(bundleName, true, null);
  }

  /**
   * Create an instance of a message handler for a resource bundle in safe
   * mode with a specific class loader to load bundle
   * 
   * @param bundleName
   *            The bundle name.
   */
  public MessageHandler(String bundleName, ClassLoader loader) {
    this(bundleName, true, loader);
  }

  /**
   * Create an instance of a message handler for a resource bundle with a
   * specific class loader to load bundle
   * 
   * @param bundleName
   *            The bundle name.
   * @param safeMode
   *            If <code>true</code> then no exceptions are thrown if the
   *            resource bundle is not found or if a key is not found in the
   *            bundle.
   * @param loader
   *            Specific class loader to load bundle.
   */
  public MessageHandler(String bundleName, boolean safeMode,
                        ClassLoader loader) {
    this.bundleName = bundleName;
    this.safeMode = safeMode;
    this.loader = loader;
  }

  private static ULocale getLocale(String localeName) {
    if ((localeName == null) || (localeName.trim().length() < 1)) {
      return ULocale.getDefault();
    } else {
      try {
        return new ULocale(localeName.trim());
      } catch (Exception e) {
        logger.log(Level.SEVERE, "Failed to create locale "
            + localeName + " : " + e.getLocalizedMessage(), e);
      }
      // fallback to the default locale if we failed
      logger.log(Level.WARNING, "Use default locale "
          + ULocale.getDefault().getDisplayName());
      return ULocale.getDefault();
    }
  }

  private ResourceBundle getBundle(ULocale locale, String bundleName) {
    try {
      if (locale != null) {
        if (loader == null) {
          return ResourceBundle.getBundle(bundleName,
                                          locale.toLocale());
        } else {
          return ResourceBundle.getBundle(bundleName,
                                          locale.toLocale(), loader);
        }
      } else {
        if (loader == null) {
          return ResourceBundle.getBundle(bundleName);
        } else {
          return ResourceBundle.getBundle(bundleName,
                                          Locale.getDefault(), loader);
        }
      }
    } catch (RuntimeException e) {
      if (!safeMode) {
        throw e;
      }
      logger.log(Level.SEVERE, "Failed to find resource bundle: "
          + bundleName, e);
    }
    return null;
  }

  /**
   * Returns the resource bundle for a given locale.
   * 
   * @param localeName
   *            The name of the locale.
   * @return The resource bundle or <code>null</code> if not found.
   */
  public ResourceBundle getResourceBundle(String localeName) {
    ULocale locale = getLocale(localeName);
    return getBundle(locale, this.bundleName);
  }

  private String prefixMessage(ResourceBundle bundle, String key) {
    String value = bundle.getString(key);
    String prefix = null;
    try {
      prefix = bundle.getString(MESSAGE_PREFIX);
    } catch (Exception ignore) {
      // ignore
    }
    String code = null;
    try {
      code = bundle.getString(key + KEY_CODE_SUFFIX);
    } catch (Exception ignore) {
      // ignore
    }
    StringBuilder msg = new StringBuilder();
    if (prefix != null) {
      msg.append(prefix.trim());
    }
    if (code != null) {
      for (int i = code.trim().length(); i < 5; i++) {
        msg.append('0');
      }
      msg.append(code.trim());
    }
    if (msg.length() > 0) {
      msg.append(": ");
    }
    msg.append(value);
    return msg.toString();
  }

  /**
   * This will look up a localized message.
   * 
   * @param localeName
   *            The name of the locale to use when looking up the message.
   * @param key
   *            The resource key.
   * @param args
   *            The optional arguments.
   * @return The localized message.
   */
  public String getLocalizedMessageWithLocale(String localeName, String key,
                                              Object... args) {
    String value = null;
    ResourceBundle bundle = getResourceBundle(localeName);
    if (bundle != null) {
      try {
        value = prefixMessage(bundle, key);
      } catch (RuntimeException e) {
        if (!safeMode) {
          throw e;
        }
        logger.log(Level.WARNING, "Failed to find resource key: " + key); // ,
        // e);
      }
    }
    if (value == null) {
      StringBuilder v = new StringBuilder();
      if (bundle == null) {
        v.append('!');
        v.append(bundleName);
        v.append(": ");
      }
      v.append(key);
      if (args != null) {
        for (int i = 0; i < args.length; i++) {
          v.append(" {");
          v.append(i);
          v.append('}');
        }
      }
      value = v.toString();
    }
    if ((args != null) && (args.length > 0)) {
      return MessageFormat.format(value, args);
    } else {
      return value;
    }
  }

  /**
   * This will look up a localized message.
   * 
   * @param key
   *            The resource key.
   * @param args
   *            The optional arguments.
   * @return The localized message.
   */
  public String getLocalizedMessage(String key, Object... args) {
    return getLocalizedMessageWithLocale(null, key, args);
  }
}
