/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.impl;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.Format;
import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Describes a message code as defined by the LOG-R2 SWG Requirements.
 * {@link OaasMessageCode} subclasses must specify a
 * {@link MessageCodeConfiguration} annotation to describe their configuration
 * w.r.t prefix and ranges.
 */
public abstract class OaasMessageCode {
  /** An empty array of Object */
  private static final Object[] OBJECT_EMPTYARRAY = new Object[0];

  private String _name;
  private String _loggerName;
  private String _rbName;
  private CharSequence _prefixCache; // the cached prefix value for this code

  public OaasMessageCode() {
  }

  /**
   * @return The name for this message. It is composed of two parts, the
   *         message ID plus a suffix providing enough information to
   *         understand the code in its context. The format is
   *         CIVXXNNNNS_suffix, _suffix is optional but recommended.
   * @see http://wiki.francelab.fr.ibm.com/Lab/Optim/ODMSrnd/LoggingInODM
   */
  public final String name() {
    return _name;
  }

  /**
   * @return The logger name to use for this message.
   */
  public final String getLoggerName() {
    return _loggerName;
  }

  /**
   * @return The logger to use for this message.
   */
  public Logger getLogger() {
    return Logger.getLogger(getLoggerName());
  }

  /**
   * @return The resource bundle name to use to lookup values for this
   *         message. It defaults to the actual message code class name.
   */
  public final String getResourceBundleName() {
    return _rbName;
  }

  protected String getResourceValue() {
    return ResourceBundle.getBundle(getResourceBundleName()).getString(name());
  }

  /**
   * @return The message log level extracted from its ID.
   * @see http://wiki.francelab.fr.ibm.com/Lab/Optim/ODMSrnd/LoggingInODM
   * @y.exclude
   */
  public final Level getLevel() {
    Level level = Level.INFO;
    switch (name().charAt(9)) {
      case 'I':
        level = Level.INFO;
        break;
      case 'W':
        level = Level.WARNING;
        break;
      case 'E':
        level = Level.SEVERE;
        break;
      case 'F':
        level = Level.FINEST;
        break;
    }
    return level;
  }

  /**
   * Logs this message to its assigned logger. As a shortcut to {
   * {@link #log(Throwable, Object...)}, it will check if the last parameter
   * is an exception and if so it will handle it as if
   * {@link #log(Throwable, Object...)} was called.
   * 
   * @param parameters
   *            The list of parameters to use when formatting the message text.
   */
  public final void log(Object... parameters) {
    Throwable throwable = null;
    if (parameters != null && parameters.length > 0
        && parameters[parameters.length - 1] instanceof Throwable) {
      // log(Object ... ) may lead to someone doing log(param1, param2,
      // param3, throwable)
      // we will handle it as if it was log(throwable, param1, param2,
      // param3)
      // so help this guy to match our method signature.
      //
      // we can still pass all parameters to message format, at most it
      // will be one too many
      // parameters but this is not an issue with the messageformat class.
      throwable = (Throwable) parameters[parameters.length - 1];
    }
    log(throwable, parameters);
  }

  /**
   * Logs this message and links {@link Throwable} to the logger assigned to
   * the message.
   * 
   * @param throwable
   *            The throwable to be logged.
   */
  public final void log(Throwable throwable) {
    log(throwable, (Object[]) null);
  }

  /**
   * Logs this message and links {@link Throwable} to the logger assigned to
   * the message.
   * 
   * @param throwable
   *            The throwable to be logged.
   * @param parameters
   *            The list of parameters to use when formatting the message text.
   */
  public final void log(Throwable throwable, Object... parameters) {
    log(getLogger(),throwable,parameters);		
  }

  /**
   * Logs this message and links {@link Throwable} to the logger assigned to
   * the message.
   * @param logger
   *            The logger to use
   * @param throwable
   *            The throwable to be logged.
   * @param parameters
   *            The list of parameters to use when formatting the message text.
   */
  public final void log(Logger logger, Throwable throwable, Object... parameters) {
    Level level = getLevel();
    if (logger.isLoggable(level)) {
      if (parameters == null) {
        parameters = OBJECT_EMPTYARRAY;
      }
      StringBuffer sbMsg = formatMessage(getMessageFormat(parameters.length), parameters);
      if (throwable == null) {
        logger.log(level, sbMsg.toString()); // NOPMD
      } else {
        logger.log(level, sbMsg.toString(), throwable); // NOPMD
      }
    }
  }

  /**
   * Gets the message formatter for this code, defaulting to a string with as
   * many placeholders as there are parameters
   * 
   * @param parameters
   * @return
   */
  protected MessageFormat getMessageFormat(int parametersCount) {
    String resourceValue;
    try {
      // getResourceValue may throw a MissingResourceException among
      // others
      resourceValue = getResourceValue();
    } catch (MissingResourceException mre) {
      // The message is not a message key, use it as message format
      // directly
      resourceValue = name();
    }
    return getMessageFormat(resourceValue, parametersCount);
  }

  private MessageFormat getMessageFormat(String resourceValue,
                                         int parametersCount) {
    // get the message for this code, defaults to code if resource can't be
    // found
    MessageFormat messageFormat;
    try {
      messageFormat = new MessageFormat(resourceValue);
    } catch (Throwable th) {
      // Record the resource bundle in the message
      // in case we can't get a message format for any reason, we generate
      // a format that has enough
      // placeholders to get all the parameters values in the log
      messageFormat = generatePlaceholderFormatter(th, resourceValue,
                                                   parametersCount);
    }

    // always go through message formatter even if no parameters. It will
    // make the management of
    // resource bundles simpler w.r.t to NLS_MESSAGEFORMAT markers
    // Build the formatted message with its prefix
    return messageFormat;
  }

  /**
   * generate a format that has enough placeholders to get all the parameters
   * values in the log
   * 
   * @param cause
   *            A possible reason why the formatting did not work.
   * @param faultyFormat
   * @param paramCount
   * @return
   */
  private MessageFormat generatePlaceholderFormatter(Throwable cause,
                                                     String faultyFormat, int paramCount) {
    StringBuffer causeHint = new StringBuffer(cause.getMessage())
        .append(" [").append(faultyFormat).append("]: ");
    // we have to escape the cause message for any characters that have
    // special meaning for the
    // formatter, i.e. quotes and curly braces
    for (int i = 0; i < causeHint.length(); i++) {
      char cur = causeHint.charAt(i);
      switch (cur) {
        case '\'':
          // double the single quote
          causeHint.insert(i, '\'');
          i++;
          break;
        case '{':
        case '}':
          // put the brace within single quotes
          causeHint.insert(i, '\'');
          i += 2;
          causeHint.insert(i, '\'');
          break;
      }
    }
    causeHint.append('[').append(getResourceBundleName()).append(':')
    .append(name()).append("] ");

    // generate a format for n parameters
    for (int iP = 0; iP < paramCount; iP++) {
      causeHint.append("\"{").append(iP).append("}\"; ");
    }
    causeHint.setLength(causeHint.length() - 2);// remove last separator

    try {
      return new MessageFormat(causeHint.toString());
    } catch (Throwable th) {
      return new MessageFormat("Throwable occurred "
          + th.getClass().getCanonicalName()); // $$NON_NLS
    }
  }

  /**
   * Fail-safe method to append all parameters to a stringbuffer
   * 
   * @param sbMsg
   * @param parameters
   * @return The number of parameters handled.
   * 
   */
  private static int appendParameters(StringBuffer sbMsg, int startFromParam,
                                      Object... parameters) {
    if (parameters == null || parameters.length <= startFromParam) {
      // nothing to format
      return 0;
    }
    sbMsg.append(" [");
    for (int iP = startFromParam; iP < parameters.length; iP++) {
      sbMsg.append(iP).append('=');
      try {
        sbMsg.append(parameters[iP]); // this is null-safe, will call
        // toString() on the
        // parameter
      } catch (Throwable thParamToString) { // in case parameter cannot
        // toString() itself
        // not much that can be done here, just append the throwable
        // class name
        sbMsg.append(thParamToString.getClass().getCanonicalName());
      }
      sbMsg.append("; ");
    }
    // terminate
    sbMsg.setLength(sbMsg.length() - 2);
    sbMsg.append(']');
    return parameters.length;
  }

  /**
   * Format the message with appropriate prefix This method is protected
   * against exceptions during message formatting, and complements the format
   * if there are left over parameters
   * 
   * @param messageFormat
   * @param parameters
   * @return
   */
  protected StringBuffer formatMessage(final MessageFormat messageFormat,
                                       Object... parameters) {
    StringBuffer sbMsg;
    try {
      sbMsg = new StringBuffer(getLogMessagePrefix());
      messageFormat.format(parameters, sbMsg, null);

      Format[] fmts = messageFormat.getFormats();
      int fmtCount = fmts.length;
      // If there were extra parameters that have not been used, append
      // them at the end
      if (fmtCount < parameters.length) {
        appendParameters(sbMsg, fmtCount, parameters);
      }
    } catch (Throwable th) {
      // protect in case something wrong occurs during message formatting,
      // just list code and
      // parameters
      sbMsg = new StringBuffer(getLogMessagePrefix()).append(" ").append(messageFormat.toPattern());
      appendParameters(sbMsg, 0, parameters);
    }
    return sbMsg;
  }

  private CharSequence getLogMessagePrefix() {
    if (_prefixCache == null) {
      // note: no need to synchronize, no issue if
      // created in // at the very beginning
      _prefixCache = new StringBuilder(name().substring(0, 10)) .append(": ");
    }
    return _prefixCache;
  }

  private void logf(Level level, String fmtMsg, Object... params) {
    if (isLoggable(level)) {
      // When no parameters are passed, no need to format anyhow, we'll
      // use the raw format as message
      if (params != null && params.length > 0) {
        StringBuffer sbMsg;
        try {
          sbMsg = formatMessage(new MessageFormat(fmtMsg), params);
        } catch (Throwable th) {
          // in case we can't get a message format for any reason, we
          // append the parameters so that
          // they will still show in the log
          sbMsg = new StringBuffer(fmtMsg).append(": ");
          appendParameters(sbMsg, 0, params);
        }
        fmtMsg = sbMsg.toString();
      }
      getLogger().log(level, fmtMsg);
    }
  }

  /**
   * Log at level FINE, this method logs only for codes with F class
   * 
   * @param fmt
   *            The message format.
   * @param params
   *            When supplied, use fmt pattern to format.
   */
  public void fine(String fmt, Object... params) {
    logf(Level.FINE, fmt, params);
  }

  /**
   * Log at level FINEST, this method logs only for codes with F class
   * 
   * @param fmt
   * @param params
   */
  public void finest(String fmt, Object... params) {
    logf(Level.FINEST, fmt, params);
  }

  /**
   * Log at level fine, but with the message format from another MessageCode
   * 
   * @param code
   * @param params
   */
  public void fine(OaasMessageCode code, Object... params) {
    fine(code.getResourceValue(), params);
  }

  /**
   * front-end for the entering() logger
   * 
   * @param sourceClass
   * @param sourceMethod
   * @param params
   */
  public void entering(String sourceClass, String sourceMethod,
                       Object... params) {
    getLogger().entering(sourceClass, sourceMethod, params);
  }

  /**
   * Extract formatted message String for this code (to inject in exceptions)
   * 
   * @param parameters
   * @return
   */
  public String extractMessage(Object... parameters) {
    return formatMessage(getMessageFormat(parameters.length), parameters)
        .toString();
  }

  /**
   * Checks if this message code is loggable, both its logger and itself must
   * be enabled for the level
   * 
   * @param level
   * @return
   */
  public boolean isLoggable(Level level) {
    return level.intValue() >= getLevel().intValue()
        && getLogger().isLoggable(level);
  }

  /**
   * Initializes all static fields from the given message code class.<BR>
   * When it encounters a field of type <code>MessageCode</code>, which is
   * null, it initializes this field to a new instance of the leaf-most
   * subclass of clazz or field.getType(), and set the field name, loggerName
   * and resourceBundleName.<BR>
   * If the field is not null, it sets its name, loggerName and
   * resourceBundleName if they are null.
   */
  public static void initializeCodes(Class<? extends OaasMessageCode> clazz,
                                     String loggerName, String resourceBundleName) {
    for (Field field : clazz.getDeclaredFields()) {
      Class<?> fieldType = field.getType();
      if (Modifier.isStatic(field.getModifiers())
          && OaasMessageCode.class.isAssignableFrom(fieldType)) {
        try {
          // Get the class variable's value
          OaasMessageCode code = (OaasMessageCode) field.get(null);

          // If not initialized yet, create an empty instance
          if (code == null) {
            code = (OaasMessageCode) (clazz
                .isAssignableFrom(fieldType) ? fieldType
                                             : clazz).newInstance();
          }

          // set the fields that are not initialized yet
          if (code._loggerName == null) {
            code._loggerName = loggerName;
          }
          if (code._rbName == null) {
            code._rbName = resourceBundleName;
          }
          if (code._name == null) {
            code._name = field.getName();
          }

          // set the class variable (initialize this field)
          field.set(null, code);

        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      }
    }
  }

  /**
   * Similar to {@link #initializeCodes(Class, String, String)} but defaulting
   * to the classname as resource bundle name.
   */
  public static void initializeCodes(Class<? extends OaasMessageCode> clazz, String loggerName) {
    initializeCodes(clazz, loggerName, clazz.getName());
  }
}