/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.impl;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.ParseException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.entity.GzipCompressingEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.FileEntity;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.ibm.optim.oaas.client.InternalException;
import com.ibm.optim.oaas.client.InvalidClientIdException;
import com.ibm.optim.oaas.client.InvalidParameterException;
import com.ibm.optim.oaas.client.OaasRuntimeException;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.ParsingException;
import com.ibm.optim.oaas.client.ResourceAccessException;
import com.ibm.optim.oaas.client.SecurityException;
import com.ibm.optim.oaas.client.ServerExceptionData;
import com.ibm.optim.oaas.client.job.UndefinedSubscriptionException;
import com.ibm.optim.oaas.client.job.impl.JobMessageCodes;

public class RestClient {
  private static final long GZIP_CONTENT_LENGTH_THRESHOLD = 512;

  private static final Logger logger = Logger.getLogger(RestClient.class.getName());
  private static final Level logLevel = Level.FINE;

  protected final String server_url; // the base server url
  private String authToken;

  private String clientid;
  private String secret;

  protected final CloseableHttpClient httpClient; // client to send http requests
  protected final ObjectMapper mapper; // to convert from/to json

  protected boolean compression = true;

  // can be used to ask for a retry on 413, for example set to 5 for a retry in 5 seconds
  protected int retryOn413Secs = 0;

  /**
   * Constructor.
   * 
   * @param url
   *          The server URL.
   * @param token
   *          Authorization token to access the server.
   */
  public RestClient(CloseableHttpClient client, String url, String token) {
    if (url != null) {
      url = url.trim();
    }
    // add ending slash if necessary
    if ((url != null) && !url.endsWith("/")) {
      url = url + "/";
    }
    this.server_url = url;

    this.httpClient = client;

    this.mapper = new ObjectMapper();

    this.authToken = token;
  }

  public RestClient(CloseableHttpClient client, String url, String clientid, String secret) {
    if (url != null) {
      url = url.trim();
    }
    // add ending slash if necessary
    if ((url != null) && !url.endsWith("/")) {
      url = url + "/";
    }
    this.server_url = url;

    this.httpClient = client;

    this.mapper = new ObjectMapper();

    this.clientid = clientid;
    this.secret = secret;
  }

  public String getBaseUrl() {
    return server_url;
  }

  public boolean isCompressionEnabled() {
    return compression;
  }

  public void setCompression(boolean compression) {
    this.compression = compression;
  }

  protected HttpClientContext getHttpContext() {
    HttpClientContext context = HttpClientContext.create();
    return context;
  }

  protected void setJsonMedia(HttpRequestBase request) {
    request.addHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_JSON);
    // seems APIm has issues with content-type set on DELETE
    // removing it as a DELETE has no body anyway (or should not)
    if (!request.getMethod().equalsIgnoreCase("DELETE")){
      request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);
    }
  }

  protected void setTextMedia(HttpRequestBase request) {
    request.addHeader(HttpHeaders.ACCEPT, MediaType.TEXT_PLAIN);
    request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.TEXT_PLAIN);
  }

  private void setOptionalCompressedEntity(final HttpEntityEnclosingRequestBase request,
                                           final HttpEntity entity) {
    if (compression && ((entity.getContentLength() < 0) || (entity.getContentLength() > GZIP_CONTENT_LENGTH_THRESHOLD))) {
      GzipCompressingEntity gzipped = new GzipCompressingEntity(entity);
      if (logger.isLoggable(logLevel)) {
        logger.log(logLevel, request.getURI().toASCIIString() + ": compressing json entity " + toString(gzipped));
      }
      request.setEntity(gzipped);
    } else {
      if (logger.isLoggable(logLevel)) {
        logger.log(logLevel, request.getURI().toASCIIString() + ": not compressing json entity " + toString(entity));
      }
      request.setEntity(entity);
    }
  }

  private String toString(HttpEntity entity) {
    ObjectNode obj = JsonNodeFactory.instance.objectNode();
    put(obj, entity.getContentEncoding());
    put(obj, entity.getContentType());
    return obj.toString();
  }

  protected void setJsonEntity(HttpEntityEnclosingRequestBase request, Object obj) throws OperationException {
    try {
      setOptionalCompressedEntity(request, new StringEntity(mapper.writeValueAsString(obj), "UTF-8"));
    }
    catch (Exception e) {
      throw new InternalException(e);
    }
  }

  protected void setTextEntity(HttpEntityEnclosingRequestBase request, String obj) throws OperationException {
    try {
      setOptionalCompressedEntity(request, new StringEntity(obj, "UTF-8"));
    }
    catch (Exception e) {
      throw new InternalException(e);
    }
  }

  protected String createResource(String path) {
    return server_url + path;
  }

  /**
   * Returns the id that is specified in the Location header. This is typically used after a
   * creation.
   * 
   * @param response
   * @return The object id or <code>null</code> if it does not exist.
   */
  protected String getIdFromLocation(CloseableHttpResponse response) {
    Header header = response.getFirstHeader(HttpHeaders.LOCATION);
    if (header != null) {
      String value = header.toString();
      int index = value.lastIndexOf("/");
      if (index != -1) {
        return value.substring(index + 1);
      }
    }
    return null;
  }

  /**
   * Handle the status when it is not accepted so that an exception is thrown
   * 
   * @param request
   *          The request.
   * @param response
   *          The response.
   * @throws OperationException
   */
  protected void handleStatus(HttpRequestBase request, CloseableHttpResponse response) throws OperationException {
    try {
      handleStatusImpl(request, response);
    } catch (OperationException e) {
      ServerExceptionData data = e.getServerExceptionData();
      if (data != null) {
        if (ClientMessageCodes.AKCJC5004E_PARSING_EXCEPTION.name().equals(data.getCode())) {
          throw new ParsingException(e, e.getServerExceptionData().getMessage());
        } else if (ClientMessageCodes.AKCJC5005E_INVALID_PARAMETER_EXCEPTION.name().equals(data.getCode())) {
          throw new InvalidParameterException(data.getParameters().toArray(), e);
        } else if (ClientMessageCodes.AKCJC5047E_INVALID_CLIENT_ID_EXCEPTION.name().equals(data.getCode())) {
          throw new InvalidClientIdException(data.getParameters().toArray(), e);
        } else if (ClientMessageCodes.AKCJC5040E_NOT_ALLOWED_EXCEPTION.name().equals(data.getCode())) {
          throw new ResourceAccessException(data.getParameters().toArray(), e);
        } else if (JobMessageCodes.AKCJC5110E_UNDEFINED_SUBSCRIPTION_EXCEPTION.name().equals(data.getCode())) {
          throw new UndefinedSubscriptionException(data.getParameters().toArray(), e);
        } else if (ClientMessageCodes.AKCJC5041E_TOKEN_NOT_PROVIDED.name().equals(data.getCode())) {
          throw new SecurityException(ClientMessageCodes.AKCJC5041E_TOKEN_NOT_PROVIDED.name(), e,
                                      data.getParameters().toArray());
        }
      }
      throw e;
    }
  }

  protected void handleStatusImpl(HttpRequestBase request, CloseableHttpResponse response) throws OperationException {
    // read the exception if possible
    String gatewayMessage = null;
    ServerExceptionData ex = null;
    Header exceptionHeader = response.getFirstHeader(ServerExceptionDataImpl.HTTP_X_EXCEPTION);
    boolean isException = exceptionHeader != null && !exceptionHeader.getValue().isEmpty()
        && exceptionHeader.getValue().equalsIgnoreCase("true") && response.getEntity() != null;
    if (isException) {
      try {
        ex = mapper.readValue(response.getEntity().getContent(), ServerExceptionDataImpl.class);

      } catch (Exception e) {// in case there is a parsing issue
        logger.warning(e.getMessage());
      }
    } else if (response.getEntity() != null) {
      String content=null;
      try {
        content = EntityUtils.toString(response.getEntity());
        if (content != null && content.length() != 0) {
          JsonNode json = mapper.readTree(content);
          if (json != null && json.get("moreInformation") != null) {
            gatewayMessage = json.get("moreInformation").asText();
          }
        }
      } catch (ParseException | IOException e) {
        // sometimes the content is not JSON, but a bad block of HTML...that contains the real error!
        // at least log it, so that we can have an idea
        logger.warning("Failed to parse json entity on error: " + e.getLocalizedMessage());
        logger.fine(content);
      }
    }

    if (gatewayMessage != null) {
      ClientMessageCodes.AKCJC5010E_OPERATION_GATEWAY_EXCEPTION.log(response.getStatusLine()
                                                                    .getStatusCode(), request.getMethod(), request.getURI().toString(), gatewayMessage);
      throw new OperationException(request.getMethod(), request.getURI().toString(), response
                                   .getStatusLine().getStatusCode(), gatewayMessage);
    }
    ClientMessageCodes.AKCJC5001E_OPERATION_EXCEPTION.log(response.getStatusLine().getStatusCode(),
                                                          request.getMethod(), request.getURI().toString());
    throw new OperationException(request.getMethod(), request.getURI().toString(), response
                                 .getStatusLine().getStatusCode(), ex);

  }

  protected String buildURI(String uri, Object... params) throws OperationException {
    try {
      URIBuilder builder = new URIBuilder(uri);
      for (int i = 0; i + 1 < params.length; i += 2) {
        if (params[i + 1] != null) {
          builder.addParameter(params[i].toString(), params[i + 1].toString());
        }
      }
      return builder.build().toString();
    } catch (Exception e) {
      throw new InternalException(e);
    }
  }

  protected <E> E getList(String uri, TypeReference<E> type) throws OperationException {
    HttpGet request = new HttpGet(uri);
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          if (response.getEntity() != null) {
            return mapper.readValue(response.getEntity().getContent(), type);
          } else {
            return null;
          }
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected <E> E getList(String uri, TypeReference<E> type, E defaultValue) throws OperationException {
    HttpGet request = new HttpGet(uri);
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          if (response.getEntity() != null) {
            return mapper.readValue(response.getEntity().getContent(), type);
          } else {
            return null;
          }
        case HttpStatus.SC_NO_CONTENT:
          return defaultValue;
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected <E> E get(String uri, TypeReference<E> type, boolean mustExists) throws OperationException {
    HttpGet request = new HttpGet(uri);
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          if (response.getEntity() != null) {
            return mapper.readValue(response.getEntity().getContent(), type);
          } else {
            return null;
          }
        case HttpStatus.SC_NOT_FOUND:
          if (!mustExists) {
            return null;
          }

        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected <E> E get(String uri, Class<E> type, boolean mustExists) throws OperationException {
    HttpGet request = new HttpGet(uri);
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          if (response.getEntity() != null) {
            return mapper.readValue(response.getEntity().getContent(), type);
          } else {
            return null;
          }
        case HttpStatus.SC_NO_CONTENT:
        case HttpStatus.SC_NOT_FOUND:
          if (!mustExists) {
            return null;
          }

        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected void put(String uri, Object obj) throws OperationException {
    HttpPut request = new HttpPut(uri);
    setJsonMedia(request);
    if (obj != null) {
      setJsonEntity(request, obj);
    }
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_NO_CONTENT:
          return;
        default:
          handleStatus(request, response);
          return;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected void putText(String uri, String obj) throws OperationException {
    HttpPut request = new HttpPut(uri);
    setTextMedia(request);
    if (obj != null) {
      setTextEntity(request, obj);
    }
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_NO_CONTENT:
          return;
        default:
          handleStatus(request, response);
          return;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected void post(String uri, Object obj) throws OperationException {
    HttpPost request = new HttpPost(uri);
    setJsonMedia(request);
    if (obj != null) {
      setJsonEntity(request, obj);
    }
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_NO_CONTENT:
          return;
        default:
          handleStatus(request, response);
          return;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected String createMultiPart(String uri, String [] name, String [] filename, boolean [] asJson, Object [] obj) throws OperationException {
    HttpPost request = new HttpPost(uri);
    try {
      MultipartEntityBuilder builder = MultipartEntityBuilder.create();
      builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
      for (int i = 0; i < name.length; i++) {
        if ( obj[i] != null) {
          if ( asJson[i]) {
            builder.addTextBody(name[i], mapper.writeValueAsString(obj[i]), ContentType.create("text/plain", Charset.forName("UTF-8")));
          }
          else if ( obj[i] instanceof File) {
            builder.addBinaryBody(name[i], (File)obj[i], ContentType.APPLICATION_OCTET_STREAM, filename[i]);
          }
          else if ( obj[i] instanceof InputStream) {
            builder.addBinaryBody(name[i], (InputStream)obj[i], ContentType.APPLICATION_OCTET_STREAM, filename[i]);
          }
        }
      }
      setOptionalCompressedEntity(request, builder.build());
      CloseableHttpResponse response = execute(request);
      try {
        switch (response.getStatusLine().getStatusCode()) {
          case HttpStatus.SC_CREATED:
            String id = getIdFromLocation(response);
            EntityUtils.consumeQuietly(response.getEntity());
            return id;
          default:
            handleStatus(request, response);
            return null;
        }
      }
      finally {
        response.close();
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected void postText(String uri, String obj) throws OperationException {
    HttpPost request = new HttpPost(uri);
    setTextMedia(request);
    if (obj != null) {
      setTextEntity(request, obj);
    }
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_NO_CONTENT:
          return;
        default:
          handleStatus(request, response);
          return;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected void throwOperationException(HttpUriRequest request, Exception e) throws OperationException {
    if (e instanceof OaasRuntimeException) {
      throw (OaasRuntimeException) e;
    }
    ClientMessageCodes.AKCJC5001E_OPERATION_EXCEPTION.log(0, request.getMethod(), request.getURI().toString());
    throw new OperationException(request.getMethod(), request.getURI().toString(), 0, e);
  }

  protected String create(String uri, Object obj) throws OperationException {
    HttpPost request = new HttpPost(uri);
    setJsonMedia(request);
    if (obj != null) {
      setJsonEntity(request, obj);
    }
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_CREATED:
          String id = getIdFromLocation(response);
          EntityUtils.consumeQuietly(response.getEntity());
          return id;
        case HttpStatus.SC_REQUEST_TOO_LONG:
          // https://tools.ietf.org/html/rfc7231#page-61
          // The 413 (Payload Too Large) status code indicates that the server is
          // refusing to process a request because the request payload is larger
          // than the server is willing or able to process.  The server MAY close
          // the connection to prevent the client from continuing the request.
          //
          // If the condition is temporary, the server SHOULD generate a
          // Retry-After header field to indicate that it is temporary and after
          // what time the client MAY try again.
          if (retryOn413Secs > 0) {
            // https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html
            //  The Retry-After response-header field can be used with a 503 (Service Unavailable)
            // response to indicate how long the service is expected to be unavailable to the requesting client.
            // This field MAY also be used with any 3xx (Redirection) response to indicate the minimum time the
            // user-agent is asked wait before issuing the redirected request. The value of this field can be
            // either an HTTP-date or an integer number of seconds (in decimal) after the time of the response.
            //
            // Retry-After  = "Retry-After" ":" ( HTTP-date | delta-seconds )
            //
            // Two examples of its use are
            //
            // Retry-After: Fri, 31 Dec 1999 23:59:59 GMT
            // Retry-After: 120
            //
            // In the latter example, the delay is 2 minutes. 
            response.addHeader("Retry-After", Integer.toString(retryOn413Secs));
          }
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  private String toString(HttpUriRequest request) {
    ObjectNode obj = JsonNodeFactory.instance.objectNode();
    obj.put("method", request.getMethod());
    obj.put("uri", request.getURI().toASCIIString());
    if (request instanceof HttpEntityEnclosingRequestBase) {
      ObjectNode e = JsonNodeFactory.instance.objectNode();
      HttpEntityEnclosingRequestBase req = (HttpEntityEnclosingRequestBase) request;
      if (req.getEntity() != null) {
        put(e, req.getEntity().getContentEncoding());
        put(e, req.getEntity().getContentType());
        obj.set("entity", e);
      }
    }
    for (Header header : request.getAllHeaders()) {
      put(obj, header);
    }
    return obj.toString();
  }

  private void put(ObjectNode obj, Header header) {
    if (header != null) {
      obj.put(header.getName(), header.getValue());
    }
  }

  protected void delete(String uri) throws OperationException {
    HttpDelete request = new HttpDelete(uri); //AppscanFindingExpected-RTC28711   
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
        case HttpStatus.SC_NO_CONTENT:
        case HttpStatus.SC_NOT_FOUND: // idempotent, must be ok
          return;
        default:
          handleStatus(request, response);
          return;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected DeletionStatus deleteWithStatus(String uri) throws OperationException {
    HttpDelete request = new HttpDelete(uri); //AppscanFindingExpected-RTC28711
    setJsonMedia(request);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_NO_CONTENT:
        case HttpStatus.SC_NOT_FOUND: // idempotent, must be ok
          return null;
        case HttpStatus.SC_OK:
          if (response.getEntity() != null) {
            return mapper.readValue(response.getEntity().getContent(), DeletionResponseImpl.class).getStatus();
          }
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected void getFile(String uri, File archive) throws IOException, OperationException {
    HttpGet request = new HttpGet(uri);
    request.addHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_OCTET_STREAM);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          OutputStream os = new BufferedOutputStream(new FileOutputStream(archive));
          InputStream is = null;
          try {
            is = response.getEntity().getContent();
            byte[] buffer = new byte[1024];
            int length;
            while ((length = is.read(buffer)) > 0) {
              os.write(buffer, 0, length);
            }
          } catch (EOFException e) {
            // premature end of file (in case of gzip empty content)
            // this is fine.
          } finally {
            if (is != null) {
              is.close();
            }
            if (os != null) {
              os.close();
            }
          }
          break;
        default:
          handleStatus(request, response);
          break;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected void getStream(String uri, OutputStream ostream) throws IOException, OperationException {
    HttpGet request = new HttpGet(uri);
    request.addHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_OCTET_STREAM);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          OutputStream os = new BufferedOutputStream(ostream);
          InputStream is = null;
          try {
            is = response.getEntity().getContent();
            byte[] buffer = new byte[1024];
            int length;
            while ((length = is.read(buffer)) > 0) {
              os.write(buffer, 0, length);
            }
          } finally {
            is.close();
            os.close();
          }
          break;
        default:
          handleStatus(request, response);
          break;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  private static class CloseResponseInputStream extends BufferedInputStream {
    CloseableHttpResponse _response;

    public CloseResponseInputStream(InputStream in, CloseableHttpResponse response) {
      super(in);
      _response = response;
    }

    @Override
    public void close() throws IOException {
      super.close();
      _response.close();
    }
  }

  protected InputStream getStream(String uri, String mediaType) throws IOException,
  OperationException {
    HttpGet request = new HttpGet(uri);
    request.addHeader(HttpHeaders.ACCEPT, mediaType);
    CloseableHttpResponse response = null;
    try {
      response = execute(request);
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          CloseResponseInputStream res = new CloseResponseInputStream(response.getEntity().getContent(), response);
          response = null;
          return res;
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    } finally {
      if (response != null) {
        response.close();
      }
    }
  }

  protected InputStream getStream(String uri) throws IOException, OperationException {
    return getStream(uri, MediaType.APPLICATION_OCTET_STREAM);
  }

  protected <T> T getStream(String uri, ObjectMapper mapper, TypeReference<T> type)
      throws IOException, OperationException {
    HttpGet request = new HttpGet(uri);
    request.addHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_OCTET_STREAM);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          return mapper.readValue(response.getEntity().getContent(), type);
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected <T> T getStream(String uri, ObjectMapper mapper, Class<T> type) throws IOException,
  OperationException {
    HttpGet request = new HttpGet(uri);
    request.addHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_OCTET_STREAM);
    try (CloseableHttpResponse response = execute(request)) {
      switch (response.getStatusLine().getStatusCode()) {
        case HttpStatus.SC_OK:
          return mapper.readValue(response.getEntity().getContent(), type);
        default:
          handleStatus(request, response);
          return null;
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
      return null;
    }
  }

  protected CloseableHttpResponse execute(HttpUriRequest req) throws ClientProtocolException, IOException {
    if (logger.isLoggable(logLevel)) {
      logger.log(logLevel, "Executing request: " + toString(req));
    }
    if (authToken != null) {
      req.setHeader("Authorization", authToken);
    }
    if (clientid != null) {
      req.setHeader("X-IBM-Client-Id", clientid);
    }
    if (secret != null) {
      req.setHeader("X-IBM-Client-Secret", secret);
    }
    CloseableHttpResponse response = httpClient.execute(req);
    // Header encoding = response.getFirstHeader("Content-Encoding");
    // if (encoding!=null) LOG.log(Level.INFO,"Encoding is "+ encoding.getValue());
    return response;
  }

  protected void putFile(String uri, File archive) throws OperationException {
    HttpPut request = new HttpPut(uri);
    try {
      request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_OCTET_STREAM);
      setOptionalCompressedEntity(request, new FileEntity(archive));
      try (CloseableHttpResponse response = execute(request)) {
        switch (response.getStatusLine().getStatusCode()) {
          case HttpStatus.SC_NO_CONTENT:
            return;
          default:
            handleStatus(request, response);
            return;
        }
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  protected void putStream(String uri, InputStream is) throws OperationException {
    HttpPut request = new HttpPut(uri);
    try {
      request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_OCTET_STREAM);
      setOptionalCompressedEntity(request, new InputStreamEntity(is, -1));
      try (CloseableHttpResponse response = execute(request)) {
        switch (response.getStatusLine().getStatusCode()) {
          case HttpStatus.SC_NO_CONTENT:
            return;
          default:
            handleStatus(request, response);
            return;
        }
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  static class JsonEntity extends AbstractHttpEntity {
    Object obj;
    ObjectMapper mapper;

    public JsonEntity(ObjectMapper mapper, Object obj) {
      this.obj = obj;
      this.mapper = mapper;
    }

    @Override
    public boolean isRepeatable() {
      return true;
    }

    @Override
    public long getContentLength() {
      return -1;
    }

    @Override
    public boolean isStreaming() {
      return false;
    }

    @Override
    public InputStream getContent() throws IOException {
      throw new UnsupportedOperationException();
    }

    @Override
    public void writeTo(final OutputStream outstream) throws IOException {
      mapper.writeValue(outstream, obj);
    }
  };

  protected void putStream(String uri, ObjectMapper mapper, Object obj) throws OperationException {
    HttpPut request = new HttpPut(uri);
    try {
      request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_OCTET_STREAM);
      setOptionalCompressedEntity(request, new JsonEntity(mapper, obj));
      try (CloseableHttpResponse response = execute(request)) {
        switch (response.getStatusLine().getStatusCode()) {
          case HttpStatus.SC_NO_CONTENT:
            return;
          default:
            handleStatus(request, response);
            return;
        }
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }
}