/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import com.ibm.optim.oaas.client.job.model.JobExecutionStatus;

/**
 * Provides notifications when a job is executed and monitored by an executor.
 * 
 * @see JobExecutor
 *
 */
public interface JobCallback {
  /**
   * Called when a job has been created.
   * 
   * @see JobExecutionStatus#CREATED
   * @param response
   *            The job response providing data about the executing job.
   */
  public void created(JobResponse response);

  /**
   * Called when a job has been submitted.
   * 
   * @see JobExecutionStatus#NOT_STARTED
   * @param response
   *            The job response providing data about the executing job.
   */
  public void submitted(JobResponse response);

  /**
   * Called when a job has started to run. Note that this call can be skipped if
   * the job execution is very short.
   * 
   * @see JobExecutionStatus#RUNNING
   * @param response
   *            The job response providing data about the executing job.
   */
  public void running(JobResponse response);

  /**
   * Called when a job has been processed.
   * 
   * @see JobExecutionStatus#PROCESSED
   * @param response
   *            The job response providing data about the executing job.
   */
  public void processed(JobResponse response);

  /**
   * Called when a job has failed.
   * 
   * @see JobExecutionStatus#FAILED
   * @param response
   *            The job response providing data about the executing job.
   */
  public void failed(JobResponse response);

  /**
   * Called when a job is being interrupted or has already been interrupted.
   * 
   * @see JobExecutionStatus#INTERRUPTING
   * @see JobExecutionStatus#INTERRUPTED
   * @param response
   *            The job response providing data about the executing job.
   */
  public void interruption(JobResponse response);

  /**
   * Called when an exception has occurred.
   * 
   * @param response
   *            The job response providing data about the executing job.
   * @param e
   *            The raised exception.
   */
  public void exception(JobResponse response, Exception e);

  /**
   * Called when the executor has completed the processing of the job request.
   * 
   * @param response
   *            The job response providing data about the executing job.
   */
  public void completed(JobResponse response);
}