/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.job.model.Job;
import com.ibm.optim.oaas.client.job.model.JobAttachment;
import com.ibm.optim.oaas.client.job.model.JobAttachmentCreationData;
import com.ibm.optim.oaas.client.job.model.JobAttachmentType;
import com.ibm.optim.oaas.client.job.model.JobCreationData;
import com.ibm.optim.oaas.client.job.model.JobExecutionStatus;
import com.ibm.optim.oaas.client.job.model.JobFailureInfo;
import com.ibm.optim.oaas.client.job.model.JobLogItem;

/**
 * Interface of a job client used to create, submit and monitor optimization
 * jobs.
 *
 */
public interface JobClient {
  /**
   * Deletes all jobs. Jobs currently being executed will be killed before being deleted.
   * 
   * @throws OperationException
   *             if any other remote exception was raised.
   */
  public void deleteAllJobs() throws OperationException;

  /**
   * Returns all the jobs.
   * 
   * @return The jobs.
   * @throws OperationException
   *             if any other remote exception was raised.
   */
  public List<? extends Job> getAllJobs() throws OperationException;

  /**
   * Creates a new job. Data can be passed to initialize the job. The
   * application ID refers to the type of engine (for example {@code opl}
   * or {@code cplex}). There are two ways to manage the application ID:
   * 
   * &lt;ul&gt;
   * &lt;li&gt;
   * The application ID can be specified explicitly at creation time. In this
   * case, the attachments can be declared and uploaded incrementally. When
   * the job is submitted, the list of attachments must be compatible with the
   * application ID, otherwise an exception will be returned.
   * &lt;/li&gt;
   * &lt;li&gt;
   * The application ID can be left unspecified at creation time. In this case, the full
   * list of attachments must be declared, and the application ID will be deduced
   * from the attachment extensions.
   * &lt;/li&gt;
   * &lt;ul&gt;
   * 
   * <p>
   * Note that when the attachments are declared and length is specified, the
   * subscription limits are also pre-checked so that an exception will be
   * raised before actually uploading the content.
   * </p>
   * 
   * @param data The data used to initialize the job.
   * @return The ID of the newly created job.
   * @throws SubscriptionException if the subscription is invalid or a limit was reached.
   * @throws ValidationException if the application or attachments are invalid.
   * @throws OperationException if any other remote exception is raised such as a system
   *             issue or authentication issue.
   */
  public String createJob(JobCreationData data) throws OperationException,
  SubscriptionException, ValidationException;

  /**
   * Creates a new job by copying an existing job. The existing job
   * cannot currently be running or waiting for execution. All creation data
   * is copied over to the new job. By default, input attachments are copied
   * to the new job. If a shallow copy is requested, the contents will point to the existing job
   * and if it is deleted, accessing the referenced input attachments will raise an exception. Output
   * attachments are ignored. Optionally, job creation data can be passed
   * to override the parameters and declare additional or replacement input attachments.
   * @see #recreateJob(String, JobCreationData, boolean) 
   * 
   * @param jobid The job ID of the job to copy.
   * @param data The data used to override the template value or {@code null} if not specified.
   * @param shallow Indicates if a shallow copy is requested. If {@code true}, the job contents point to the existing job.
   * @return The created job ID.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   * @throws ValidationException
   *             if the application or attachments are invalid.
   * @throws OperationException
   *             if any other remote exception is raised such as a system
   *             issue or authentication issue.
   */
  public String copyJob(String jobid, JobCreationData data, boolean shallow) throws OperationException,
  SubscriptionException, ValidationException, JobNotFoundException;

  /**
   * Creates a new job by replacing an existing job.
   * <p>
   * This can be used to resubmit a failed job for example. The existing job
   * cannot currently be running or waiting for execution. All creation data
   * is copied over to the new job. Input attachments are declared
   * in the new job and the contents are transferred to the new job. Output
   * attachments are ignored. Optionally, job creation data can be passed
   * to override the parameters and declare additional input attachments.
   * The existing job is automatically deleted.
   * </p>
   * 
   * @param jobid The job ID of the template job.
   * @param data The data used to override the existing values, or <code>null</code> if not specified.
   * @param execute Indicates if the job should be immediately submitted for execution. 
   * @return The created job id.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   * @throws ValidationException
   *             if the application or attachments are invalid.
   * @throws OperationException
   *             if any other remote exception is raised, such as a system
   *             issue or authentication issue.
   */
  public String recreateJob(String jobid, JobCreationData data, boolean execute) throws OperationException,
  SubscriptionException, ValidationException,JobNotFoundException;

  /**
   * Returns a given job.
   * 
   * @param jobid
   *            The job ID.
   * @return The job.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public Job getJob(String jobid) throws OperationException,
  JobNotFoundException;

  /**
   * Deletes a job. If the job is currently running, it will be killed before being 
   * deleted. If the job cannot be found, no action is taken.
   * 
   * @param jobid
   *            The job ID.
   * @return <code>true</code> if it was deleted, <code>false</code>
   *         otherwise.
   * @throws OperationException
   *             if any other remote exception was raised.
   */
  public boolean deleteJob(String jobid) throws OperationException;

  /**
   * Returns the job execution status. Depending on the status, additional
   * information will be available in the job description. For example,
   * if the status is <code>FAILED</code>, additional information will be available in the
   * <code>failure</code> property of the job.
   * @see #getJob(String)
   * 
   * @param jobid
   *            The job ID.
   * @return The job execution status.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public JobExecutionStatus getJobExecutionStatus(String jobid)
      throws OperationException, JobNotFoundException;

  /**
   * Submits a job for execution. If the job does not currently have the state
   * <code>CREATED</code> (for example, it is running or has already completed),
   * an exception will be returned.
   * 
   * @param jobid
   *            The job ID.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   * @throws ValidationException
   *             if the application or attachments are invalid.
   */
  public void executeJob(String jobid) throws OperationException,
  JobNotFoundException, SubscriptionException, ValidationException;

  /**
   * Aborts execution of a job. If the job is currently not running, nothing is done. When
   * a job is aborted, a clean stop is attempted.
   * 
   * @param jobid
   *            The job ID.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public void abortJob(String jobid) throws OperationException,
  JobNotFoundException;

  /**
   * Kills the execution of a job. If the job is currently not running, no action is taken.
   * 
   * @param jobid
   *            The job ID.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public void killJob(String jobid) throws OperationException,
  JobNotFoundException;

  /**
   * Returns the list of job attachments for a job.
   * 
   * @param jobid
   *            The job ID.
   * @return The list of attachments.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public List<? extends JobAttachment> getJobAttachments(String jobid)
      throws OperationException, JobNotFoundException;

  /**
   * Returns the list of job attachments of a given type for a job.
   * 
   * @param jobid
   *            The job ID.
   * @param type The type of the job attachment (an input attachment or an output attachment).
   * @return The list of attachments.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public List<? extends JobAttachment> getJobAttachments(String jobid, JobAttachmentType type)
      throws OperationException, JobNotFoundException;
  /**
   * Deletes all job attachments of a given job.
   * 
   * @param jobid
   *            The job ID.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public void deleteJobAttachments(String jobid) throws OperationException,
  JobNotFoundException;

  /**
   * Creates a job attachment for a given job.
   * 
   * @param jobid
   *            The job ID.
   * @param att
   *            The attachment data used to initialize the attachment.
   * @return The attachment ID (attachment name).
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   * @throws ValidationException if the application or attachments are invalid.
   */
  public String createJobAttachment(String jobid,
                                    JobAttachmentCreationData att) throws OperationException,
  JobNotFoundException, SubscriptionException, ValidationException;

  /**
   * Returns the job attachment (if any) of a given job.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The job attachment ID.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @return The attachment.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public JobAttachment getJobAttachment(String jobid, String attid)
      throws OperationException, JobNotFoundException,
      AttachmentNotFoundException;

  /**
   * Deletes a job attachment.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @return {@code true} if the job attachment was deleted.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public boolean deleteJobAttachment(String jobid, String attid)
      throws OperationException, JobNotFoundException;

  /**
   * Uploads job attachment contents as a file.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param file
   *            The content as a file.
   * @throws IOException
   *             if an exception occurred while accessing the file.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   * @throws OperationException
   *             if any other remote exception was raised.	 
   */
  public void uploadJobAttachment(String jobid, String attid, File file)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException, SubscriptionException;

  /**
   * Uploads job attachment contents as a stream.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param stream
   *            The content as a stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   */
  public void uploadJobAttachment(String jobid, String attid,
                                  InputStream stream) throws IOException, OperationException,
  JobNotFoundException, AttachmentNotFoundException,
  SubscriptionException;

  /**
   * Uploads job attachment contents with a custom writer.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param writer
   *            The content writer.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   */
  public void uploadJobAttachment(String jobid, String attid, AttachmentContentWriter writer) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException, SubscriptionException;

  /**
   * Uploads job attachment contents as a JSON stream.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param mapper
   *            The JSON mapper.
   * @param obj
   *            The object to be serialized as JSON.
   * @throws IOException
   *             if an exception occurred while serializing the object.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   * @throws SubscriptionException
   *             if the subscription is invalid or a limit was reached.
   */
  public void uploadJobAttachment(String jobid, String attid,
                                  ObjectMapper mapper, Object obj) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException, SubscriptionException;

  /**
   * Downloads job attachment contents and stores them as a file.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param file
   *            The content as a file.
   * @throws IOException
   *             if an exception occurred while accessing the file.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public void downloadJobAttachment(String jobid, String attid, File file)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException;

  /**
   * Downloads job attachment contents and copies them to an output stream.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param stream
   *            The content as a stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public void downloadJobAttachment(String jobid, String attid,
                                    OutputStream stream) throws IOException, OperationException,
  JobNotFoundException, AttachmentNotFoundException;

  /**
   * Downloads job attachment contents as a JSON object.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param mapper
   *            The JSON mapper.
   * @param type
   *            The type to deserialize.
   * @param <T> The type of the return object.
   * @return The stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public <T> T downloadJobAttachment(String jobid, String attid,
                                     ObjectMapper mapper, TypeReference<T> type) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException;

  /**
   * Downloads job attachment contents as a JSON object.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @param mapper
   *            The JSON mapper.
   * @param type
   *            The type to deserialize.
   * @param <T> The type of the return object.
   * @return The stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public <T> T downloadJobAttachment(String jobid, String attid,
                                     ObjectMapper mapper, Class<T> type) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException;

  /**
   * Downloads job attachment contents as a stream. The stream must be closed
   * to release resources.
   * 
   * @param jobid
   *            The job ID.
   * @param attid
   *            The attachment ID.
   * @return The stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws AttachmentNotFoundException
   *             if the attachment was not found.
   */
  public InputStream downloadJobAttachment(String jobid, String attid)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException;

  /**
   * Downloads the log and stores it as a file.
   * 
   * @param jobid
   *            The job ID.
   * @param file
   *            The log file.
   * @throws IOException
   *             if an exception occurred while accessing the file.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws OperationException
   *             if any other remote exception was raised.
   */
  public void downloadLog(String jobid, File file) throws IOException,
  OperationException, JobNotFoundException;

  /**
   * Downloads the log and copies it to an output stream.
   * 
   * @param jobid
   *            The job ID.
   * @param stream
   *            The log stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws JobNotFoundException
   *             if the job was not found.
   * @throws OperationException
   *             if any other remote exception was raised.
   */
  public void downloadLog(String jobid, OutputStream stream)
      throws IOException, OperationException, JobNotFoundException;

  /**
   * Downloads the log as a stream. The stream must be closed to release
   * resources.
   * 
   * @param jobid The job ID.
   * @return The log stream.
   * @throws IOException
   *             if an exception occurred while accessing the stream.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public InputStream downloadLog(String jobid) throws IOException,
  OperationException, JobNotFoundException;

  /**
   * Returns job failure information for the given job.
   * 
   * @param jobid
   *            The job ID.
   * @return The failure information.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public JobFailureInfo getFailureInfo(String jobid)
      throws OperationException, JobNotFoundException;

  /**
   * Returns the list of the log items of a given job.
   * <p>
   * Each job generates log items with an index sequence number
   * starting at 0.
   * This list can be very large
   * and only a maximum number of log items, defined internally by the service,
   * will be returned at once.
   * </p>
   * <p>
   * The returned list is always ordered by sequence index.
   * To know if more log items are available, specify the next
   * starting sequence index.
   * </p>
   * 
   * @see JobClient#getJobLogItems(String, long)
   * @see JobClient#getJobLogItems(String, long, boolean)
   * 
   * @param jobid
   *            The job ID.
   * @return The list of log items.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public List<? extends JobLogItem> getJobLogItems(String jobid)
      throws OperationException, JobNotFoundException;

  /**
   * Returns the list of the log items of a given job starting at a given index
   * (included).
   * <p>
   * Each job generates log items with an index sequence number
   * starting at 0.
   * This list can be very large
   * and only a maximum number of log items, defined internally by the service,
   * will be returned at once. 
   * </p>
   * <p>
   * The returned list is always ordered by sequence index.
   * To know if more log items are available, specify the
   * starting sequence index.
   * </p>
   * 
   * @see JobClient#getJobLogItems(String, long, boolean)
   * 
   * @param jobid
   *            The job ID.
   * @param start
   *            The start index.
   * @return The list of log items.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public List<? extends JobLogItem> getJobLogItems(String jobid, long start)
      throws OperationException, JobNotFoundException;

  /**
   * Returns the list of log items of a given job starting at a given index
   * (included).
   * <p>
   * This list can be very large
   * and only a maximum number of log items, defined internally by the service,
   * will be returned at once. 
   * The returned list is always ordered by sequence index.
   * To know if more log items are available, specify the
   * starting sequence index as the last sequence index + 1.
   * </p>
   * <p>
   * The continuous mode is designed to iterate over the log items as they become available until
   * the job ends to retrieve and display the live logs.
   * If the continuous flag is set, the server will return the next elements with no gap. It will also 
   * set the missing flag of a log item if the log item is still not available after an internal timeout. 
   * It will also set the log item stop flag when the sequence is completed. 
   * Here is a code snippet showing how to use the continuous mode.
   * </p>
   * {@code
   *	boolean stop = false;
   *  long index = 0;
   *  while (!stop) {
   *    List<? extends JobLogItem> items = jobclient.getJobLogItems(jobid, index, true);
   *    if ((items != null) && !items.isEmpty()) {
   *      display(items);
   *      JobLogItem last = items.get(items.size() - 1);
   *      stop = last.stop();
   *      index = last.getSeqid() + 1;
   *    } else {
   *      Thread.sleep(1000); // no new log so wait a little
   *    }
   *  }
   * }
   * 
   * @param jobid
   *            The job ID.
   * @param start
   *            The start index. 
   * @param continuous
   *            Indicates if server will try to return log elements in a continuous sequence  
   * @return The list of log items.
   * @throws OperationException
   *             if any other remote exception was raised.
   * @throws JobNotFoundException
   *             if the job was not found.
   */
  public List<? extends JobLogItem> getJobLogItems(String jobid, long start, boolean continuous)
      throws OperationException, JobNotFoundException;

  /**
   * Creates a new object to create and initialize a new job.
   * 
   * @return A job creation data instance.
   */
  public JobCreationData newJobCreationData();

  /**
   * Creates new input attachment data to be added to a job.
   * 
   * @param name
   *            The attachment name.
   * @return The attachment data.
   */
  public JobAttachmentCreationData newInputJobAttachment(String name);

  /**
   * Creates a new job request builder using this client.
   * 
   * @return The job request.
   */
  public JobRequestBuilder newRequest();
}