/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.LineNumberReader;
import java.io.StringReader;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.ibm.optim.oaas.client.OaasException;
import com.ibm.optim.oaas.client.http.HttpClientFactory;
import com.ibm.optim.oaas.client.impl.ClientMessageCodes;
import com.ibm.optim.oaas.client.job.impl.JobClientBuilderImpl;

/**
 * Factory to create and configure job clients.
 *
 */
public class JobClientFactory {
  /**
   * This is the text ({@value #BLUEMIX_SERVICE_LABEL}) that is used to search for a definition of
   * the service in Bluemix when no name is provided.
   */
  public static final String BLUEMIX_SERVICE_LABEL = "docplexcloud";

  private static final String BLUEMIX_PROPERTY_NAME = "name";
  private static final String BLUEMIX_PROPERTY_LABEL = "label";

  private static final Logger logger = Logger.getLogger(JobClientFactory.class.getName());
  private static final ObjectReader objectReader;

  static {
    final ObjectMapper mapper = new ObjectMapper();
    objectReader = mapper.reader();
  }

  /**
   * Creates a default job client to connect to the given base URL using the
   * given client ID. The default job client uses the default HTTP client.
   * 
   * @param baseURL
   *            The base URL.
   * @param clientid
   *            The client ID of the API key.
   * @return The configured job client.
   */
  public static JobClient createDefault(String baseURL, String clientid) {
    return new JobClientBuilderImpl(baseURL, clientid, null).http(HttpClientFactory.createDefault()).build();
  }

  /**
   * Returns a builder to configure the job client.
   * 
   * @param baseURL
   *            The base URL.
   * @param clientid
   *            The client ID of the API key.
   * @return The builder.
   */
  public static JobClientBuilder custom(String baseURL, String clientid) {
    return new JobClientBuilderImpl(baseURL, clientid, null);
  }

  /**
   * Returns a builder to configure the job client.
   * 
   * @param baseURL
   *            The base URL.
   * @param clientid
   *            The client ID of the API key.
   * @param secret
   *            The client secret of the API key.
   * @return The builder.
   */
  public static JobClientBuilder custom(String baseURL,String clientid, String secret) {
    return new JobClientBuilderImpl(baseURL, clientid, secret);
  }

  /**
   * Creates a default job client to connect to the given base URL using the
   * given client ID and client secret. The default job client uses the default HTTP client.
   * 
   * @param baseURL
   *            The server URL.
   * @param clientid
   *            The client ID of the API key.
   * @param secret
   *            The client secret of the API key.
   * @return The configured job client.
   */
  public static JobClient createDefault(String baseURL, String clientid, String secret) {
    return new JobClientBuilderImpl(baseURL, clientid, secret).build();
  }

  private static String[] getBluemixCredentials(JsonNode node) {
    if (node == null) {
      return null;
    }
    // credentials:
    // 0 = url
    // 1 = client_id
    // 2 = subscriptionId
    JsonNode nurl = node.get("url");
    JsonNode nclient_id = node.get("client_id");
    JsonNode nsubscriptionId = node.get("subscriptionId");
    String url = (nurl != null) ? nurl.textValue() : null;
    String client_id = (nclient_id != null) ? nclient_id.textValue() : null;
    String subscriptionId = (nsubscriptionId != null) ? nsubscriptionId.textValue() : null;
    if ((url != null) && (client_id != null)) {
      if (subscriptionId != null) {
        return new String[] { url, client_id, subscriptionId };
      } else {
        return new String[] { url, client_id };
      }
    }
    // we failed
    return null;
  }

  private static String[] getBluemixCredentials(final String name) throws OaasException {
    final String vcapServices = System.getenv("VCAP_SERVICES");
    return getBluemixCredentials(vcapServices, name);
  }

  private static boolean matches(final JsonNode node, final String name) {
    if (name != null) {
      JsonNode n = node.get(BLUEMIX_PROPERTY_NAME);
      if ((n != null) && name.equalsIgnoreCase(n.textValue())) {
        return true;
      }
    } else {
      // we are looking for a label that starts with "docplexcloud" (and not a user-defined service)
      JsonNode n = node.get(BLUEMIX_PROPERTY_LABEL);
      if (n != null) {
        String value = n.textValue();
        if ((value != null) && value.toLowerCase().startsWith(BLUEMIX_SERVICE_LABEL)) {
          return true;
        }
      }
    }
    // if we get here then not a match
    return false;
  }

  static String[] getBluemixCredentials(final String vcapServices,
                                        final String name) throws OaasException {
    if ((vcapServices != null) && !vcapServices.trim().isEmpty()) {
      // credentials:
      // 0 = url
      // 1 = client_id
      // 2 = subscriptionId
      try (LineNumberReader in = new LineNumberReader(new StringReader(vcapServices))) {
        JsonNode vcap = objectReader.readTree(in);
        Iterator<Entry<String, JsonNode>> it = vcap.fields();
        while (it.hasNext()) {
          Entry<String, JsonNode> entry = it.next();
          if (entry.getValue() instanceof ArrayNode) {
            ArrayNode arr = (ArrayNode) entry.getValue();
            for (JsonNode node : arr) {
              if (matches(node, name)) {
                final String[] credentials = getBluemixCredentials(node.get("credentials"));
                if (credentials != null) {
                  return credentials;
                }
              }
            }
          } else {
            if (matches(entry.getValue(), name)) {
              final String[] credentials = getBluemixCredentials(entry.getValue().get("credentials"));
              if (credentials != null) {
                return credentials;
              }
            }
          }
        }
      }
      catch (Exception e) {
        logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
      }
    }

    // if we get here then we did not find the url and client_id
    final String error = (name != null) ? name : BLUEMIX_SERVICE_LABEL + '*';
    final Object[] args = { ((name != null) ? BLUEMIX_PROPERTY_NAME : BLUEMIX_PROPERTY_LABEL), error };
    throw new OaasException(ClientMessageCodes.AKCJC5060E_BLUEMIX_CREDENTIALS_MISSING.name(), args) {
      /**
       * Default.
       */
      private static final long serialVersionUID = 1L;

      @Override
      protected String getResourceBundleName() {
        return ClientMessageCodes.class.getName();
      }
    };
  }

  /**
   * Creates a default job client to connect to the IBM Decision Optimization service by looking up the
   * service with the label starting with {@link #BLUEMIX_SERVICE_LABEL} in the VCAP_SERVICES.
   * The default job client uses the default HTTP client.
   * 
   * @return The configured job client.
   * @throws OaasException if the service credentials can not be found.
   * 
   * @see JobClientFactory#createDefaultFromBluemix(String)
   */
  public static JobClient createDefaultFromBluemix() throws OaasException {
    return createDefaultFromBluemix(null);
  }

  /**
   * Creates a default job client to connect to the IBM Decision Optimization service by looking up the
   * service name in the VCAP_SERVICES.
   * The default job client uses the default HTTP client.
   * 
   * @param serviceName The name of the IBM Decision Optimization service to be used.
   * @return The configured job client.
   * @throws OaasException if the service credentials can not be found.
   * 
   * <p>
   * The service name is the name that was given to the service and bound to the Bluemix application.
   * </p>
   */
  public static JobClient createDefaultFromBluemix(final String serviceName) throws OaasException {
    final String[] credentials = getBluemixCredentials(serviceName);
    return new JobClientBuilderImpl(credentials[0], credentials[1], null).http(HttpClientFactory.createDefault()).build();
  }

  /**
   * Returns a builder to configure the job client.
   * 
   * @param serviceName The name of the IBM Decision Optimization service to be used.
   * @return The builder.
   * @throws OaasException if the service credentials can not be found.
   * 
   * <p>
   * The service name is the name that was given to the service and bound to the Bluemix application.
   * </p>
   */
  public static JobClientBuilder customFromBluemix(final String serviceName) throws OaasException {
    final String[] credentials = getBluemixCredentials(serviceName);
    return new JobClientBuilderImpl(credentials[0], credentials[1], null);
  }

  /**
   * Returns a builder to configure the job client by looking for an IBM Decision Optimization
   * service with the label starting with {@link #BLUEMIX_SERVICE_LABEL} in the VCAP_SERVICES.
   * 
   * @return The builder.
   * @throws OaasException if the service credentials can not be found.
   * 
   * @see JobClientFactory#customFromBluemix(String)
   */
  public static JobClientBuilder customFromBluemix() throws OaasException {
    return customFromBluemix(null);
  }
}