/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.IOException;
import java.util.concurrent.Future;

import com.ibm.optim.oaas.client.OperationException;

/**
 * A job executor is in charge of submitting and monitoring the execution of
 * jobs.
 * 
 * @see JobExecutorBuilder
 * @see JobExecutorFactory
 */
public interface JobExecutor {
  /**
   * Stops the execution of the executor and releases all used resources.
   */
  public void shutdown();

  /**
   * Executes a job request. The returned future will be done when the job has
   * finished or an exception occurred.
   * 
   * @param request The request.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the operation was interrupted.
   * @throws JobException if the job failed for any reason.
   */
  public Future<JobResponse> execute(JobRequest request)
      throws OperationException, IOException, InterruptedException, JobException;

  /**
   * Executes a job request and calls a callback at major steps of the
   * execution. The returned future will be done when the job has finished or
   * an exception occurred.
   * 
   * @param request The request.
   * @param callback The call back.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the operation was interrupted.
   * @throws JobException if the job failed for any reason.
   */
  public Future<JobResponse> execute(JobRequest request, JobCallback callback)
      throws OperationException, IOException, InterruptedException, JobException;


  /**
   * Creates a job request but does not submit the job.
   * The returned future will be done when the job has been created with all the input attachment uploaded. 
   * @see #monitor(JobRequest, String, JobCallback)
   * @param request The request.
   * @param callback The call back.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the operation was interrupted.
   * @throws JobException if the job failed for any reason.
   */
  public Future<JobResponse> create(JobRequest request, JobCallback callback)
      throws OperationException, IOException, InterruptedException,JobException;

  /**
   * Submits a job request but does not monitor the execution nor process the outputs and logs.
   * The returned future will be done when the job has been submitted with all the input attachment uploaded. 
   * @see #monitor(JobRequest, String, JobCallback)
   * @param request The request.
   * @param callback The call back.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the operation was interrupted.
   * @throws JobException if the job failed for any reason.
   */
  public Future<JobResponse> submit(JobRequest request, JobCallback callback)
      throws OperationException, IOException, InterruptedException, JobException;


  /**
   * Monitor a job request that was submitted. The job will not be created and attachment will not be uploaded.
   * The returned future will be done when the job is completed.
   * @see #submit(JobRequest, JobCallback)
   * 
   * @param request The request.
   * @param jobid The ID of the job to monitor.
   * @param callback The call back or {@code null} if not defined.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the operation was interrupted.
   * @throws JobException if the job failed for any reason.
   */
  public Future<JobResponse> monitor(JobRequest request, String jobid, JobCallback callback)
      throws OperationException, IOException, InterruptedException, JobException;
}