/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Builder used to configure a job executor.
 *
 */
public interface JobExecutorBuilder {
  /**
   * Default number of threads.
   */
  public static int DEFAULT_THREADS = 10;

  /**
   * Default time interval used to check job status in milliseconds.
   */
  public static int DEFAULT_INTERVAL = 5000;

  /**
   * Default timeout for the job monitoring. If the timeout is reached, an
   * interrupted exception will be thrown.
   */
  public static int DEFAULT_TIMEOUT = -1;

  /**
   * Default maximum retry attempts on calls to servers, when applicable.
   */
  public static int DEFAULT_RETRY = 3;

  /**
   * Default initial retry delay in milliseconds.
   */
  public static int DEFAULT_RETRY_DELAY = 4000;

  /**
   * Sets the number of threads.
   * 
   * @param threads
   *            Number of threads.
   * @return The builder.
   */
  public JobExecutorBuilder threads(int threads);

  /**
   * Sets the time interval used to check the job status.
   * 
   * @param interval
   *            The time interval in milliseconds.
   * @param unit
   *             The time unit.
   * @return The builder.
   */
  public JobExecutorBuilder interval(long interval, TimeUnit unit);

  /**
   * Sets the timeout for the job monitoring in milliseconds. If the timeout is reached,
   * an interrupted exception will be thrown.
   * 
   * @param timeout
   *            The timeout or -1 for no timeout.
   * @param unit
   *             The time unit.
   * @return The builder.
   */
  public JobExecutorBuilder timeout(long timeout, TimeUnit unit);

  /**
   * Sets the executor service used to manage threads and tasks within the job
   * executor.
   * 
   * @param service
   *            The executor service.
   * @return The builder.
   */
  public JobExecutorBuilder service(ExecutorService service);

  /**
   * Sets maximum retry attempts on calls to servers, when applicable.
   * 
   * @param retry
   *            The maximum.
   * @return The builder.
   */
  public JobExecutorBuilder retry(int retry);

  /**
   * Sets the initial retry delay. Retry delay is multiplied by 2 at each
   * retry of a given call.
   * 
   * @param delay
   *            The delay.
   * @param unit
   *             The time unit.
   * @return The builder.
   */
  public JobExecutorBuilder retryDelay(long delay,TimeUnit unit);

  /**
   * Returns the configure job executor.
   * 
   * @return The job executor.
   */
  public JobExecutor build();
}