/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.IOException;

import com.ibm.optim.oaas.client.OperationException;

/**
 * Interface for a job input source. The job executor has predefined functions to declare and upload attachments
 * from streams or files. A job input lets you define a custom source and when the executor is ready to
 * upload the content of the attachment, it will call the {@code upload} function that you can define.
 */
public interface JobInput {
  /**
   * Returns the name of the input source. The name will be used to declare the attachment.
   * 
   * @return The input name.
   */
  public String getName();

  /**
   * Returns the length of the input source or -1 if unknown.
   * 
   * @return The input source length.
   */
  public long getLength();

  /**
   * Indicates if the upload can be repeated. If the upload is repeatable,
   * the executor will automatically retry to upload the content upon failure.
   * 
   * @return <code>true</code> if the content can be repeated for retry.
   */
  public boolean isRepeatable();

  /**
   * Uploads this source as the contents of an attachment to a given job. The
   * attachment was declared by the executor with the name returned by the
   * job input.
   * 
   * @param client The job client.
   * @param jobid The job ID.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws AttachmentNotFoundException if the attachment was not declared or not found.
   * @throws JobNotFoundException if the job is not found with the given id.
   * @throws SubscriptionException if there is not a valid subscription for this job.
   */
  public void upload(JobClient client, String jobid)
      throws OperationException, IOException, JobNotFoundException,
      AttachmentNotFoundException, SubscriptionException;
}