/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.IOException;

import com.ibm.optim.oaas.client.OperationException;

/**
 * Interface of a job output source. The job executor has predefined functions to declare and download attachments
 * to streams or files. A job output lets you define a custom source and when the executor is ready to
 * download the content of the attachment, it will call the <code>download</code> function that you can define.
 */
public interface JobOutput {
  /**
   * Returns the name of the output source. 
   * 
   * @return The output source name.
   */
  public String getName();

  /**
   * Returns the content of the attachment as specified in the builder such as a
   * file, stream or object.
   * 
   * @return The content.
   */
  Object getContent();

  /**
   * Downloads an attachment.
   * 
   * @param client The job client.
   * @param jobid The job ID.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws AttachmentNotFoundException if the attachment was not declared or not found.
   * @throws JobNotFoundException if the job is not found with the given id.
   */
  public void download(JobClient client, String jobid)
      throws JobNotFoundException, AttachmentNotFoundException,
      OperationException, IOException;

  /**
   * Sets the name of the output. This is called by the executor when the output name is
   * retrieved.
   * @param name The output name.
   */
  public void setName(String name);
}