/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.concurrent.Future;

import com.ibm.icu.text.DateFormat;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.job.impl.JobRequestImpl;
import com.ibm.optim.oaas.client.job.model.JobCreationData;

/**
 * Interface of a job request.
 * 
 * @see JobClientBuilder#build()
 */
public interface JobRequest {
  /**
   * Date format that can be specified for the live log.
   */
  public static DateFormat UTC_ISO8601_DATE_FORMAT = JobRequestImpl.UTC_ISO8601_DATE_FORMAT;

  /**
   * Returns the job client.
   * 
   * @return The job client.
   */
  public JobClient getClient();

  /**
   * Returns the job data used to create and initialize the job.
   * 
   * @return The job creation data.
   */
  public JobCreationData getData();

  /**
   * Returns the list of configured input sources.
   * 
   * @return The list of input sources.
   */
  public List<? extends JobInput> getInput();

  /**
   * Returns the list of job result output values.
   * 
   * @return The list of job output.
   */
  public List<? extends JobOutput> getOutput();

  /**
   * Returns The list of job log output values.
   * 
   * @return The list of job log output.
   */
  public List<? extends JobLogOutput> getLogOutput();

  /**
   * Indicates if the job will be deleted upon completion.
   * A job is completed if its status
   * is one of <code>PROCESSED</code>, <code>FAILED</code>, or <code>INTERRUPTED</code>.
   * 
   * @return The job delete flag.
   */
  public boolean isDeleteOnCompletion();


  /**
   * Returns the job monitoring timeout.
   * @return The monitoring timeout.
   */
  long getTimeout();

  /**
   * Executes this request by submitting it to a job executor.
   * 
   * @param executor The job executor.
   * @return The future job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the job was interrupted.
   * @throws JobException if the job failed.
   * @see JobExecutor#execute(JobRequest)
   */
  public Future<JobResponse> execute(JobExecutor executor)
      throws OperationException, IOException, InterruptedException, JobException;
  /**
   * Creates the job as defined in this request.
   * @param executor  The job executor.
   * @return The future job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the job was interrupted.
   * @throws JobException if the job failed.
   */
  Future<JobResponse> create(JobExecutor executor) throws OperationException,
  IOException, InterruptedException, JobException;

  /**
   * Returns The job ID once the job has been submitted.
   * 
   * @return The job ID or <code>null</code> if the request has not yet been submitted.
   */
  String getJobId();

  /**
   * Returns the live log stream if defined.
   * @return the live log stream or <code>null</code> if not defined.
   */
  OutputStream getLivelog();

  /**
   * Returns the live log date format.
   * @return The date format.
   */
  DateFormat getLivelogDateFormat();

  /**
   * Returns the job to copy if defined.
   * @return the job ID to copy or <code>null</code> if undefined.
   */
  String getCopyJobId();

  /**
   * Indicates if a shallow copy was requested.
   * @return <code>true</code> if shallow copy, <code>false</code> otherwise
   * @see #getCopyJobId()
   */
  boolean getShallowCopy();

  /**
   * Returns the job to recreate if defined.
   * @return the job ID to recreate or <code>null</code> if undefined.
   */
  String getRecreateJobId();
}