/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ibm.icu.text.DateFormat;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.job.model.Job;
import com.ibm.optim.oaas.client.job.model.JobFailureType;

/**
 * Builder to configure a job request.
 *
 */
public interface JobRequestBuilder {
  /**
   * Sets the application ID. The application ID refers to the type of engine (for example
   * {@code opl} or {@code cplex}). There are two ways to manage the application ID:
   * &lt;ul&gt;
   * &lt;li&gt;
   * The application ID can be specified explicitly at creation time. In this case, the attachments
   * can be declared and uploaded incrementally. When the job is submitted, the list of attachments
   * must be compatible with the application ID, otherwise an exception will be returned.
   * &lt;/li&gt;
   * &lt;li&gt;
   * The application ID can be left unspecified at creation time. In this case, the full list of
   * attachments must be declared, and the application ID will be deduced from the attachment
   * extensions.
   * &lt;/li&gt;
   * &lt;/ul&gt;
   * 
   * 
   * @param applicationId
   *          The application ID.
   * @return The builder.
   */
  JobRequestBuilder application(String applicationId);

  /**
   * Sets the application version to use during solve. If this version is not available a
   * {@link JobFailureType#PROCESSING} failure is raised. If nothing is specified latest version is
   * used. It is possible for a solved job to get version used during solve using
   * {@link Job#getApplicationVersionUsed()}
   * 
   * @param version
   *          The application version.
   * @return The builder.
   */
  JobRequestBuilder version(String version);

  /**
   * Sets the client name used to identify the client submitting a job. The client name may be used
   * to distinguish between different clients using the same subscription.
   * 
   * @param name
   *          The client name.
   * @return The builder.
   */
  JobRequestBuilder clientName(String name);

  /**
   * Sets the client email address. The email address is used for notification of job completion as
   * well as alerts.
   * 
   * @param email
   *          The client email address.
   * @return The builder.
   */
  JobRequestBuilder clientEmail(String email);

  /**
   * Sets the value of a parameter.
   * 
   * @param name
   *          The parameter name.
   * @param value
   *          The parameter value.
   * @return The builder.
   */
  JobRequestBuilder parameter(String name, Object value);

  /**
   * Sets an input source to the given file. Several input sources can be defined.
   * 
   * @param file
   *          The file.
   * @return The builder.
   */
  JobRequestBuilder input(File file);

  /**
   * Tries to create a job, upload attachments, and submit a job in one call to avoid network round
   * trips. This optimized mode may not work if attachments are too big or if infrastructure does
   * not support multipart messages. Auto retry is not available in this mode. Only File and
   * InputStream attachments are supported in batch mode.
   * 
   * @return The builder.
   */
  JobRequestBuilder useBatchSubmitMode();

  /**
   * Sets an input source to the given file with a specific name. The name must be a file name with
   * an extension valid for this job. Several input sources can be defined.
   * 
   * @param name
   *          The attachment name.
   * @param file
   *          The file.
   * @return The builder.
   */
  JobRequestBuilder input(String name, File file);

  /**
   * Sets an input source to the given URL with a specific name. The name must be a file name with
   * an extension valid for this job. Several input sources can be defined.
   * 
   * @param name
   *          The attachment name.
   * @param url
   *          The URL.
   * @return The builder.
   */
  JobRequestBuilder input(String name, URL url);

  /**
   * Sets an input source to the given stream. The name must be a file name with an extension valid
   * for this job. Several input sources can be defined.
   * 
   * @param name
   *          The attachment name.
   * @param stream
   *          The stream.
   * @return The builder.
   */
  JobRequestBuilder input(String name, InputStream stream);

  /**
   * Sets the input to the given JSON document. The name must be a file name with an extension valid
   * for this job. Several input sources can be defined.
   * 
   * @param name
   *          The attachment name.
   * @param mapper
   *          The mapper used to serialize the object.
   * @param input
   *          The input object.
   * @return The builder.
   */
  JobRequestBuilder input(String name, ObjectMapper mapper, Object input);

  /**
   * Adds a custom job input source. The name of the input source will be used to declare the
   * attachment, and when the executor is ready to upload the content, it delegates it to the job
   * input.
   * 
   * @param input
   *          The custom job input.
   * 
   * @return The builder.
   */
  JobRequestBuilder input(JobInput input);

  /**
   * Sets the result output to a file. Single output attachment is processed by this call. For
   * multiple result attachments use {@link JobRequestBuilder#output(String, File)} or
   * {@link JobClient#downloadJobAttachment(String, String)}. Several output commands can be
   * combined in the same request.
   * 
   * @param file
   *          The file to write to.
   * @return The builder.
   */
  JobRequestBuilder output(File file);

  /**
   * Sets the result output to a stream. Single output attachment is processed by this call. For
   * multiple result attachments use {@link JobRequestBuilder#output(String, OutputStream)} or
   * {@link JobClient#downloadJobAttachment(String, String)}. Several output commands can be
   * combined in the same request.
   * 
   * @param stream
   *          The stream to write to.
   * @return The builder.
   */
  JobRequestBuilder output(OutputStream stream);

  /**
   * Adds a custom result output. When the executor is ready to get the output attachment, it sets
   * the name of the job output and it delegates the upload to it. Several output commands can be
   * combined in the same request.
   * 
   * @param output The custom output.
   * @return The builder.
   */
  JobRequestBuilder output(JobOutput output);

  /**
   * Sets the result output to a JSON object. Single output attachment is processed by this call.
   * For multiple result attachments use
   * {@link JobRequestBuilder#output(String, ObjectMapper, TypeReference)} or
   * {@link JobClient#downloadJobAttachment(String, String)}. Several output
   * commands can be combined in the same request.
   * 
   * @param mapper The JSON mapper.
   * @param type The class of the object type.
   * @param <T> The object type.
   * @return The builder.
   */
  <T> JobRequestBuilder output(ObjectMapper mapper, TypeReference<T> type);

  /**
   * Sets the result output to a JSON object. Single output attachment is processed by this call.
   * For multiple result attachments use {@link JobRequestBuilder#output(String, ObjectMapper, Class)}
   * or {@link JobClient#downloadJobAttachment(String, String)}. Several output commands
   * can be combined in the same request.
   * 
   * @param mapper The JSON mapper.
   * @param type The class of the object type.
   * @param <T> The object type.
   * @return The builder.
   */
  <T> JobRequestBuilder output(ObjectMapper mapper, Class<T> type);

  /**
   * Sets the log output to a file. The log is retrieved at the end of the job execution and copied
   * into the given file. Several log commands can be combined in the same request.
   * 
   * @param file The file to write to.
   * @return The builder.
   */
  JobRequestBuilder log(File file);

  /**
   * Sets the log output to a stream. The log is retrieved at the end of the job execution and
   * copied into the given stream. Several log commands can be combined in the same request.
   * 
   * @param stream The stream to write to.
   * @return The builder.
   */
  JobRequestBuilder log(OutputStream stream);

  /**
   * Sets the live log output to a stream. The live log is updated incrementally while the job is
   * running. Only one output stream can be defined for a request.
   * 
   * @param stream The stream to write to.
   * @return The builder.
   */
  JobRequestBuilder livelog(OutputStream stream);

  /**
   * Sets the live log output to a stream. The live log is updated incrementally while the job is
   * running. Only one output stream can be defined for a request.
   * 
   * @param stream The stream to write to.
   * @param format Date formatter for the dates in the log.
   * @return The builder.
   */
  JobRequestBuilder livelog(OutputStream stream, DateFormat format);

  /**
   * Indicates that the job must be deleted on completion. A job is completed if its status is one
   * of {@code PROCESSED}, {@code FAILED}, or {@code INTERRUPTED}.
   * 
   * @param delete {@code true} if it must be deleted, {@code false} otherwise.
   * @return The builder.
   */
  JobRequestBuilder deleteOnCompletion(boolean delete);

  /**
   * Sets the timeout for the job monitoring. If the timeout value is reached, an interrupted
   * exception will be thrown. By default, the timeout defined in the executor is used. The timeout
   * only stops the monitoring, and jobs may still be running. To limit the job execution time,
   * refer to {@link com.ibm.optim.oaas.client.job.model.JobParameters#TIME_LIMIT} parameter.
   * 
   * @param timeout The timeout: -1 for no timeout or 0 for the executor default timeout.
   * @param unit The time unit.
   * @return The builder.
   */
  public JobRequestBuilder timeout(long timeout, TimeUnit unit);

  /**
   * Returns the configured job request.
   * 
   * @return The job request.
   */
  JobRequest build();

  /**
   * Builds the request and executes it using a given job executor
   * 
   * @param executor The job executor.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the job was interrupted.
   * @throws JobException if the job failed.
   * @see JobExecutor#execute(JobRequest)
   */
  public Future<JobResponse> execute(JobExecutor executor) throws OperationException, IOException,
  InterruptedException, JobException;

  /**
   * Builds the request and creates the job using a given job executor.
   * 
   * @param executor The job executor.
   * @return The future of the job response.
   * @throws OperationException if any other remote exception was raised.
   * @throws IOException if an I-O exception was raised.
   * @throws InterruptedException if the job was interrupted.
   * @throws JobException if the job failed.
   * @see JobExecutor#create(JobRequest, JobCallback)
   */
  public Future<JobResponse> create(JobExecutor executor) throws OperationException, IOException,
  InterruptedException, JobException;

  /**
   * Indicates that the new job should be constructed by copying an existing job. Parameters and
   * input attachments are automatically copied to the new job from the existing job. The builder
   * can be used to override parameters and define additional input attachments.
   * 
   * @param jobId The ID of the job to copy.
   * @return The builder.
   */
  JobRequestBuilder copy(String jobId);

  /**
   * Indicates that the new job should be constructed by copying an existing job. Parameters and
   * input attachments are automatically copied to the new job from the existing job. The builder
   * can be used to override parameters and define additional input attachments.
   * 
   * @param jobId The ID of the job to copy.
   * @param shallow {@code true} if a shallow copy is requested.
   * @return The builder.
   */
  JobRequestBuilder copy(String jobId, boolean shallow);

  /**
   * Indicates that the new job should be constructed by replacing an existing job. Parameters and
   * input attachments are automatically imported in the created job. The builder can be used to
   * override parameters and define additional input attachments.
   * 
   * @param jobId The ID of the job to replace.
   * @return The builder.
   */
  JobRequestBuilder recreate(String jobId);

  /**
   * Sets the result output to a file for a particular output attachment. Several output commands
   * can be combined in the same request.
   * 
   * @param attid The output attachment ID.
   * @param file The file to write to.
   * @return The builder.
   */
  JobRequestBuilder output(String attid, File file);

  /**
   * Sets the result output to a JSON object for a particular output attachment. Several output
   * commands can be combined in the same request.
   * 
   * @param attid The output attachment ID.
   * @param mapper The JSON mapper.
   * @param type The class of the output type.
   * @param <T> The output type.
   * @return The builder.
   */
  <T> JobRequestBuilder output(String attid, ObjectMapper mapper, Class<T> type);

  /**
   * Sets the result output to a JSON object for a particular output attachment. Several output
   * commands can be combined in the same request.
   * 
   * @param attid The output attachment ID.
   * @param mapper The JSON mapper.
   * @param type The type reference of the output type.
   * @param <T> The output type.
   * @return The builder.
   */
  <T> JobRequestBuilder output(String attid, ObjectMapper mapper, TypeReference<T> type);

  /**
   * Sets the result output to a stream for a particular output attachment. Several output commands
   * can be combined in the same request.
   * 
   * @param attid The output attachment ID.
   * @param stream The stream to write to.
   * @return The builder.
   */
  JobRequestBuilder output(String attid, OutputStream stream);
}