/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.List;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.entity.GzipCompressingEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.ServerExceptionData;
import com.ibm.optim.oaas.client.impl.DeletionStatus;
import com.ibm.optim.oaas.client.impl.MediaType;
import com.ibm.optim.oaas.client.impl.RestClient;
import com.ibm.optim.oaas.client.impl.Validate;
import com.ibm.optim.oaas.client.job.AttachmentContentWriter;
import com.ibm.optim.oaas.client.job.AttachmentNotFoundException;
import com.ibm.optim.oaas.client.job.AttachmentStorageLimitException;
import com.ibm.optim.oaas.client.job.EmptyAttachmentException;
import com.ibm.optim.oaas.client.job.FormatNotSupportedException;
import com.ibm.optim.oaas.client.job.IllegalJobStatusException;
import com.ibm.optim.oaas.client.job.InvalidApplicationException;
import com.ibm.optim.oaas.client.job.InvalidAttachmentNameException;
import com.ibm.optim.oaas.client.job.InvalidAttachmentsException;
import com.ibm.optim.oaas.client.job.JobClient;
import com.ibm.optim.oaas.client.job.JobLimitException;
import com.ibm.optim.oaas.client.job.JobNotFoundException;
import com.ibm.optim.oaas.client.job.MissingAttachmentsException;
import com.ibm.optim.oaas.client.job.StorageLimitException;
import com.ibm.optim.oaas.client.job.SubscriptionException;
import com.ibm.optim.oaas.client.job.UndefinedApplicationException;
import com.ibm.optim.oaas.client.job.UnrecognizedAttachmentsException;
import com.ibm.optim.oaas.client.job.ValidationException;
import com.ibm.optim.oaas.client.job.model.JobAttachmentCreationData;
import com.ibm.optim.oaas.client.job.model.JobAttachmentType;
import com.ibm.optim.oaas.client.job.model.JobCreationData;
import com.ibm.optim.oaas.client.job.model.JobExecutionStatus;
import com.ibm.optim.oaas.client.job.model.JobFailureInfo;
import com.ibm.optim.oaas.client.job.model.JobLogItem;
import com.ibm.optim.oaas.client.job.model.JobStatus;
import com.ibm.optim.oaas.client.job.model.impl.JobAttachmentImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobCreationDataImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobFailureInfoImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobLogItemImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobStatusImpl;

public class JobClientImpl extends RestClient implements JobClient {
  public JobClientImpl(CloseableHttpClient client, String url, String token) {
    super(client, url, token);
  }

  public JobClientImpl(CloseableHttpClient client, String url, String clientid, String secret) {
    super(client, url, clientid, secret);
  }

  public void deleteAllJobs() throws OperationException {
    delete(getBaseUrl() + "jobs");
    JobMessageCodes.AKCJC5214I_JOBS_DELETED.log();
  }

  public List<JobImpl> getAllJobs() throws OperationException {
    return getList(getBaseUrl() + "jobs",
                   new TypeReference<List<JobImpl>>() {
    });
  }

  public String createJob(JobCreationData data) throws OperationException,
  SubscriptionException, ValidationException {
    Validate.notNull(data, "data");
    try {
      String jobid = create(getBaseUrl() + "jobs", data);
      JobMessageCodes.AKCJC5201I_JOB_CREATED.log(jobid);
      return jobid;
    } catch (OperationException e) {
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }
  }

  public String submitJob(JobCreationData data, Object... atts) throws OperationException, SubscriptionException, ValidationException {

    Validate.notNull(data, "data");
    if ( data.getAttachments().size() != atts.length) {
      throw new IllegalArgumentException("Invalid number of attachments specified.");
    }
    String [] name = new String[atts.length+1], filename = new String[atts.length+1]; 
    boolean [] asJson = new boolean[atts.length+1];
    Object [] obj = new Object[atts.length+1];
    name[0] = "OaasJobCreationData";
    asJson[0] = true;
    obj[0] = data;
    for (int i = 0; i < data.getAttachments().size(); i++) {
      JobAttachmentCreationData att = data.getAttachments().get(i);
      name[i+1] = "OaasJobAttachment_" + i;
      filename[i+1] = att.getName();
      asJson[i+1] = false;
      obj[i+1] = atts[i];
    }
    try {
      String jobid = createMultiPart(getBaseUrl() + "jobs/submit", name, filename, asJson, obj);
      JobMessageCodes.AKCJC5201I_JOB_CREATED.log(jobid);
      return jobid;
    } catch (OperationException e) {
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }

  }

  public String copyJob(String jobid, JobCreationData data, boolean shallow) throws OperationException,
  SubscriptionException, ValidationException,JobNotFoundException{
    Validate.notNull(jobid, "jobid");
    try {
      String newjobid = create(getBaseUrl() + "jobs/" + encodeName(jobid) + "/copy?shallow="+shallow, data);
      JobMessageCodes.AKCJC5201I_JOB_CREATED.log(newjobid);
      return newjobid;
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }
  }

  public String recreateJob(String jobid, JobCreationData data, boolean execute) throws OperationException,
  SubscriptionException, ValidationException,JobNotFoundException{
    Validate.notNull(jobid, "jobid");
    try {
      String newjobid = create(getBaseUrl() + "jobs/" + encodeName(jobid) + "/recreate?execute="+execute, data);
      JobMessageCodes.AKCJC5201I_JOB_CREATED.log(newjobid);
      return newjobid;
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }
  }

  public JobImpl getJob(String id) throws OperationException,
  JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      return get(getBaseUrl() + "jobs/" + encodeName(id), JobImpl.class, true);
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  protected void checkJobNotFound(OperationException e)
      throws JobNotFoundException {
    ServerExceptionData data = e.getServerExceptionData();
    if (data != null
        && JobNotFoundException.class.getName().equals(
                                                       data.getExceptionclass())) {
      String jobid = "?";
      if (data.getParameters().size() > 0)
        jobid = data.getParameters().get(0);
      JobMessageCodes.AKCJC5101E_JOB_NOT_FOUND_EXCEPTION.log(jobid);
      throw new JobNotFoundException(jobid, e);
    }
  }

  private void checkAttNotFound(OperationException e)
      throws AttachmentNotFoundException {
    ServerExceptionData data = e.getServerExceptionData();
    if (data != null
        && AttachmentNotFoundException.class.getName().equals(
                                                              data.getExceptionclass())) {
      String jobid = "?";
      if (data.getParameters().size() > 0)
        jobid = data.getParameters().get(0);
      String attid = "?";
      if (data.getParameters().size() > 1)
        attid = data.getParameters().get(1);
      JobMessageCodes.AKCJC5102E_ATTACHMENT_NOT_FOUND_EXCEPTION.log(
                                                                    jobid, attid);
      throw new AttachmentNotFoundException(jobid, attid, e);
    }
  }

  private void checkSubscription(OperationException e)
      throws SubscriptionException {
    ServerExceptionData data = e.getServerExceptionData();
    if (data == null)
      return;
    if (JobLimitException.class.getName().equals(
                                                 data.getExceptionclass())) {
      JobMessageCodes.AKCJC5111E_JOB_LIMIT_EXCEPTION.log(data
                                                         .getParameters().toArray());
      throw new JobLimitException(data.getParameters().toArray(), e);
    } else if (AttachmentStorageLimitException.class.getName().equals(
                                                                      data.getExceptionclass())) {
      JobMessageCodes.AKCJC5112E_ATTACHMENT_STORAGE_LIMIT_EXCEPTION
      .log(data.getParameters().toArray());
      throw new AttachmentStorageLimitException(data.getParameters()
                                                .toArray(), e);
    } else if (StorageLimitException.class.getName().equals(
                                                            data.getExceptionclass())) {
      JobMessageCodes.AKCJC5113E_STORAGE_LIMIT_EXCEPTION.log(data
                                                             .getParameters().toArray());
      throw new StorageLimitException(data.getParameters().toArray(), e);
    } 		
  }

  private void checkValidation(OperationException e)
      throws ValidationException {
    ServerExceptionData data = e.getServerExceptionData();
    if (data == null)
      return;
    if (UndefinedApplicationException.class.getName().equals(
                                                             data.getExceptionclass())) {
      JobMessageCodes.AKCJC5120E_UNDEFINED_APPLICATION_EXCEPTION.log(data
                                                                     .getParameters().toArray());
      throw new UndefinedApplicationException(data.getParameters()
                                              .toArray(), e);
    } else if (InvalidApplicationException.class.getName().equals(
                                                                  data.getExceptionclass())) {
      JobMessageCodes.AKCJC5121E_INVALID_APPLICATION_EXCEPTION.log(data
                                                                   .getParameters().toArray());
      throw new InvalidApplicationException(data.getParameters()
                                            .toArray(), e);
    } else if (InvalidAttachmentsException.class.getName().equals(
                                                                  data.getExceptionclass())) {
      JobMessageCodes.AKCJC5122E_INVALID_ATTACHMENTS_EXCEPTION.log(data
                                                                   .getParameters().toArray());
      throw new InvalidAttachmentsException(data.getParameters()
                                            .toArray(), e);
    } else if (MissingAttachmentsException.class.getName().equals(
                                                                  data.getExceptionclass())) {
      JobMessageCodes.AKCJC5123E_MISSING_ATTACHMENTS_EXCEPTION.log(data
                                                                   .getParameters().toArray());
      throw new MissingAttachmentsException(data.getParameters()
                                            .toArray(), e);
    } else if (UnrecognizedAttachmentsException.class.getName().equals(
                                                                       data.getExceptionclass())) {
      JobMessageCodes.AKCJC5124E_UNRECOGNIZED_ATTACHMENTS_EXCEPTION
      .log(data.getParameters().toArray());
      throw new UnrecognizedAttachmentsException(data.getParameters()
                                                 .toArray(), e);
    } else if (IllegalJobStatusException.class.getName().equals(
                                                                data.getExceptionclass())) {
      JobMessageCodes.AKCJC5126E_JOB_STATUS_EXCEPTION.log(data
                                                          .getParameters().toArray());
      throw new IllegalJobStatusException(data.getParameters().toArray(),
                                          e);
    } else if (FormatNotSupportedException.class.getName().equals(
                                                                  data.getExceptionclass())) {
      JobMessageCodes.AKCJC5128E_FORMAT_NOT_SUPPORTED_EXCEPTION.log(data
                                                                    .getParameters().toArray());
      throw new FormatNotSupportedException(data.getParameters().toArray(),
                                            e);
    } else if (InvalidAttachmentNameException.class.getName().equals(
                                                                     data.getExceptionclass())) {
      JobMessageCodes.AKCJC5129E_INVALID_ATTACHMENT_NAME_EXCEPTION.log(data
                                                                       .getParameters().toArray());
      throw new InvalidAttachmentNameException(data.getParameters().toArray(),
                                               e);
    } else if (EmptyAttachmentException.class.getName().equals(
                                                               data.getExceptionclass())) {
      JobMessageCodes.AKCJC5130E_EMPTY_ATTACHMENT_EXCEPTION.log(data
                                                                .getParameters().toArray());
      throw new EmptyAttachmentException(data.getParameters().toArray(),
                                         e);
    }
  }

  public boolean deleteJob(String id) throws OperationException {
    Validate.notNull(id, "id");
    DeletionStatus status = deleteWithStatus(getBaseUrl() + "jobs/" + encodeName(id));
    if (status.equals(DeletionStatus.DELETED))
      JobMessageCodes.AKCJC5213I_JOB_DELETED.log(id);
    return status.equals(DeletionStatus.DELETED);
  }

  public JobExecutionStatus getJobExecutionStatus(String id)
      throws OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      JobStatus status =  get(getBaseUrl() + "jobs/" + encodeName(id) + "/execute",
                              JobStatusImpl.class, true);
      return status==null?null:status.getExecutionStatus();
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public void executeJob(String id) throws OperationException,
  JobNotFoundException, SubscriptionException, ValidationException {
    Validate.notNull(id, "id");
    try {
      post(getBaseUrl() + "jobs/" + encodeName(id) + "/execute", null);
      JobMessageCodes.AKCJC5203I_JOB_SUBMITTED.log(id);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }
  }

  public void abortJob(String id) throws OperationException,
  JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      delete(getBaseUrl() + "jobs/" + encodeName(id) + "/execute");
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public void killJob(String id) throws OperationException,
  JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      delete(getBaseUrl() + "jobs/" + encodeName(id) + "/execute?kill=true");
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public List<JobAttachmentImpl> getJobAttachments(String id)
      throws OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      return getList(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments",
                     new TypeReference<List<JobAttachmentImpl>>() {
      });
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public List<JobAttachmentImpl> getJobAttachments(String id, JobAttachmentType type)
      throws OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(type, "type");
    try {
      String uri = buildURI(
                            getBaseUrl() + "jobs/" + encodeName(id) + "/attachments", "type",
                            type.name());

      return getList(uri,
                     new TypeReference<List<JobAttachmentImpl>>() {
      });
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }
  public void deleteJobAttachments(String id) throws OperationException,
  JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      delete(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments");
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public String createJobAttachment(String id, JobAttachmentCreationData att)
      throws OperationException, JobNotFoundException,
      SubscriptionException, ValidationException {
    Validate.notNull(id, "id");
    Validate.notNull(att, "data");
    try {
      return create(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments", att);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkSubscription(e);
      checkValidation(e);
      throw e;
    }
  }

  public JobAttachmentImpl getJobAttachment(String id, String attid)
      throws OperationException, JobNotFoundException,
      AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    try {
      return get(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid),
                 JobAttachmentImpl.class, true);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public boolean deleteJobAttachment(String id, String attid)
      throws OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    try {
      DeletionStatus status = deleteWithStatus(getBaseUrl() + "jobs/"
          + encodeName(id) + "/attachments/" + encodeName(attid));
      return status.equals(DeletionStatus.DELETED);
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public void uploadJobAttachment(String id, String attid, File archive)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException, SubscriptionException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(archive, "file");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5208F_ATTACHMENT_UPLOADING.log(id, attid);
      putFile(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)
      + "/blob", archive);
      JobMessageCodes.AKCJC5202I_ATTACHMENT_UPLOADED.log(id, attid,
                                                         System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      checkSubscription(e);
      throw e;
    }
  }

  protected String encodeName(Object key){
    String encodedKey=null;
    try {
      encodedKey = URLEncoder.encode(key.toString(),"UTF-8");
      encodedKey=encodedKey.replace("+", "%20");
    } catch (Exception e) {
      // should not happen
    }
    return encodedKey;
  }

  public void uploadJobAttachment(String id, String attid, InputStream archive)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException, SubscriptionException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(archive, "stream");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5208F_ATTACHMENT_UPLOADING.log(id, attid);
      putStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)+ "/blob", archive);
      JobMessageCodes.AKCJC5202I_ATTACHMENT_UPLOADED.log(id, attid,
                                                         System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      checkSubscription(e);
      throw e;
    } 
  }

  public void uploadJobAttachment(String id, String attid,
                                  ObjectMapper mapper, Object obj) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException, SubscriptionException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(mapper, "mapper");
    Validate.notNull(obj, "obj");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5208F_ATTACHMENT_UPLOADING.log(id, attid);
      putStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)
      + "/blob", mapper, obj);
      JobMessageCodes.AKCJC5202I_ATTACHMENT_UPLOADED.log(id, attid,
                                                         System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      checkSubscription(e);
      throw e;
    }
  }

  public void downloadJobAttachment(String id, String attid, File archive)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(archive, "file");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5209F_ATTACHMENT_DOWNLOADING.log(id, attid);
      getFile(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)
      + "/blob", archive);
      JobMessageCodes.AKCJC5210I_ATTACHMENT_DOWNLOADED.log(id, attid,
                                                           System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public void downloadJobAttachment(String id, String attid,
                                    OutputStream stream) throws IOException, OperationException,
  JobNotFoundException, AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(stream, "stream");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5209F_ATTACHMENT_DOWNLOADING.log(id, attid);
      getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)
      + "/blob", stream);
      JobMessageCodes.AKCJC5210I_ATTACHMENT_DOWNLOADED.log(id, attid,
                                                           System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public InputStream downloadJobAttachment(String id, String attid)
      throws IOException, OperationException, JobNotFoundException,
      AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    try {
      JobMessageCodes.AKCJC5209F_ATTACHMENT_DOWNLOADING.log(id, attid);
      return getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/"
          + encodeName(attid) + "/blob");
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public <T> T downloadJobAttachment(String id, String attid,
                                     ObjectMapper mapper, TypeReference<T> type) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(mapper, "mapper");
    Validate.notNull(type, "type");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5209F_ATTACHMENT_DOWNLOADING.log(id, attid);
      T res = getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/"
          + encodeName(attid) + "/blob", mapper, type);
      JobMessageCodes.AKCJC5210I_ATTACHMENT_DOWNLOADED.log(id, attid,
                                                           System.currentTimeMillis() - start);
      return res;
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public <T> T downloadJobAttachment(String id, String attid,
                                     ObjectMapper mapper, Class<T> type) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(mapper, "mapper");
    Validate.notNull(type, "type");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5209F_ATTACHMENT_DOWNLOADING.log(id, attid);
      T res = getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/"
          + encodeName(attid) + "/blob", mapper, type);
      JobMessageCodes.AKCJC5210I_ATTACHMENT_DOWNLOADED.log(id, attid,
                                                           System.currentTimeMillis() - start);
      return res;
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      throw e;
    }
  }

  public void downloadLog(String id, File archive) throws IOException,
  OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(archive, "file");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5211F_LOG_DOWNLOADING.log(id);
      getFile(getBaseUrl() + "jobs/" + encodeName(id) + "/log/blob", archive);
      JobMessageCodes.AKCJC5212I_LOG_DOWNLOADED.log(id,
                                                    System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public void downloadLog(String id, OutputStream stream) throws IOException,
  OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    Validate.notNull(stream, "stream");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5211F_LOG_DOWNLOADING.log(id);
      getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/log/blob", stream);
      JobMessageCodes.AKCJC5212I_LOG_DOWNLOADED.log(id,
                                                    System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public InputStream downloadLog(String id) throws IOException,
  OperationException, JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      JobMessageCodes.AKCJC5211F_LOG_DOWNLOADING.log(id);
      return getStream(getBaseUrl() + "jobs/" + encodeName(id) + "/log/blob");
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public JobFailureInfo getFailureInfo(String id) throws OperationException,
  JobNotFoundException {
    Validate.notNull(id, "id");
    try {
      return get(getBaseUrl() + "jobs/" + encodeName(id) + "/failure",
                 JobFailureInfoImpl.class, false);
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public List<JobLogItemImpl> getJobLogItems(String jobid)
      throws OperationException, JobNotFoundException {
    Validate.notNull(jobid, "id");
    try {
      return getList(getBaseUrl() + "jobs/" + encodeName(jobid) + "/log/items",
                     new TypeReference<List<JobLogItemImpl>>() {
      });
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public List<JobLogItemImpl> getJobLogItems(String jobid, long start)
      throws OperationException, JobNotFoundException {
    Validate.notNull(jobid, "id");
    try {
      String uri = buildURI(
                            getBaseUrl() + "jobs/" + encodeName(jobid) + "/log/items", "start",
                            Long.toString(start));
      return getList(uri, new TypeReference<List<JobLogItemImpl>>() {
      });
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  public List<? extends JobLogItem> getJobLogItems(String jobid, long start, boolean continuous)
      throws OperationException, JobNotFoundException{
    Validate.notNull(jobid, "id");
    try {
      String uri = buildURI(
                            getBaseUrl() + "jobs/" + encodeName(jobid) + "/log/items", "start",
                            Long.toString(start), "continuous",continuous);
      return getList(uri, new TypeReference<List<JobLogItemImpl>>() {
      });
    } catch (OperationException e) {
      checkJobNotFound(e);
      throw e;
    }
  }

  @Override
  public JobCreationDataImpl newJobCreationData() {
    return new JobCreationDataImpl();
  }

  @Override
  public JobAttachmentImpl newInputJobAttachment(String name) {
    Validate.notNull(name, "name");
    return new JobAttachmentImpl(name);
  }

  @Override
  public JobRequestBuilderImpl newRequest() {
    return new JobRequestBuilderImpl(this);
  }



  static class InternalCustomEntity extends AbstractHttpEntity {
    AttachmentContentWriter entity;

    public InternalCustomEntity(AttachmentContentWriter entity) {
      this.entity=entity;
    }

    @Override
    public boolean isRepeatable() {
      return entity.isRepeatable();
    }

    @Override
    public long getContentLength() {
      return -1;
    }

    @Override
    public boolean isStreaming() {
      return false;
    }

    @Override
    public InputStream getContent() throws IOException {
      throw new UnsupportedOperationException();
    }

    @Override
    public void writeTo(final OutputStream outstream) throws IOException {
      entity.writeTo(outstream);
    }

  };


  protected void putStream(String uri, AttachmentContentWriter entity) throws OperationException {
    HttpPut request = new HttpPut(uri);
    try {
      request.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_OCTET_STREAM);
      if (compression){
        request.setEntity(new GzipCompressingEntity(new InternalCustomEntity(entity)));
      } else {
        request.setEntity(new InternalCustomEntity(entity)); 
      }
      try (CloseableHttpResponse response = execute(request)) {
        switch (response.getStatusLine().getStatusCode()) {
          case HttpStatus.SC_NO_CONTENT:
            return;
          default:
            handleStatus(request, response);
            return;
        }
      }
    } catch (OperationException e) {
      throw e;
    } catch (Exception e) {
      throwOperationException(request, e);
    }
  }

  public void uploadJobAttachment(String id, String attid,
                                  AttachmentContentWriter writer) throws IOException,
  OperationException, JobNotFoundException,
  AttachmentNotFoundException, SubscriptionException {
    Validate.notNull(id, "id");
    Validate.notNull(attid, "attid");
    Validate.notNull(writer, "writer");
    try {
      long start = System.currentTimeMillis();
      JobMessageCodes.AKCJC5208F_ATTACHMENT_UPLOADING.log(id, attid);
      putStream(getBaseUrl() + "jobs/" + encodeName(id) + "/attachments/" + encodeName(attid)
      + "/blob", writer);
      JobMessageCodes.AKCJC5202I_ATTACHMENT_UPLOADED.log(id, attid,
                                                         System.currentTimeMillis() - start);
    } catch (OperationException e) {
      checkJobNotFound(e);
      checkAttNotFound(e);
      checkSubscription(e);
      throw e;
    }
  }

  public void dropAll() throws OperationException {
    String uri = buildURI(getBaseUrl() + "admin/jobs/drop");
    post(uri, null);
  }
}