/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.impl;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.ibm.optim.oaas.client.impl.Validate;
import com.ibm.optim.oaas.client.job.JobExecutor;
import com.ibm.optim.oaas.client.job.JobExecutorBuilder;

public class JobExecutorBuilderImpl implements JobExecutorBuilder {
  private long timeout = DEFAULT_TIMEOUT;
  private long interval = DEFAULT_INTERVAL;
  private int threads = DEFAULT_THREADS;
  private int retry = DEFAULT_RETRY;
  private long retryDelay = DEFAULT_RETRY_DELAY;

  private ExecutorService service;

  @Override
  public JobExecutorBuilder threads(int threads) {
    Validate.strictlyPositive(threads, "threads");
    this.threads = threads;
    return this;
  }

  @Override
  public JobExecutorBuilder interval(long interval, TimeUnit unit) {
    Validate.strictlyPositive(interval, "interval");
    Validate.notNull(unit,  "unit");
    this.interval = unit.toMillis(interval);
    return this;
  }

  @Override
  public JobExecutorBuilder timeout(long timeout, TimeUnit unit) {
    Validate.notNull(unit,  "unit");
    if (timeout>0)
      this.timeout=unit.toMillis(timeout);
    else
      this.timeout=timeout;
    return this;
  }

  @Override
  public JobExecutorBuilder service(ExecutorService service) {
    Validate.notNull(service, "service");
    this.service = service;
    return this;
  }

  @Override
  public JobExecutorBuilder retry(int retry) {
    this.retry = retry;
    return this;
  }

  @Override
  public JobExecutorBuilder retryDelay(long delay, TimeUnit unit) {
    Validate.strictlyPositive(delay, "delay");
    Validate.notNull(unit,  "unit");
    this.retryDelay = unit.toMillis(delay);
    return this;
  }

  @Override
  public JobExecutor build() {
    ExecutorService serviceConfig = service;
    if (serviceConfig == null) {
      serviceConfig = Executors.newFixedThreadPool(threads);
    }
    return new JobExecutorImpl(serviceConfig, interval, timeout, retry, retryDelay);
  }
}