/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import com.ibm.icu.text.DateFormat;
import com.ibm.optim.oaas.client.OaasException;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.impl.ClientMessageCodes;
import com.ibm.optim.oaas.client.job.AttachmentNotFoundException;
import com.ibm.optim.oaas.client.job.JobCallback;
import com.ibm.optim.oaas.client.job.JobClient;
import com.ibm.optim.oaas.client.job.JobExecutor;
import com.ibm.optim.oaas.client.job.JobInput;
import com.ibm.optim.oaas.client.job.JobNotFoundException;
import com.ibm.optim.oaas.client.job.JobOutput;
import com.ibm.optim.oaas.client.job.JobRequest;
import com.ibm.optim.oaas.client.job.JobResponse;
import com.ibm.optim.oaas.client.job.SubscriptionException;
import com.ibm.optim.oaas.client.job.ValidationException;
import com.ibm.optim.oaas.client.job.model.JobAttachment;
import com.ibm.optim.oaas.client.job.model.JobAttachmentType;
import com.ibm.optim.oaas.client.job.model.JobCreationData;
import com.ibm.optim.oaas.client.job.model.JobExecutionStatus;
import com.ibm.optim.oaas.client.job.model.JobLogItem;
import com.ibm.optim.oaas.client.job.model.JobLogRecord;
import com.ibm.optim.oaas.client.job.model.impl.JobAttachmentImpl;
import com.ibm.optim.oaas.client.job.model.impl.JobImpl;

public class JobExecutorImpl implements JobExecutor {
  private long timeout;
  private long interval;
  private int retry;
  private long retryDelay;

  private ExecutorService service;

  class RetryLoop {
    int _count = 1;
    long delay = retryDelay;

    public RetryLoop() {
    }

    boolean shouldRetry() {
      return _count <= retry;
    }

    public void next() {
      _count++;
    }

    public void exception(OperationException e) throws OperationException {
      if (_count == retry) {
        throw e;
      } else {
        if (e.getCode() == 0 && e.getCause() != null) {
          ClientMessageCodes.AKCJC5006I_RETRY_OPERATION_CAUSE_EXCEPTION.log(e.getCause()
            .getLocalizedMessage(), e.getOperation(), e.getURI(), _count);
        } else {
          ClientMessageCodes.AKCJC5003I_RETRY_OPERATION_EXCEPTION.log(e.getCode(),
            e.getOperation(), e.getURI(), _count);
        }
      }
      try {
        Thread.sleep(delay);
        delay *= 2;
      }
      catch (InterruptedException e1) {

      }
    }
  }

  public JobExecutorImpl(ExecutorService service, long interval, long timeout, int retry,
                         long retryDelay) {
    this.service = service;
    this.interval = interval;
    this.timeout = timeout;
    this.retry = retry;
    this.retryDelay = retryDelay;
  }

  public void start() {

  }

  @Override
  public void shutdown() {
    service.shutdown();
  }

  @Override
  public Future<JobResponse> create(final JobRequest request, final JobCallback callback)
                                                                                         throws OperationException,
                                                                                         IOException,
                                                                                         InterruptedException,
                                                                                         SubscriptionException,
                                                                                         ValidationException,
                                                                                         JobNotFoundException {
    return executeMain(request, callback, true, false, false);
  }

  @Override
  public Future<JobResponse> submit(final JobRequest request, final JobCallback callback)
                                                                                         throws OperationException,
                                                                                         IOException,
                                                                                         InterruptedException,
                                                                                         SubscriptionException,
                                                                                         ValidationException,
                                                                                         JobNotFoundException {

    return executeMain(request, callback, true, true, false);
  }

  @Override
  public Future<JobResponse> monitor(final JobRequest request, String jobid,
                                     final JobCallback callback) throws OperationException,
                                                                IOException, InterruptedException,
                                                                SubscriptionException,
                                                                ValidationException,
                                                                JobNotFoundException {

    JobRequestImpl r = (JobRequestImpl) request;
    r.setJobId(jobid);

    return executeMain(r, callback, false, false, true);
  }

  @Override
  public Future<JobResponse> execute(final JobRequest request, final JobCallback callback)
                                                                                          throws OperationException,
                                                                                          IOException,
                                                                                          InterruptedException,
                                                                                          SubscriptionException,
                                                                                          ValidationException,
                                                                                          JobNotFoundException {
    return executeMain(request, callback, true, true, true);
  }

  public Future<JobResponse> executeMain(final JobRequest request, final JobCallback callback,
                                         final boolean create, final boolean submit,
                                         final boolean monitor) throws OperationException,
                                                               IOException, InterruptedException,
                                                               SubscriptionException,
                                                               ValidationException,
                                                               JobNotFoundException {

    final JobRequestImpl r = (JobRequestImpl) request;
    Future<JobResponse> result = service.submit(new Callable<JobResponse>() {
      @Override
      public JobResponse call() throws Exception {

        final JobResponseImpl response = new JobResponseImpl(r.getClient(), r.getData(), r
          .getOutput(), r.getLogOutput());
        return executeImpl((JobRequestImpl) request, callback, response, create, submit, monitor);
      }
    });
    r.setSubmitted(true);
    return result;

  }

  @Override
  public Future<JobResponse> execute(final JobRequest request) throws OperationException,
                                                              IOException, InterruptedException,
                                                              SubscriptionException,
                                                              ValidationException,
                                                              JobNotFoundException {
    return execute(request, null);
  }

  protected String createJob(JobClient client, JobCreationData data) throws SubscriptionException,
                                                                    ValidationException,
                                                                    OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        return client.createJob(data);
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
    return null; // should not get here
  }

  protected String batchSubmitJob(JobClientImpl client, JobCreationData data, List<JobInput> inputs)
                                                                                                    throws SubscriptionException,
                                                                                                    ValidationException,
                                                                                                    OperationException {

    Object[] atts = new Object[inputs.size()];
    for (int i = 0; i < inputs.size(); i++) {
      JobInput input = inputs.get(i);
      if (input instanceof JobStreamInputImpl) {
        atts[i] = ((JobStreamInputImpl) input).getInputStream();
      } else if (input instanceof JobFileInputImpl) {
        atts[i] = ((JobFileInputImpl) input).getFile();
      } else {
        throw new IllegalArgumentException(
          "Unsupported JobInput type for '" + input.getName()
              + "', only File and InputStream attachments are supported in batch mode.");
      }
    }
    return client.submitJob(data, atts);

  }

  protected String copyJob(JobClient client, String jobid, JobCreationData data, boolean shallow)
                                                                                                 throws SubscriptionException,
                                                                                                 ValidationException,
                                                                                                 OperationException,
                                                                                                 JobNotFoundException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        return client.copyJob(jobid, data, shallow);
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
    return null; // should not get here
  }

  protected String recreateJob(JobClient client, String jobid, JobCreationData data)
                                                                                    throws SubscriptionException,
                                                                                    ValidationException,
                                                                                    OperationException,
                                                                                    JobNotFoundException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        return client.recreateJob(jobid, data, false);
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
    return null; // should not get here
  }

  protected void upload(JobClientImpl client, String jobid, JobInput input)
                                                                           throws JobNotFoundException,
                                                                           AttachmentNotFoundException,
                                                                           SubscriptionException,
                                                                           IOException,
                                                                           OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        input.upload(client, jobid);
        return;
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
  }

  protected void download(JobClientImpl client, String jobid, JobOutput output)
                                                                               throws JobNotFoundException,
                                                                               AttachmentNotFoundException,
                                                                               SubscriptionException,
                                                                               IOException,
                                                                               OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        output.download(client, jobid);
        return;
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
  }

  protected void downloadLog(JobClientImpl client, String jobid, JobLogOutputImpl output)
                                                                                         throws JobNotFoundException,
                                                                                         AttachmentNotFoundException,
                                                                                         SubscriptionException,
                                                                                         IOException,
                                                                                         OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        output.download(client, jobid);
        return;
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
  }

  protected void submit(JobClientImpl client, String jobid) throws JobNotFoundException,
                                                           SubscriptionException,
                                                           ValidationException, OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        client.executeJob(jobid);
        return;
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
  }

  protected JobImpl getJob(JobClientImpl client, String jobid) throws JobNotFoundException,
                                                              SubscriptionException,
                                                              ValidationException,
                                                              OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        return client.getJob(jobid);
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
    return null; // should not get here
  }

  protected void deleteJob(JobClientImpl client, String jobid) throws JobNotFoundException,
                                                              SubscriptionException,
                                                              ValidationException,
                                                              OperationException {
    for (RetryLoop loop = new RetryLoop(); loop.shouldRetry(); loop.next()) {
      try {
        client.deleteJob(jobid);
        return;
      }
      catch (OperationException e) {
        loop.exception(e);
      }
    }
  }

  protected JobResponse executeImpl(JobRequestImpl request, JobCallback callback,
                                    JobResponseImpl response, boolean create, boolean submit,
                                    boolean monitor) throws OaasException, IOException,
                                                    InterruptedException {

    try {
      JobClientImpl client = request.getClient();
      // creates the job in the calling thread to get the jobId right away.
      String jobid = null;
      boolean submitted = false;
      if (create) {
        boolean upload = true;
        try {
          if (request.getCopyJobId() != null) {
            jobid = copyJob(client, request.getCopyJobId(), request.getData(),
              request.getShallowCopy());
          } else if (request.getRecreateJobId() != null) {
            jobid = recreateJob(client, request.getRecreateJobId(), request.getData());
          } else if (request.isBatchSubmitMode()) {
            jobid = batchSubmitJob(client, request.getData(), request.getInput());
            submitted = true;
            upload = submit = false;
          } else {
            jobid = createJob(client, request.getData());
          }
        }
        finally {
          request.setJobId(jobid);
          response.setJobId(request.getJobId());
          if (jobid != null && callback != null) {
            callback.created(response);
          }
        }
        if (upload) { // uploads content
          for (JobInput input : request.getInput()) {
            if (input.isRepeatable()) {
              upload(client, jobid, input);
            } else {
              input.upload(client, jobid);
            }
          }
        }
      }
      response.setJobId(request.getJobId());
      jobid = request.getJobId();
      if (submit) { // starts the job
        submit(client, jobid);
        submitted = true;
      }
      if (submitted && callback != null) {
        callback.submitted(response);
      }
      if (monitor) {
        waitForCompletion(client, request, jobid,
          request.getTimeout() == 0 ? timeout : request.getTimeout(), callback, response);
        JobImpl job = response.getJob();
        JobMessageCodes.AKCJC5215I_JOB_ENDED.log(jobid,
          (job.getSubmittedAt() == null || job.getCreatedAt() == null) ? 0 : job.getSubmittedAt()
            .getTime() - job.getCreatedAt().getTime(),
          (job.getStartedAt() == null || job.getSubmittedAt() == null) ? 0 : job.getStartedAt()
            .getTime() - job.getSubmittedAt().getTime(),
          (job.getEndedAt() == null || job.getStartedAt() == null) ? 0 : job.getEndedAt().getTime()
                                                                         - job.getStartedAt()
                                                                           .getTime());

        JobAttachment result = null;
        for (JobAttachmentImpl att : job.getImplAttachments()) {
          if (JobAttachmentType.OUTPUT_ATTACHMENT.equals(att.getType())) {
            result = att;
            break;
          }
        }
        if (result != null) { // download results
          for (JobOutput output : request.getOutput()) {
            if (output.getName() == null) {
              output.setName(result.getName());
            }
            download(client, jobid, output);
          }
        }
        // download logs
        for (JobLogOutputImpl output : request.getLogOutput()) {
          downloadLog(client, jobid, output);
        }
        // delete
        if (request.isDeleteOnCompletion()) {
          deleteJob(client, jobid);
        }
        if (callback != null) {
          callback.completed(response);
        }
      }
      return response;
    }
    catch (Exception e) {
      if (callback != null) {
        callback.exception(response, e);
      }
      throw e;
    }
    catch (Throwable e) {
      JobMessageCodes.AKCJC5302E_INTERNAL_EXCEPTION.log(e, e.getLocalizedMessage());
      throw e;
    }

  }

  private JobExecutionStatus getStatus(JobClient client, String jobid) throws JobNotFoundException {
    try {
      return client.getJobExecutionStatus(jobid);
    }
    catch (OperationException e) {
      return null; // ignore this exception so that we continue (kind of
      // retry)
    }
  }

  private List<? extends JobLogItem> getLogItems(JobClient client, String jobid, long start)
                                                                                            throws JobNotFoundException {
    try {
      return client.getJobLogItems(jobid, start, true);
    }
    catch (OperationException e) {
      return null; // ignore this exception so that we continue (kind of
      // retry)
    }
  }

  protected JobResponseImpl waitForCompletion(JobClientImpl client, JobRequest request,
                                              String jobid, long timeout, JobCallback callback,
                                              JobResponseImpl response) throws OperationException,
                                                                       InterruptedException,
                                                                       JobNotFoundException,
                                                                       SubscriptionException,
                                                                       ValidationException,
                                                                       IOException {

    long limit = System.currentTimeMillis() + timeout;
    JobExecutionStatus status = getStatus(client, jobid);
    boolean running = false;
    long sleepTime = interval;
    if (timeout > 0) {
      sleepTime = Math.min(interval, timeout);
    }

    if (request.getLivelog() == null) {
      // if no live logs then we wait until we have a completion status
      while (!JobExecutionStatus.isEnded(status)) {
        if (JobExecutionStatus.RUNNING.equals(status) && !running) {
          running = true; // call once
          JobMessageCodes.AKCJC5204I_JOB_RUNNING.log(jobid);
          if (callback != null) {
            callback.running(response);
          }
        }
        if (timeout >= 0 && System.currentTimeMillis() > limit) {
          JobMessageCodes.AKCJC5216W_JOB_MONITORING_TIMEOUT.log(jobid, timeout);
          throw new InterruptedException(
            JobMessageCodes.AKCJC5216W_JOB_MONITORING_TIMEOUT.extractMessage(jobid, timeout));
        }
        Thread.sleep(sleepTime);
        status = getStatus(client, jobid);
      }
    } else {
      // if we have live logs, we consume them until the end
      long itemIndex = 0; // current log item to get
      boolean stop = false; // flag set to true when no more logs

      // writer to write logs
      Writer writer = new BufferedWriter(new OutputStreamWriter(request.getLivelog()));

      while (!stop) {
        if (JobExecutionStatus.RUNNING.equals(status) && !running) {
          running = true; // call once
          JobMessageCodes.AKCJC5204I_JOB_RUNNING.log(jobid);
          if (callback != null) {
            callback.running(response);
          }
        }

        if (timeout >= 0 && System.currentTimeMillis() > limit) {
          JobMessageCodes.AKCJC5216W_JOB_MONITORING_TIMEOUT.log(jobid, timeout);
          throw new InterruptedException(
            JobMessageCodes.AKCJC5216W_JOB_MONITORING_TIMEOUT.extractMessage(jobid, timeout));
        }

        List<? extends JobLogItem> items = getLogItems(client, jobid, itemIndex);
        if (items != null && !items.isEmpty()) {
          // if not empty, format logs and move to next block
          formatLogItems(items, writer, request.getLivelogDateFormat());
          JobLogItem lastItem = items.get(items.size() - 1);
          itemIndex = lastItem.getSeqid() + 1;
          stop = lastItem.stop();
        } else {
          // we have no new logs, so we sleep
          Thread.sleep(sleepTime);
        }

        if (!running) {
          status = getStatus(client, jobid);
        }
      }
    }

    JobImpl job = getJob(client, jobid);
    status = job.getExecutionStatus();
    response.setJob(job);
    if (JobExecutionStatus.PROCESSED.equals(status)) {
      JobMessageCodes.AKCJC5205I_JOB_PROCESSED.log(jobid);
      if (callback != null) {
        callback.processed(response);
      }
    } else if (JobExecutionStatus.FAILED.equals(status)) {
      JobMessageCodes.AKCJC5206I_JOB_FAILED.log(jobid);
      if (callback != null) {
        callback.failed(response);
      }
    } else if (JobExecutionStatus.INTERRUPTED.equals(status)
               || JobExecutionStatus.INTERRUPTING.equals(status)) {
      JobMessageCodes.AKCJC5207I_JOB_INTERRUPTED.log(jobid);
      if (callback != null) {
        callback.interruption(response);
      }
    }

    return response;
  }

  private void formatLogItems(List<? extends JobLogItem> items, Writer writer, DateFormat format)
                                                                                                 throws IOException {
    for (JobLogItem item : items) {
      if (item.missing()) {
        writer.write(JobMessageCodes.AKCJC5241I_LOG_ITEM_NOT_AVAILABLE.extractMessage(item
          .getSeqid()));
        writer.write("\n");
      } else {
        for (JobLogRecord record : item.getEngineLogRecords()) {
          writer.write("[");
          if (format != null) {
            writer.write(format.format(record.getDate()));
          } else {
            writer.write(record.getDate().toString());
          }
          writer.write(", ");
          writer.write(record.getLevel().length() > 4 ? record.getLevel().substring(0, 4) : record
            .getLevel());
          writer.write("] ");
          writer.write(stripCRLF(record.getMessage()));
          writer.write("\r\n");
        }
      }
    }

    writer.flush();
  }

  private String stripCRLF(String text) {
    while (text.endsWith("\r") || text.endsWith("\n")) {
      text = text.substring(0, text.length() - 1);
    }
    return text;
  }
}