/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.impl;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.TimeZone;
import com.ibm.optim.oaas.client.OperationException;
import com.ibm.optim.oaas.client.job.JobException;
import com.ibm.optim.oaas.client.job.JobExecutor;
import com.ibm.optim.oaas.client.job.JobInput;
import com.ibm.optim.oaas.client.job.JobOutput;
import com.ibm.optim.oaas.client.job.JobRequest;
import com.ibm.optim.oaas.client.job.JobResponse;
import com.ibm.optim.oaas.client.job.model.impl.JobCreationDataImpl;

public class JobRequestImpl implements JobRequest {
  public static final DateFormat UTC_ISO8601_DATE_FORMAT;

  static{
    TimeZone tz = TimeZone.getTimeZone("UTC");
    DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    df.setTimeZone(tz);
    UTC_ISO8601_DATE_FORMAT = df;
  }

  private JobClientImpl client;
  private JobCreationDataImpl data;
  private List<JobInput> input;
  private List<JobOutput> output = new ArrayList<JobOutput>();
  private List<JobLogOutputImpl> logoutput = new ArrayList<JobLogOutputImpl>();
  private boolean delete;
  private long timeout;
  private String jobId;
  private OutputStream livelog;
  private DateFormat livelogDateFormat = UTC_ISO8601_DATE_FORMAT;
  private String copy;
  private boolean shallow;
  private String recreate;
  private final CountDownLatch set = new CountDownLatch(1);
  private volatile boolean submitted = false;
  private boolean batchSubmitMode;

  public JobRequestImpl(JobClientImpl client, JobCreationDataImpl data,
                        List<JobInput> input, List<JobOutput> output,
                        List<JobLogOutputImpl> logoutput, boolean delete, long timeout, OutputStream livelog, DateFormat livelogDateFormat, String copy, boolean shallow, String recreate, boolean batchSubmitMode) {
    super();
    this.client = client;
    this.data = data;
    this.input = input;
    this.output = output;
    this.logoutput = logoutput;
    this.delete = delete;
    this.timeout=timeout;
    this.livelog=livelog;
    if (livelogDateFormat!=null) {
      this.livelogDateFormat=livelogDateFormat;
    }
    this.copy=copy;
    this.shallow=shallow;
    this.recreate=recreate;
    this.batchSubmitMode=batchSubmitMode;
  }

  @Override
  public JobClientImpl getClient() {
    return client;
  }

  @Override
  public JobCreationDataImpl getData() {
    return data;
  }

  @Override
  public List<JobInput> getInput() {
    return input;
  }

  @Override
  public List<JobOutput> getOutput() {
    return output;
  }

  @Override
  public List<JobLogOutputImpl> getLogOutput() {
    return logoutput;
  }

  @Override
  public boolean isDeleteOnCompletion() {
    return delete;
  }

  @Override
  public long getTimeout(){
    return timeout;
  }

  @Override
  public OutputStream getLivelog(){
    return livelog;
  }

  @Override
  public Future<JobResponse> execute(JobExecutor executor)
      throws OperationException, IOException, InterruptedException, JobException {
    return executor.execute(this);
  }

  @Override
  public Future<JobResponse> create(JobExecutor executor)
      throws OperationException, IOException, InterruptedException, JobException {
    return executor.create(this,null);
  }

  @Override
  public String getJobId() {
    if ( submitted) {
      try {
        set.await();
      } catch (InterruptedException e) {
      }
    }
    return jobId;
  }

  public void setJobId(String jobid) {
    this.jobId = jobid;
    set.countDown();
  }

  @Override
  public DateFormat getLivelogDateFormat() {
    return livelogDateFormat;
  }


  @Override
  public String getCopyJobId() {
    return copy;
  }

  @Override
  public boolean getShallowCopy() {
    return shallow;
  }


  @Override
  public String getRecreateJobId() {
    return recreate;
  }

  void setSubmitted(boolean submitted) {
    this.submitted = submitted;
  }

  public boolean isBatchSubmitMode() {
    return batchSubmitMode;
  }
}
