/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.impl;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

/**
 * This class provides utility methods to browse through a results file returned in JSON format from
 * the server when interfacing with the java client API.
 * On the technical side, this class uses a streaming API when parsing JSON documents, in order to
 * minimize the memory footprint.
 */
public class JsonResultsReader {
  private final File jsonResultsFile;
  private final List<TableDescriptor> tablesDescriptor = new ArrayList<>();
  private final List<String> tablesName = new ArrayList<>();
  private final Map<String, TableDescriptor> tablesDescriptorByName = new LinkedHashMap<>();

  /**
   * 
   */
  public interface TableDescriptor {
    public String getName();
    public List<ColumnDescriptor<?>> getColumnsDescriptor();
    public ColumnDescriptor<?> getColumnDescriptor(int index);
    public ColumnDescriptor<?> getColumnDescriptor(String columnName);
    public List<String> getColumnsName();
  }

  /**
   * 
   */
  private class TableDescriptorImpl implements TableDescriptor {
    private final String  name;
    private final List<ColumnDescriptor<?>> columnsDescriptor;
    private final List<String> columnsName = new ArrayList<>();
    private final Map<String, ColumnDescriptor<?>> columnsDescriptorByName = new LinkedHashMap<>();

    public TableDescriptorImpl(String name) {
      this.name = name;
      this.columnsDescriptor = new ArrayList<>();
    }

    public void addColumnDescriptor(ColumnDescriptor<?> desc) {
      columnsDescriptor.add(desc);
      columnsName.add(desc.getName());
      columnsDescriptorByName.put(desc.getName(), desc);
    }

    @Override
    public String getName() {
      return name;
    }

    @Override
    public List<ColumnDescriptor<?>> getColumnsDescriptor() {
      return columnsDescriptor;
    }

    @Override
    public ColumnDescriptor<?> getColumnDescriptor(int index) {
      return columnsDescriptor.get(index);
    }

    @Override
    public ColumnDescriptor<?> getColumnDescriptor(String columnName) {
      return columnsDescriptorByName.get(columnName);
    }

    @Override
    public List<String> getColumnsName() {
      return columnsName;
    }
  }

  /**
   * 
   */
  public interface ColumnDescriptor<T> {
    public int getIndex();
    public String getName();
    public Class<?> getType();
  }

  /**
   * 
   */
  private class ColumnDescriptorImpl<T> implements ColumnDescriptor<T> {
    private final int     index;
    private final String  name;
    private final Class<T>    type;

    public ColumnDescriptorImpl(int index, String name, Class<T> type) {
      this.name = name;
      this.index = index;
      this.type = type;
    }

    @Override
    public String getName() {
      return name;
    }

    @Override
    public int getIndex() {
      return index;
    }

    @Override
    public Class<?> getType() {
      return type;
    }
  }

  /**
   * 
   */
  public interface TableRow {
    public TableDescriptor getTableDesc();
    public Object getColumnValue(int index);
    public Object getColumnValue(String columnName);
  }

  /**
   * 
   */
  private static class TableRowImpl implements TableRow {
    private final TableDescriptor tableDesc;
    private final Object[]  values;

    public TableRowImpl(TableDescriptor tableDesc) {
      this.tableDesc = tableDesc;
      List<ColumnDescriptor<?>> columnsDescriptor = tableDesc.getColumnsDescriptor();
      this.values = new Object[columnsDescriptor.size()];
    }

    @Override
    public TableDescriptor getTableDesc() {
      return tableDesc;
    }

    public void setColumnValue(String columnName, Object value) {
      ColumnDescriptor<?> colDesc = tableDesc.getColumnDescriptor(columnName);
      values[colDesc.getIndex()] = value;
    }

    @Override
    public Object getColumnValue(int index) {
      return values[index];
    }

    @Override
    public Object getColumnValue(String columnName) {
      ColumnDescriptor<?> colDesc = tableDesc.getColumnDescriptor(columnName);
      return values[colDesc.getIndex()];
    }
  }

  /**
   * 
   */
  public interface TableRowIterator {
    public boolean hasNext();
    public TableRow next() throws IOException, JsonResultsReaderException;
  }

  /**
   * 
   */
  private static class TableRowIteratorImpl implements TableRowIterator {

    private enum ParsingEntity {
      ROOT, TABLE, ROW_SET, ROW;
    }

    private final JsonResultsReader reader;
    private final JsonParser jsonParser;
    private TableRow currentRow = null;
    private TableDescriptor currentTableDesc = null;
    private ParsingEntity  state = ParsingEntity.ROOT;

    public TableRowIteratorImpl(JsonResultsReader reader) throws IOException, JsonResultsReaderException {
      this.reader = reader;
      JsonFactory factory = new JsonFactory();
      this.jsonParser = factory.createParser(reader.jsonResultsFile);

      currentRow = buildNextTableRow();
    }

    @Override
    public boolean hasNext() {
      return (currentRow != null);
    }

    @Override
    public TableRow next() throws IOException, JsonResultsReaderException {
      TableRow result = currentRow;
      currentRow = buildNextTableRow();
      return result;
    }

    private TableRowImpl buildNextTableRow() throws IOException, JsonResultsReaderException {
      switch(state) {
        case ROOT:
          while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
            if (!jsonParser.hasCurrentToken()) {
              break;  // Input JSON file is empty
            }
            if (jsonParser.getCurrentToken() != JsonToken.FIELD_NAME) {
              continue;
            }
            String tableName = jsonParser.getCurrentName();
            currentTableDesc = reader.getTableDescriptor(tableName);
            if (currentTableDesc == null) {
              throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6003E_UNKNOWN_TABLE_NAME_EXCEPTION.name(),
                                                   new Object[] {tableName});
            }

            state = ParsingEntity.TABLE;
            return buildNextTableRow();
          }
          return null;

        case TABLE:
          JsonToken token = jsonParser.nextToken();
          if (token == JsonToken.START_ARRAY) {
            state = ParsingEntity.ROW_SET;
            return buildNextTableRow();

          } else if (token == JsonToken.START_OBJECT) {
            state = ParsingEntity.ROW;
            TableRowImpl row = buildNextTableRow();
            state = ParsingEntity.ROOT;
            return row;

          } else {
            throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6001E_INVALID_FORMAT_EXCEPTION.name(),
                                                 new Object[] {JsonToken.START_ARRAY, jsonParser.getCurrentToken()});
          }

        case ROW_SET:
          if (jsonParser.nextToken() != JsonToken.END_ARRAY) {
            state = ParsingEntity.ROW;
            TableRowImpl row = buildNextTableRow();
            state = ParsingEntity.ROW_SET;
            return row;
          }
          state = ParsingEntity.ROOT;
          return buildNextTableRow();

        case ROW:
        default:
          if (jsonParser.getCurrentToken() == JsonToken.START_OBJECT) {
            JsonObjectPrinter jop = new JsonObjectPrinter();
            TableRowImpl row = new TableRowImpl(currentTableDesc);
            while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
              String columnName = jsonParser.getCurrentName();

              JsonToken node = jsonParser.nextToken();
              jop.addProperty(columnName, jsonParser);
              if (node == JsonToken.VALUE_NUMBER_INT) {
                row.setColumnValue(columnName, jsonParser.getIntValue());

              } else if (node == JsonToken.VALUE_NUMBER_FLOAT) {
                row.setColumnValue(columnName, jsonParser.getDoubleValue());

              } else if (node == JsonToken.VALUE_STRING) {
                row.setColumnValue(columnName, jsonParser.getText());

              } else if (node == JsonToken.VALUE_TRUE || node == JsonToken.VALUE_FALSE) {
                row.setColumnValue(columnName, jsonParser.getBooleanValue());

              } else {
                throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6002E_UNEXPECTED_TYPE_EXCEPTION.name(),
                                                     new Object[] {columnName, currentTableDesc.getName(), jop});
              }
            }
            return row;

          } else {
            throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6001E_INVALID_FORMAT_EXCEPTION.name(),
                                                 new Object[] {JsonToken.START_OBJECT, jsonParser.getCurrentToken()});
          }
      }
    }
  }

  /**
   * At initialisation, the constructors parses the file provided as argument in order to build the
   * schema.
   * 
   * @param jsonResults
   * @throws IOException
   * @throws JsonResultsReaderException
   */
  public JsonResultsReader(File jsonResultsFile) throws IOException, JsonResultsReaderException {
    this.jsonResultsFile = jsonResultsFile;
    buildSchema();
  }

  public List<String> getTablesName() {
    return tablesName;
  }

  public List<TableDescriptor> getTablesDescriptor() {
    return tablesDescriptor;
  }

  public TableDescriptor getTableDescriptor(String tableName) {
    return tablesDescriptorByName.get(tableName);
  }

  public TableRowIterator getAllTablesRowsIterator() throws IOException, JsonResultsReaderException {
    if (jsonResultsFile != null && jsonResultsFile.exists()) {
      return new TableRowIteratorImpl(this);
    }
    return null;
  }


  private void addTableDescriptor(TableDescriptor tableDesc) {
    tablesDescriptorByName.put(tableDesc.getName(),  tableDesc);
    tablesDescriptor.add(tableDesc);
    tablesName.add(tableDesc.getName());
  }

  private void buildSchema() throws IOException, JsonResultsReaderException {
    if (jsonResultsFile != null && jsonResultsFile.exists()) {
      JsonFactory factory = new JsonFactory();
      JsonParser jsonParser = factory.createParser(jsonResultsFile);

      while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
        if (!jsonParser.hasCurrentToken()) {
          break;  // Input JSON file is empty
        }
        if (jsonParser.getCurrentToken() != JsonToken.FIELD_NAME) {
          continue;
        }
        String tableName = jsonParser.getCurrentName();
        TableDescriptorImpl tableDesc = new TableDescriptorImpl(tableName);
        addTableDescriptor(tableDesc);
        buildTableSchema(tableDesc, jsonParser);
      }
    }
  }

  private void buildTableSchema(TableDescriptorImpl tableDesc, JsonParser jsonParser) throws IOException, JsonResultsReaderException {
    JsonToken token = jsonParser.nextToken();
    if (token == JsonToken.START_ARRAY) {
      boolean firstRow = true;
      while (jsonParser.nextToken() != JsonToken.END_ARRAY) {
        if (firstRow) {
          buildSchemaFromObject(tableDesc, jsonParser);
        }
        firstRow = false;
      }
    } else if (token == JsonToken.START_OBJECT) {
      buildSchemaFromObject(tableDesc, jsonParser);

    } else {
      throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6001E_INVALID_FORMAT_EXCEPTION.name(),
                                           new Object[] {JsonToken.START_ARRAY, jsonParser.getCurrentToken()});
    }
  }

  private void buildSchemaFromObject(TableDescriptorImpl tableDesc, JsonParser jsonParser) throws IOException, JsonResultsReaderException {
    if (jsonParser.getCurrentToken() == JsonToken.START_OBJECT) {
      JsonObjectPrinter jop = new JsonObjectPrinter();
      int columnIndex = 0;
      while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
        String columnName = jsonParser.getCurrentName();
        ColumnDescriptor<?> columnDesc;
        JsonToken node = jsonParser.nextToken();
        jop.addProperty(columnName, jsonParser);
        if (node == JsonToken.VALUE_NUMBER_INT) {
          columnDesc = new ColumnDescriptorImpl<>(columnIndex, columnName, Integer.class);

        } else if (node == JsonToken.VALUE_NUMBER_FLOAT) {
          columnDesc = new ColumnDescriptorImpl<>(columnIndex, columnName, Float.class);

        } else if (node == JsonToken.VALUE_STRING) {
          columnDesc = new ColumnDescriptorImpl<>(columnIndex, columnName, String.class);

        } else if (node == JsonToken.VALUE_TRUE || node == JsonToken.VALUE_FALSE) {
          columnDesc = new ColumnDescriptorImpl<>(columnIndex, columnName, Boolean.class);

        } else {
          throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6002E_UNEXPECTED_TYPE_EXCEPTION.name(),
                                               new Object[] {columnName, tableDesc.getName(), jop});
        }
        columnIndex++;
        tableDesc.addColumnDescriptor(columnDesc);
      }

    } else {
      throw new JsonResultsReaderException(JsonReaderMessageCodes.AKCJC6001E_INVALID_FORMAT_EXCEPTION.name(),
                                           new Object[] {JsonToken.START_OBJECT, jsonParser.getCurrentToken()});
    }
  }
}

/**
 * 
 */
class JsonObjectPrinter {
  private StringWriter sw = new StringWriter();
  private int nbProperties = 0;
  private boolean isObjectEnded = false;

  public JsonObjectPrinter() {
    sw.append('{');
  }

  public void addProperty(String name, JsonParser parser) {
    try {
      if (nbProperties > 0) {
        sw.append(",");
      }
      nbProperties++;
      sw.append("\"" + name + "\":");
      JsonToken token = parser.getCurrentToken();
      if (token == JsonToken.VALUE_STRING) {
        sw.append('\"');
        sw.append(parser.getValueAsString());
        sw.append('\"');

      } else {
        sw.append(parser.getValueAsString());
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  public void objectEnd() {
    sw.append('}');
    isObjectEnded = true;
  }

  @Override
  public String toString() {
    return sw.toString() + (isObjectEnded ? "" : "...");
  }
}