/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import java.util.Date;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Model class representing a job that can be executed on the server.
 */
@ApiModel(value = "Job", description = "A job to be submitted for execution")
public interface Job {
  /**
   * Gets the job revision value.
   * 
   * @return The job revision.
   */
  @ApiModelProperty(value = "The job revision", required = true)
  @JsonProperty("_rev")
  String getRev();

  /**
   * Gets ID of the job.
   * 
   * @return The job ID.
   */
  @ApiModelProperty(value = "The unique ID of the job", required = true)
  @JsonProperty("_id")
  String getId();

  /**
   * Gets the ID of the application associated with the job.
   * 
   * @return The application ID associated with the job.
   */
  @ApiModelProperty(value = "The ID of the application associated with the job")
  String getApplicationId();

  /**
   * Gets the version of the application associated with the submitted job.
   * 
   * @return The version of the application associated with the submitted job.
   */
  @ApiModelProperty(value = "The version of the application associated with the submitted job.")
  String getApplicationVersion();

  /**
   * Gets the version of the application used during job processing.
   * 
   * @return The version of the application used during job processing.
   */
  @ApiModelProperty(value = "The version of the application used during job processing.")
  String getApplicationVersionUsed();

  /**
   * Gets the execution status of the job.
   * 
   * @return The current execution status of the job.
   */
  @ApiModelProperty(value = "The current execution status of the job", required = true)
  JobExecutionStatus getExecutionStatus();

  /**
   * Gets the time when the job was created.
   * 
   * @return The job creation time.
   */
  @ApiModelProperty(value = "The creation time of the job", dataType = "long")
  Date getCreatedAt();

  /**
   * Gets the time when the job was started.
   * 
   * @return The job start time
   */
  @ApiModelProperty(value = "The start time of the job", dataType = "long")
  Date getStartedAt();

  /**
   * Gets the time when the job completed.
   * 
   * @return The job completion time.
   */
  @ApiModelProperty(value = "The completion time of the job", dataType = "long")
  Date getEndedAt();

  /**
   * Gets the time when the job completion status was recorded. There may be a delay between the
   * processing completion by a job processor and the time it is recorded in the job repository.
   * 
   * @see #getEndedAt()
   * 
   * @return The job recorded completion time.
   */
  @ApiModelProperty(value = "The recorded completion time of the job", dataType = "long")
  Date getEndRecordedAt();

  /**
   * Gets the time when the job details were updated.
   * 
   * @return The job completion time.
   */
  @ApiModelProperty(value = "The update time of the job details", dataType = "long")
  public Date getUpdatedAt();

  /**
   * Gets the parameters associated with the job.
   * 
   * @return A map of job parameters.
   */
  @ApiModelProperty(value = "Job parameters")
  JobParameters getParameters();

  /**
   * Gets the processor ID that is currently executing the job. This attribute is empty if the job
   * is not running.
   * 
   * @return The processor ID.
   */
  @ApiModelProperty(value = "The processor ID currently executing the job")
  String getProcessingOwner();

  /**
   * Gets the processor ID that executed the job. This attribute is empty when the job is not
   * completed.
   * 
   * @return The user ID of the previous job process owner.
   */
  @ApiModelProperty(value = "The processor ID that executed the job")
  String getLastProcessingOwner();

  /**
   * Gets the owner of the job. The owner is either the subscriber IBM Id if the job was submitted
   * using the DropSolve, or the API client id if the job was submitted using the API.
   * 
   * @return The job owner.
   */
  @ApiModelProperty(value = "The owner of the job")
  String getOwner();

  /**
   * Gets the attachments to the job.
   * 
   * @return A list of job attachments.
   */
  @ApiModelProperty(value = "Attachments to the job")
  List<JobAttachment> getAttachments();

  /**
   * Gets an attachment to the job.
   * 
   * @param name
   *          Name of attachment.
   * @return The job attachment with the given name.
   */
  JobAttachment getAttachment(String name);

  /**
   * Gets information regarding the failure of the processing of the job.
   * 
   * @return Failure information for the job.
   */
  @ApiModelProperty(value = "Job failure information")
  JobFailureInfo getFailureInfo();

  /**
   * Gets the current language for the job. Reserved for future use.
   * 
   * @return The current language for the job.
   */
  @ApiModelProperty(value = "The job language. This value is reserved for future use")
  String getLanguage();

  /**
   * Gets details about the job. The details are name value pairs that are generated by the
   * processor while the job is running.
   * 
   * @return A map of job detail properties.
   */
  @ApiModelProperty(value = "Detailed job information")
  Map<String, String> getDetails();

  /**
   * Gets information about the current solve status of the job.
   * 
   * @return Current solve status of the job.
   */
  @ApiModelProperty(value = "The current solve status of the job")
  JobSolveStatus getSolveStatus();

  /**
   * Gets information about the current interruption status of the job.
   * 
   * @return Current interruption status of the job.
   */
  @ApiModelProperty(value = "The current interruption status of the job")
  JobInterruptionStatus getInterruptionStatus();

  /**
   * Gets the time when the job was interrupted.
   * 
   * @return The job interruption time.
   */
  @ApiModelProperty(value = "The interruption time of the job", dataType = "long")
  Date getInterruptedAt();

  /**
   * Gets the time when the job was submitted.
   * 
   * @return The job interruption time.
   */
  @ApiModelProperty(value = "The submission time of the job", dataType = "long")
  Date getSubmittedAt();

  /**
   * Gets the number of log items associated with the job, once it is completed.
   * 
   * @return The number of log items.
   */
  @ApiModelProperty(value = "The number of log items associated with the job, once it is completed")
  Long getNbLogItems();

  /**
   * Gets job subscription information.
   * 
   * @return Subscription information for the job.
   */
  @ApiModelProperty(value = "Job subscription information", required = true)
  JobSubscription getSubscription();

  /**
   * Returns the subscriber information. The subscriber information is known when the jobs was
   * submitted interactively using the DropSolve.
   * 
   * @return The subscriber information.
   */
  @ApiModelProperty(value = "Job subscriber information if it was submitted interactively")
  JobSubscriber getSubscriber();

  /**
   * Returns the job API client information. The API client information is available when the job
   * was submitted using the API.
   * 
   * @return The API client information.
   */
  @ApiModelProperty(value = "Job API client information if it was submitted using the API")
  JobApiClient getApiClient();
}
