/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import java.util.List;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.ibm.optim.oaas.client.job.model.impl.JobCreationDataImpl;

/**
 * Job creation data used to initialize a new job associated with an application.
 */
@ApiModel(value = "JobCreationData", description = "Job creation data")
@JsonDeserialize(as = JobCreationDataImpl.class)
public interface JobCreationData {
  /**
   * Gets the ID of the application.
   * 
   * @return The application ID.
   */
  @ApiModelProperty(value = "The ID of the application", allowableValues = "opl, cplex, cpo, docplex")
  public abstract String getApplicationId();

  /**
   * Sets the application ID.
   * 
   * @param applicationId
   *          The application ID.
   */
  public abstract void setApplicationId(String applicationId);

  /**
   * Gets version of the application. Reserved for future use.
   * 
   * @return The version of the application.
   */
  @ApiModelProperty(value = "The version of the application, reserved for future use.")
  public abstract String getApplicationVersion();

  /**
   * Sets the version of the application. Reserved for future use.
   * 
   * @param applicationVersion
   *          Version of the application.
   */
  public abstract void setApplicationVersion(String applicationVersion);

  /**
   * Gets the parameters associated with the job. A list of parameters can be specified, and the
   * system parameters have a prefix of 'oaas', but any custom parameters or attributes can be
   * specified so that the client can store and track custom context of the job.
   * 
   * @return A map of parameters and values.
   */
  @ApiModelProperty(value = "Job parameters")
  public abstract JobParameters getParameters();

  /**
   * Gets the attachments associated with creating the job.
   * 
   * @return A list of job attachments.
   * @see #addJobAttachment(String)
   */
  @ApiModelProperty(value = "Job attachments")
  public abstract List<JobAttachmentCreationData> getAttachments();

  /**
   * Sets the client name used to identify the client submitting this job. The client name may be
   * used to distinguish between different clients using the same subscription.
   * 
   * @param name
   *          The client name.
   */
  public abstract void setClientName(String name);

  /**
   * Returns the client name used to identify the client submitting this job.
   * 
   * @return the client name
   */
  @ApiModelProperty(value = "Client name")
  public abstract String getClientName();

  /**
   * Sets the client email address used for notification.
   * 
   * @param email
   *          The client email.
   */
  public abstract void setClientEmail(String email);

  /**
   * Returns the client email used to for notification.
   * 
   * @return the client email
   */
  @ApiModelProperty(value = "Client email")
  public abstract String getClientEmail();

  /**
   * Adds an input job attachment with a given name.
   * 
   * @param name
   *          The attachment name.
   * @return The created attachment.
   */
  public abstract JobAttachmentCreationData addJobAttachment(String name);

  /**
   * Gets an attachment to the job.
   * 
   * @param name
   *          Name of attachment.
   * @return The job attachment with the given name.
   */
  JobAttachment getAttachment(String name);

}
