/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model;

/**
 * Enumeration of job execution status values.
 */
public enum JobExecutionStatus {
  /**
   * The job has been created.
   */
  CREATED,

  /**
   * The job has been sent to the execution queue and is ready to be picked by
   * a worker.
   */
  NOT_STARTED,

  /**
   * The job has been picked by a worker and is being executed.
   */
  RUNNING,

  /**
   * A job interruption has been requested.
   */
  INTERRUPTING,

  /**
   * The job has been interrupted, with further details about the interruption
   * stored directly in the job. See {@link Job#getInterruptionStatus()}.
   */
  INTERRUPTED,

  /**
   * The job had a system failure, with further details about the failure
   * stored directly in the job. See {@link Job#getFailureInfo()}.
   */
  FAILED,

  /**
   * The job has been processed successfully at the system level.
   */
  PROCESSED;

  /**
   * Check if the job is currently waiting to be processed based on the job status.
   * @param jobStatus The job status.
   * @return Returns {@code true} if the job is waiting to be processed,
   *         {@code false} otherwise.
   */
  public static boolean isWaiting(JobExecutionStatus jobStatus) {
    return NOT_STARTED.equals(jobStatus);
  }

  /**
   * Check if the job execution has completed based on the job status.
   * @param jobStatus The job status.
   * @return Returns {@code true} if the job execution completed
   *         (processed, interrupted or failed), {@code false} otherwise.
   */
  public static boolean isEnded(JobExecutionStatus jobStatus) {
    return PROCESSED.equals(jobStatus) || INTERRUPTED.equals(jobStatus)
        || FAILED.equals(jobStatus);
  }

  /**
   * Check if the job has completed its execution with a failure
   * based on the job status.
   * @param jobStatus The job status.
   * @return Returns {@code true} if the job completed execution with a
   *         failure, {@code false} otherwise.
   */
  public static boolean isFailure(JobExecutionStatus jobStatus) {
    return FAILED.equals(jobStatus);
  }

  /**
   * Check if the job is currently active based on the job status.
   * @param jobStatus The job status.
   * @return Returns {@code true} if the job is currently active,
   *         {@code false} otherwise.
   */
  public static boolean isActive(JobExecutionStatus jobStatus) {
    return RUNNING.equals(jobStatus) || INTERRUPTING.equals(jobStatus);
  }
}