/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.ibm.optim.oaas.client.job.JobClient;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

/**
 * Class representing an item in the log for a job.
 */
@ApiModel(value = "JobLogItem", description = "Job log item")
public interface JobLogItem {
  /**
   * Gets the ID of the associated job.
   * 
   * @return The job ID.
   */
  @ApiModelProperty(value = "ID of the associated job", required = true)
  @JsonProperty(value="jobid")
  public abstract String getJobId();

  /**
   * Sequence ID of the item in the job log. Each log item is given a sequence
   * number starting at 0 for each job.
   * 
   * @return The log item sequence ID.
   */
  @ApiModelProperty(value = "Log item sequence ID", required = true)
  public abstract long getSeqid();

  /**
   * Indicates if the log item is missing from the sequence when accessing the sequence in
   * continuous mode. 
   * A log item is missing if a later log item is available but this one was not stored.  
   * A log item can be missing when there was an unexpected delay in storing the log item or 
   * if there was an issue in the infrastructure.
   * @see JobClient#getJobLogItems(String, long, boolean)
   * @return the missing flag
   */
  @ApiModelProperty(value = "The log item was not stored because of delay or infrastructure issues")
  public abstract boolean missing();

  /**
   * Indicates that the log item signals the end of the log item sequence when accessing the sequence in
   * continuous mode. 
   * Getting the next log items should not be attempted.
   * If the flag is {@code true}, it also guarantee that the job has ended.
   * 
   * @see JobClient#getJobLogItems(String, long, boolean)
   * @return the stop flag
   */
  @ApiModelProperty(value = "Signals the end of the log item list")
  public abstract boolean stop();

  /**
   * Gets the optimization engine log records associated with the job.
   * 
   * @return A list of log records from the optimization engine execution.
   */
  @ApiModelProperty(value = "Log records from the optimization engine")
  @JsonProperty(value="records")
  public abstract List<JobLogRecord> getEngineLogRecords();
}