/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model;

import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;

import io.swagger.annotations.ApiModelProperty;

/**
 * Stores various job parameters. System parameters start with a prefix of <code>oaas</code>, and
 * additional custom properties can be added to keep track of job context.
 * 
 */
public interface JobParameters extends Map<String, Object> {
  /**
   * Time limits in milliseconds. This can be used to setup a time limit that is smaller than the
   * limit defined by the subscription.
   */
  public static final String TIME_LIMIT = "oaas.timeLimit";

  /**
   * Log size limit in characters. This can be used to setup a time limit that is smaller than the
   * limit defined by the subscription.
   */
  public static final String LOG_LIMIT = "oaas.logLimit";

  /**
   * Specifies the name of the OPL run configuration to be executed by the OPL Worker.
   */
  public static final String OPL_RUN_CONFIG = "oaas.oplRunConfig";

  /**
   * Specifies the format for returned results. JSON is the default format. Other formats may or may
   * not be supported by each application type.
   */
  public static final String RESULTS_FORMAT = "oaas.resultsFormat";

  /**
   * String value for {@link #RESULTS_FORMAT} parameter to specify TEXT as results format. This
   * format may or may not be supported by a given application.
   */
  public static final String RESULTS_FORMAT_TEXT = "TEXT";

  /**
   * String value for {@link #RESULTS_FORMAT} parameter to specify XML as results format. This format
   * may or may not be supported by a given application.
   */
  public static final String RESULTS_FORMAT_XML = "XML";

  /**
   * String value for {@link #RESULTS_FORMAT} parameter to specify JSON as results format. This is
   * the default format, supported by all applications.
   */
  public static final String RESULTS_FORMAT_JSON = "JSON";

  /**
   * String value for {@link #RESULTS_FORMAT} parameter to specify XLSX as results format. This
   * format may or may not be supported by a given application.
   */
  public static final String RESULTS_FORMAT_XLSX = "XLSX";

  /**
   * Specifies the webhook to be called on job completion. A webhook is an HTTP URL that will be
   * accessed by the DOcloud service upon job completion to notify the client. This is useful when
   * the job is submitted by a web application and when polling the jobs status is not desirable.
   * The DOcloud service will issue a POST on the passed URL with the <code>job</code> query
   * parameter to indicate the job id that completed. The client can then access the API to retrieve
   * the job status and results.
   */
  public static final String WEBHOOK = "oaas.webhook";

  /**
   * Indicates if the job was submitted interactively using the DropSolve. This parameter is used
   * internally, but may be accessed to distinguish between jobs submitted by API or interactively
   * using the DropSolve.
   */
  public static final String DROPSOLVE = "oaas.dropsolve";

  /**
   * Indicates that an email notification will be sent upon completion of the job.
   */
  public static final String NOTIFY_COMPLETION = "oaas.notifyCompletion";

  /**
   * Indicates that an email alert will be sent when a the jobs reaches the specified alert time
   * limit.
   */
  public static final String ALERT_TIME = "oaas.alertTime";

  /**
   * Only used in Reserved Bare Metal subscription, this parameter specifies cores used to process job. 
   * Value can vary from 1 to max number of cores of processing instance (currently 14 cores). 
   * If not specified, job uses all instance's cores
   */
  public static final String JOB_CORES = "oaas.jobCores";

  /**
   * Only used in Reserved Bare Metal subscription, this parameter specifies maximum memory megabytes 
   * used to process job. Value can vary from 512MB to max memory of processing instance (currently 252GB).
   * If not specified, job uses all instance's memory. 
   */
  public static final String JOB_MEMORY_MBYTES = "oaas.jobMemoryMbytes";

  /**
   * Indicates which Bluemix region was used if the job was submitted via Bluemix and if, and only if,
   * the Bluemix region has been detected. This parameter is used
   * internally, but may be accessed to distinguish from which Bluemix region the
   * job was submitted if known.
   */
  public static final String BLUEMIX_REGION = "oaas.bluemix.region";

  /**
   * Returns the time limit.
   * 
   * @see #TIME_LIMIT
   * @return The time limit.
   */
  @ApiModelProperty(value = "Time limits in milliseconds")
  @JsonProperty(value = TIME_LIMIT)
  public Long getTimeLimit();

  /**
   * Sets the time limit.
   * 
   * @param value
   *          The time limit.
   * @see #TIME_LIMIT
   */
  public void setTimeLimit(Long value);

  /**
   * Returns the log limit.
   * 
   * @see #LOG_LIMIT
   * @return The log limit.
   */
  @ApiModelProperty(value = "Log size limit in characters")
  @JsonProperty(value = LOG_LIMIT)
  public Long getLogLimit();

  /**
   * Sets the log limit.
   * 
   * @param value
   *          The log limit.
   * @see #LOG_LIMIT
   */
  public void setLogLimit(Long value);

  /**
   * Returns the OPL run configuration.
   * 
   * @see #OPL_RUN_CONFIG
   * @return The OPL run configuration.
   */
  @ApiModelProperty(value = "Specify the name of the OPL run configuration to be executed")
  @JsonProperty(value = OPL_RUN_CONFIG)
  public String getOplRunConfig();

  /**
   * Sets the OPL run configuration.
   * 
   * @param value
   *          The OPL run configuration.
   * @see #OPL_RUN_CONFIG
   */
  public void setOplRunConfig(String value);

  /**
   * Returns results format.
   * 
   * @see #RESULTS_FORMAT
   * @return The results format.
   */
  @ApiModelProperty(value = "Specify the format for returned results, default is JSON", allowableValues = RESULTS_FORMAT_JSON
      + "," + RESULTS_FORMAT_XML + "," + RESULTS_FORMAT_TEXT)
  @JsonProperty(value = RESULTS_FORMAT)
  public String getResultsFormat();

  /**
   * Sets the results format.
   * 
   * @param value
   *          The results format.
   * @see #RESULTS_FORMAT
   */
  public void setResultsFormat(String value);

  /**
   * Returns the webhook.
   * 
   * @see #WEBHOOK
   * @return The web hooks.
   */
  @ApiModelProperty(value = "Specifies the webhook to be called on job completion")
  @JsonProperty(value = WEBHOOK)
  public String getWebhook();

  /**
   * Sets the webhook.
   * 
   * @param value
   *          The the webhook.
   * @see #WEBHOOK
   */
  public void setWebhook(String value);

  /**
   * The DropSolve flag.
   * 
   * @see #DROPSOLVE
   * @return The time limit.
   */
  @ApiModelProperty(value = "Internal use, indicates if job was submitted by the DropSolve")
  @JsonProperty(value = DROPSOLVE)
  public Boolean getDropSolve();

  /**
   * Returns the notify completion flag.
   * 
   * @see #NOTIFY_COMPLETION
   * @return <code>true</code> if notification is requested, <code>false</code> otherwise
   */
  @ApiModelProperty(value = "Indicates that an email notification will be sent upon completion of the job")
  @JsonProperty(value = NOTIFY_COMPLETION)
  public Boolean getNotifyCompletion();

  /**
   * Sets the notify completion flag.
   * 
   * @see #NOTIFY_COMPLETION
   * @param notify
   *          <code>true</code> if notification is requested, <code>false</code> otherwise
   */
  public void setNotifyCompletion(Boolean notify);

  /**
   * Returns the alert limit.
   * 
   * @see #ALERT_TIME
   * @return The alert limit or <code>null</code> if undefined.
   */
  @ApiModelProperty(value = "Alert limit in milliseconds")
  @JsonProperty(value = ALERT_TIME)
  public Long getAlertTime();

  /**
   * Returns cores used to process job.
   * 
   * @see #JOB_CORES
   * @return Cores used to process job.
   */
  @ApiModelProperty(value = "Cores used to process job")
  @JsonProperty(value = JOB_CORES)
  public Integer getJobCores();

  /**
   * Sets the number of cores used to process the job.
   * 
   * @param cores The number of cores used to process the job.
   * @see #JOB_CORES
   */
  public void setJobCores(Integer cores);

  /**
   * Returns maximum memory megabytes used to process the job.
   * 
   * @see #JOB_MEMORY_MBYTES
   * @return Maximum memory megabytes used to process the job.
   */
  @ApiModelProperty(value = "Maximum memory megabytes used to process job")
  @JsonProperty(value = JOB_MEMORY_MBYTES)
  public Integer getJobMemoryMbytes();

  /**
   * Sets maximum memory megabytes used to process the job.
   * 
   * @param mbytes The maximum memory megabytes used to process the job.
   * @see #JOB_MEMORY_MBYTES
   */
  public void setJobMemoryMbytes(Integer mbytes);

  /**
   * Returns the Bluemix region that submitted this job (if known).
   * 
   * @see #BLUEMIX_REGION
   * @return The Bluemix region.
   */
  @ApiModelProperty(value = "Bluemix region that submitted this job")
  @JsonProperty(value = BLUEMIX_REGION)
  public String getBluemixRegion();

  /**
   * Sets the Bluemix region that created this job (if known).
   * 
   * @param bluemixRegion The Bluemix region.
   * @see #BLUEMIX_REGION
   */
  public void setBluemixRegion(String bluemixRegion);

  /**
   * Sets the alert limit.
   * 
   * @param value
   *          The alert limit.
   * @see #ALERT_TIME
   */
  public void setAlertTime(Long value);

  /**
   * Returns the parameter value as a Boolean.
   * 
   * @param name
   *          The parameter name.
   * @return the parameter value.
   */
  public Boolean getAsBoolean(String name);

  /**
   * Returns the parameter value as a String.
   * 
   * @param name
   *          The parameter name.
   * @return the parameter value.
   */
  public String getAsString(String name);

  /**
   * Returns the parameter value as a Double.
   * 
   * @param name
   *          The parameter name.
   * @return the parameter value.
   */
  public Double getAsDouble(String name);

  /**
   * Returns the parameter value as a Double.
   * 
   * @param name
   *          The parameter name.
   * @return the parameter value.
   */
  public Long getAsLong(String name);

  /**
   * Returns the parameter value as a Integer.
   * 
   * @param name
   *          The parameter name.
   * @return the parameter value.
   */
  public Integer getAsInteger(String name);

  /**
   * Returns the parameter value as a Boolean with a default value.
   * 
   * @param name
   *          The parameter name.
   * @param defaultValue
   *          default value
   * @return the parameter value.
   */
  public boolean getAsBoolean(String name, boolean defaultValue);

  /**
   * Returns the parameter value as a String with a default value.
   * 
   * @param name
   *          The parameter name.
   * @param defaultValue
   *          default value
   * @return the parameter value.
   */
  public String getAsString(String name, String defaultValue);

  /**
   * Returns the parameter value as a Double with a default value.
   * 
   * @param name
   *          The parameter name.
   * @param defaultValue
   *          default value
   * @return the parameter value.
   */
  public double getAsDouble(String name, double defaultValue);

  /**
   * Returns the parameter value as a Double with a default value.
   * 
   * @param name
   *          The parameter name.
   * @param defaultValue
   *          default value
   * @return the parameter value.
   */
  public long getAsLong(String name, long defaultValue);

  /**
   * Returns the parameter value as a Integer with a default value.
   * 
   * @param name
   *          The parameter name.
   * @param defaultValue
   *          default value
   * @return the parameter value.
   */
  public int getAsInteger(String name, int defaultValue);
}
