/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ibm.optim.oaas.client.job.model.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonView;
import com.ibm.optim.oaas.client.job.model.Job;
import com.ibm.optim.oaas.client.job.model.JobAttachment;
import com.ibm.optim.oaas.client.job.model.JobExecutionStatus;
import com.ibm.optim.oaas.client.job.model.JobInterruptionStatus;
import com.ibm.optim.oaas.client.job.model.JobParameters;
import com.ibm.optim.oaas.client.job.model.JobSolveStatus;

@JsonAutoDetect(fieldVisibility = Visibility.ANY, getterVisibility = Visibility.NONE, setterVisibility = Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class JobImpl extends DocumentImpl implements Job {
  public final static String TYPE = "job";

  @JsonView(StorageView.class)
  @JsonProperty("type")
  private final String type = TYPE;
  private String applicationId;
  private String applicationVersion;
  private String applicationVersionUsed;
  private String language = "en-US";
  private JobExecutionStatus executionStatus;
  private JobSolveStatus solveStatus;
  private JobInterruptionStatus interruptionStatus;
  private String owner;
  private String processingOwner;
  private String lastProcessingOwner;
  private Date createdAt;
  private Date submittedAt;
  private Date submittedToProcessorAt;
  private Date startedAt;
  private Date interruptedAt;
  private Date endedAt;
  private Date endRecordedAt;
  private Date updatedAt;

  private final JobParametersImpl parameters = new JobParametersImpl();

  private Map<String, String> details;

  private Map<String, String> systemDetails;

  private List<JobAttachmentImpl> attachments = new ArrayList<JobAttachmentImpl>();

  private JobFailureInfoImpl failure;

  private Long nbLogItems;

  private JobSubscriptionImpl subscription = new JobSubscriptionImpl();

  private Integer initialQueueIndex;

  private Integer currentQueueIndex;

  @JsonView(StorageView.class)
  private List<String> merged;

  @JsonView(StorageView.class)
  private String token;

  private JobApiClientImpl apiClient;

  private JobSubscriberImpl subscriber;

  @JsonIgnore
  private JobDetailsImpl lastDetails;

  @JsonView(StorageView.class)
  private int storageVersion;

  @JsonIgnore
  private boolean ignoreJobCount;

  public JobImpl() {

  }

  public Integer getInitialQueueIndex() {
    return initialQueueIndex;
  }

  public void setInitialQueueIndex(Integer initialQueueIndex) {
    this.initialQueueIndex = initialQueueIndex;
  }

  public Integer getCurrentQueueIndex() {
    return currentQueueIndex;
  }

  public void setCurrentQueueIndex(Integer currentQueueIndex) {
    this.currentQueueIndex = currentQueueIndex;
  }

  public List<String> getMerged() {
    return merged;
  }

  public void addToMerged(List<String> tomerge) {
    if (merged == null) {
      merged = new ArrayList<String>();
      merged.addAll(tomerge);
    } else {
      for (String rev : tomerge) {
        if (!merged.contains(rev)) {
          merged.add(rev);
        }
      }
    }
  }

  public List<String> getUnmergedConflicts() {

    if (getConflicts() == null) {
      return Collections.emptyList();
    }

    if (merged == null) {
      return getConflicts();
    } else {
      List<String> list = new ArrayList<String>();
      for (String rev : getConflicts()) {
        if (!merged.contains(rev)) {
          list.add(rev);
        }
      }
      return list;
    }

  }

  public boolean needMerge() {
    return getConflicts() != null && (merged == null || !merged.containsAll(getConflicts()));
  }

  @Override
  public String getApplicationId() {
    return applicationId;
  }

  public void setApplicationId(String applicationId) {
    this.applicationId = applicationId;
  }

  @Override
  public String getApplicationVersion() {
    return applicationVersion;
  }

  public void setApplicationVersion(String applicationVersion) {
    this.applicationVersion = applicationVersion;
  }

  @Override
  public String getApplicationVersionUsed() {
    return applicationVersionUsed;
  }

  public void setApplicationVersionUsed(String applicationVersionUsed) {
    this.applicationVersionUsed = applicationVersionUsed;
  }

  @Override
  public JobExecutionStatus getExecutionStatus() {
    return this.executionStatus;
  }

  public void setExecutionStatus(JobExecutionStatus status) {
    this.executionStatus = status;
  }

  @Override
  public Date getCreatedAt() {
    return this.createdAt;
  }

  public void setCreatedAt(Date createdAt) {
    this.createdAt = createdAt;
  }

  @Override
  public Date getStartedAt() {
    return startedAt;
  }

  public void setStartedAt(Date startedAt) {
    this.startedAt = startedAt;
  }

  @Override
  public Date getEndedAt() {
    return this.endedAt;
  }

  public void setEndedAt(Date endedAt) {
    this.endedAt = endedAt;
  }

  @Override
  public Date getEndRecordedAt() {
    return endRecordedAt;
  }

  public void setEndRecordedAt(Date endRecordedAt) {
    this.endRecordedAt = endRecordedAt;
  }

  @Override
  public Date getUpdatedAt() {
    return updatedAt;
  }

  public void setUpdatedAt(Date updatedAt) {
    this.updatedAt = updatedAt;
  }

  @Override
  public JobParameters getParameters() {
    return parameters;
  }

  public void setParameter(String name, String value) {
    parameters.put(name, value);
  }

  public void setParameters(Map<String, String> parameters) {
    this.parameters.clear();
    this.parameters.putAll(parameters);
  }

  public void setParameters(JobParameters parameters) {
    this.parameters.clear();
    this.parameters.putAll(parameters);
  }

  @Override
  public String getProcessingOwner() {
    return processingOwner;
  }

  public void setProcessingOwner(String owner) {
    lastProcessingOwner = processingOwner;
    this.processingOwner = owner;
  }

  @Override
  public String getLastProcessingOwner() {
    return lastProcessingOwner;
  }

  @Override
  public String getOwner() {
    return owner;
  }

  public void setOwner(String owner) {
    this.owner = owner;
  }

  @Override
  public String toString() {
    return "[id=" + getId() + ",status=" + getExecutionStatus() + ",owner=" + getOwner()
    + ",processing=" + getProcessingOwner() + "]";
  }

  /**
   * Swagger does not generate the doc of "? extends" so doing a trick.
   */
  @SuppressWarnings("unchecked")
  @Override
  public List<JobAttachment> getAttachments() {
    List<?> list = attachments;
    return (List<JobAttachment>) list;
  }

  public List<JobAttachmentImpl> getImplAttachments() {
    return attachments;
  }

  public void setAttachments(List<JobAttachmentImpl> attachments) {
    this.attachments = attachments;
  }

  @Override
  public JobAttachmentImpl getAttachment(String name) {
    for (JobAttachmentImpl att : attachments) {
      if (att.getName().equals(name)) {
        return att;
      }
    }
    return null;
  }

  @Override
  public JobFailureInfoImpl getFailureInfo() {
    return failure;
  }

  public void setFailureInfo(JobFailureInfoImpl failure) {
    this.failure = failure;
  }

  @Override
  public String getLanguage() {
    return language;
  }

  public void setLanguage(String language) {
    this.language = language;
  }

  @Override
  public Map<String, String> getDetails() {
    return details;
  }

  public void setSystemDetails(Map<String, String> details) {
    this.systemDetails = details;
  }

  public Map<String, String> getSystemDetails() {
    return systemDetails;
  }

  public void setDetails(Map<String, String> details) {
    this.details = details;
  }

  @Override
  public JobSolveStatus getSolveStatus() {
    return solveStatus;
  }

  public void setSolveStatus(JobSolveStatus solveStatus) {
    this.solveStatus = solveStatus;
  }

  @Override
  public JobInterruptionStatus getInterruptionStatus() {
    return interruptionStatus;
  }

  public void setInterruptionStatus(JobInterruptionStatus interruptionStatus) {
    this.interruptionStatus = interruptionStatus;
  }

  @Override
  public Date getInterruptedAt() {
    return interruptedAt;
  }

  public void setInterruptedAt(Date interruptedAt) {
    this.interruptedAt = interruptedAt;
  }

  @Override
  public Date getSubmittedAt() {
    return submittedAt;
  }

  public void setSubmittedAt(Date submittedAt) {
    this.submittedAt = submittedAt;
  }

  public Date getSubmittedToProcessorAt() {
    return submittedToProcessorAt;
  }

  public void setSubmittedToProcessorAt(Date submittedToProcessorAt) {
    this.submittedToProcessorAt = submittedToProcessorAt;
  }

  @Override
  public Long getNbLogItems() {
    return nbLogItems;
  }

  public void setNbLogItems(Long nbLogItems) {
    this.nbLogItems = nbLogItems;
  }

  @Override
  public JobSubscriptionImpl getSubscription() {
    return subscription;
  }

  public void setSubscription(JobSubscriptionImpl subscription) {
    this.subscription = subscription;
  }

  public String getToken() {
    return token;
  }

  public void setToken(String token) {
    this.token = token;
  }

  public String getSubscriptionId() {
    return subscription != null ? subscription.getId() : null;
  }

  @Override
  public JobApiClientImpl getApiClient() {
    return apiClient;
  }

  public void setApiClient(JobApiClientImpl apiClient) {
    this.apiClient = apiClient;
  }

  @Override
  public JobSubscriberImpl getSubscriber() {
    return subscriber;
  }

  public void setSubscriber(JobSubscriberImpl subscriber) {
    this.subscriber = subscriber;
  }

  public int getStorageVersion() {
    return storageVersion;
  }

  public void setStorageVersion(int storageVersion) {
    this.storageVersion = storageVersion;
  }

  public JobDetailsImpl getLastDetails() {
    return lastDetails;
  }

  public void setLastDetails(JobDetailsImpl lastDetails) {
    this.lastDetails = lastDetails;
  }

  public String getETag() {
    if (lastDetails != null) {
      return getRev() + lastDetails.getRev();
    } else {
      return getRev();
    }
  }

  public boolean isIgnoreJobCount() {
    return ignoreJobCount;
  }

  public void setIgnoreJobCount(boolean ignoreJobCount) {
    this.ignoreJobCount = ignoreJobCount;
  }

  public String getEmailForNotification() {
    if (getApiClient() != null) {
      return getApiClient().getEmail();
    } else if (getSubscriber() != null) {
      return getSubscriber().getEmail();
    }
    return null;
  }
}
